package fr.emac.gind.workflow.generator;

/*
 * #%L
 * abstract-workflow-deduction
 * %%
 * Copyright (C) 2014 - 2016 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import java.util.Optional;
import java.util.TreeMap;
import java.util.UUID;

import javax.xml.namespace.QName;

import org.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.emac.gind.commons.utils.color.ColorHelper;
import fr.emac.gind.commons.utils.jaxb.JSONJAXBContext;
import fr.emac.gind.commons.utils.jaxb.SOAException;
import fr.emac.gind.commons.utils.list.ListUtil;
import fr.emac.gind.gov.core.client.CoreGovClient;
import fr.emac.gind.gov.core_gov.GJaxbQuery;
import fr.emac.gind.gov.core_gov.GJaxbQueryResponse;
import fr.emac.gind.gov.core_gov.GJaxbSelectedKnowledgeSpace;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveMetaModel;
import fr.emac.gind.models.core.service.ModelsGovImpl;
import fr.emac.gind.models.generic.modeler.GenericModelHelper;
import fr.emac.gind.models.generic.modeler.GenericModelManager;
import fr.emac.gind.workflow.report.GJaxbReport;
import fr.emac.gind.workflow.report.GJaxbStatusType;
import fr.gind.emac.gov.core_gov.CoreGov;
import fr.gind.emac.gov.core_gov.QueryFault;

/**
 *
 *
 * @author: Nicolas Salatge (nicolas.salatge@mines-albi.fr)
 */
public class ProcessDeductionHelper {

  private static Logger LOG = LoggerFactory.getLogger(ProcessDeductionHelper.class.getName());

  private static  Map<GJaxbNode, String> organizationColors = new HashMap<GJaxbNode, String>();

  public static TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>> createMapObjectivesByEffectOrRiskByPriority(
      String currentCollaborationName, String currentKnowledgeSpaceName,
      GJaxbReport report, JSONArray ordering, CoreGovClient coreClient) throws SOAException, QueryFault {
    TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>> objectivesByEffectOrRiskByPiority = new TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>>();
    for(int i = 0; i < ordering.length(); i++) {
      GJaxbNode effectOrRisk = JSONJAXBContext.getInstance().unmarshall("{ \"node\" : "  + ordering.getJSONObject(i).getJSONObject("object").toString() + " }", GJaxbNode.class);

      // find associated objectives (treated or prevent)
      Map<GJaxbNode, List<GJaxbNode>> objectivesByEffectOrRisk = new HashMap<GJaxbNode, List<GJaxbNode>>();
      GJaxbGenericModel assoObj = coreClient.singleQuery(new String("match (n1)-[:`{http://fr.emac.gind/core-model}Concerns`]->(n2 { modelNodeId: '" + effectOrRisk.getId() + "_c__${collaboration}_k__${knowledgeSpace}' }) return n1").replace("${collaboration}", currentCollaborationName).replace("${knowledgeSpace}", currentKnowledgeSpaceName), currentCollaborationName, currentKnowledgeSpaceName);
      if(assoObj == null || assoObj.getNode() == null || assoObj.getNode().size() == 0) {
        report.getResult().getAnalyticReport().setStatus(GJaxbStatusType.INCOMPLETE);
        report.getResult().getAnalyticReport().setReportMessage(report.getResult().getAnalyticReport().getReportMessage() + "Error: Impossible to find at least one objective corresponding to this effect or risk: " + GenericModelHelper.getName(effectOrRisk) + "\n");
      }
      if(assoObj != null && assoObj.getNode() != null) {
        assoObj.getNode().forEach(obj -> { obj.setUserData("associatedEffectOrRisk", effectOrRisk); });
        objectivesByEffectOrRisk.put(effectOrRisk, assoObj.getNode());
      }
      objectivesByEffectOrRiskByPiority.put(new Integer(i), objectivesByEffectOrRisk);
    }
    return objectivesByEffectOrRiskByPiority;
  }

  public static Map<GJaxbNode, GJaxbGenericModel> createMapPathByObjective(String currentCollaborationName,
      String currentKnowledgeSpaceName, GJaxbReport report,
      TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>> objectivesByEffectOrRiskByPiority, CoreGovClient coreClient)
          throws QueryFault {
    Map<GJaxbNode, GJaxbGenericModel> pathByObj = new HashMap<GJaxbNode, GJaxbGenericModel>(); 
    for(Map<GJaxbNode, List<GJaxbNode>> effectOrRiskByObjectives: objectivesByEffectOrRiskByPiority.values()) {
      for(List<GJaxbNode> objectives: effectOrRiskByObjectives.values()) {
        for(GJaxbNode obj: objectives) {

          List<GJaxbGenericModel> pathsModel = coreClient.multipleQuery(new String("match p = allShortestPaths((a:organization)-[:`{http://fr.emac.gind/core-model}Provides`|:`{http://fr.emac.gind/core-model}Satisfies`|:`{http://fr.emac.gind/core-model}Concerns`|:near*]-(o:objective { modelNodeId : '" + obj.getId() + "_c__${collaboration}_k__${knowledgeSpace}'} )) " + 
              " where filter(x in nodes(p) where x:`${collaboration}`:`${knowledgeSpace}`) and any(c in nodes(p) where c:function and c.node_status = 'ACTIF') and any(o in nodes(p) where o:objective and o.node_status = 'ACTIF' and o.modelNodeId = '" + obj.getId() + "_c__${collaboration}_k__${knowledgeSpace}') and a.node_status = 'ACTIF' and o.node_status = 'ACTIF' return p  ").replace("${collaboration}", currentCollaborationName).replace("${knowledgeSpace}", currentKnowledgeSpaceName), currentCollaborationName, currentKnowledgeSpaceName);

          if(pathsModel != null && !pathsModel.isEmpty()) {
            Entry<Double, GJaxbGenericModel> bestPath = findBestPath(pathsModel);


            pathByObj.put(obj, bestPath.getValue());
          } else {
            pathByObj.put(obj, null);
          }
        }
      }
    }
    return pathByObj;
  }



  public static Entry<Double, GJaxbGenericModel> findBestPath(
      List<GJaxbGenericModel> pathsModel) {

    TreeMap<Double, List<GJaxbGenericModel>> orderingPathsByPertinenceIndice = new TreeMap<Double, List<GJaxbGenericModel>>();
    double pertinenceIndiceMax = 0;
    GJaxbGenericModel bestPath = null;
    for(GJaxbGenericModel path: pathsModel) {
      double pertinenceIndice = calculatePertinenceIndice(path);


      if(orderingPathsByPertinenceIndice.get(pertinenceIndice) == null) {
        orderingPathsByPertinenceIndice.put(pertinenceIndice, new ArrayList<GJaxbGenericModel>());
      }
      orderingPathsByPertinenceIndice.get(pertinenceIndice).add(path);
    }

    pertinenceIndiceMax = orderingPathsByPertinenceIndice.lastKey();
    List<GJaxbGenericModel> bestPaths = orderingPathsByPertinenceIndice.get(pertinenceIndiceMax);


    int numberOfNodesMin = bestPaths.get(0).getNode().size();
    for(GJaxbGenericModel path: bestPaths) {
      if(path.getNode().size() <= numberOfNodesMin) {
        numberOfNodesMin = path.getNode().size();
        bestPath = path;
      }
    }


    Entry<Double, GJaxbGenericModel> entry = new AbstractMap.SimpleEntry<Double, GJaxbGenericModel>(pertinenceIndiceMax, bestPath);
    return entry;
  }

  public static double calculatePertinenceIndice(GJaxbGenericModel path) {
    double res = 1;
    for(GJaxbEdge edge: path.getEdge()) {
      GJaxbProperty satisfiesProp = GenericModelHelper.findProperty("coverage", edge.getProperty());
      if(satisfiesProp != null && satisfiesProp.getValue() != null) {
        double coverage = Double.parseDouble(satisfiesProp.getValue())/100;
        res = res * coverage;
      } else {
        GJaxbProperty nearProp = GenericModelHelper.findProperty("near at", edge.getProperty());
        if(nearProp != null && nearProp.getValue() != null) {
          double near = Double.parseDouble(nearProp.getValue())/100;
          res = res * near;
        } 
      }
    }
    return res*100;
  }


  public static String printPath(final GJaxbGenericModel path) {
    String pathS = "";
    List<GJaxbNode> startingNodes = findStartingNodes(path);
    assert startingNodes != null && !startingNodes.isEmpty();
    GJaxbNode currentNode = startingNodes.get(0);
    boolean reverse = false;
    List<GJaxbEdge> edges = new ArrayList<GJaxbEdge>(path.getEdge());
    while(currentNode != null) {
      GJaxbEdge currentEdge = findEdgeWhereSourceIs(edges, currentNode);
      if(currentEdge != null && reverse == false) {
        pathS = pathS + "(" + GenericModelHelper.getName(currentNode) + ")" + "->";
        currentNode = currentEdge.getTarget();
        edges.remove(currentEdge);
      } else {
        currentEdge = findEdgeWhereTargetIs(edges, currentNode);
        if(currentEdge != null) {
          reverse = true;
          pathS = pathS + "(" + GenericModelHelper.getName(currentNode) + ")" + "<-";
          currentNode = currentEdge.getSource();
          edges.remove(currentEdge);
        } else {
          pathS = pathS + "(" + GenericModelHelper.getName(currentNode) + ")";
          currentNode = null;
        }
      } 
    }
    //
    return pathS;
  }



  private static List<GJaxbNode> findStartingNodes(final GJaxbGenericModel path) {
    List<GJaxbNode> nonStartingNodes = path.getEdge().parallelStream().map(e -> {
      return e.getSource();
    }).collect(Collectors.toList());
    nonStartingNodes = ListUtil.deleteDuplicate(nonStartingNodes);
    
    List<GJaxbNode> res = new ArrayList<GJaxbNode>();
    for(GJaxbNode node: path.getNode()) {
      res.add(GenericModelHelper.cloneNode(node));
    }
    res.removeAll(nonStartingNodes);
    return res;
    }

  private static GJaxbEdge findEdgeWhereSourceIs(List<GJaxbEdge> edges,
      GJaxbNode currentNode) {
    for(GJaxbEdge edge: edges) {
      if(edge.getSource().getId().equals(currentNode.getId())) {
        return edge;
      }
    }
    return null;
  }

  private static GJaxbEdge findEdgeWhereTargetIs(List<GJaxbEdge> edges,
      GJaxbNode currentNode) {
    for(GJaxbEdge edge: edges) {
      if(edge.getTarget().getId().equals(currentNode.getId())) {
        return edge;
      }
    }
    return null;
  }

  public static GJaxbNode findFirstStartEvent(List<GJaxbNode> nodes) {
    Optional<GJaxbNode> optNode = nodes.parallelStream().filter(n ->{
      return n.getRole().contains("startEvent");
    }).findAny();
    if(optNode.isPresent()) return optNode.get();
    else throw new RuntimeException("no start event in process!");
  }

  public static GJaxbNode findFirstInPath(List<GJaxbNode> nodes) {
    for(GJaxbNode node: nodes) {
      GJaxbProperty first = GenericModelHelper.findProperty("firstInPath", node.getProperty());
      if(first != null && first.getValue() != null && Boolean.valueOf(first.getValue()) == true) {
        return node;
      }
    }
    return null;
  }


  public static void showProcess(GJaxbNode startEvent,
      TreeMap<Integer, Entry<GJaxbNode, List<GJaxbNode>>> orderingGateways, GJaxbNode endEvent) {
    List<Object> process = new ArrayList<Object>();
    process.add(startEvent);
    for(Entry<GJaxbNode, List<GJaxbNode>> gateway: orderingGateways.values()) {
      if(gateway.getValue().size() > 1) {
        process.add(gateway.getValue());
      } else {
        process.add(gateway.getValue().get(0));
      }
    }
    process.add(endEvent);


    int i = 1;
    for(Object step: process) {
      if(step instanceof GJaxbNode) {

      } else {
        System.out.print("step " + i + ": ");
        if(step != null) {
          for(GJaxbNode node: (List<GJaxbNode>)step) {
            System.out.print(GenericModelHelper.getName(node) + ", ");
          }
        }

      }
      i = i + 1;
    }


  }

  public static GJaxbNode findOrganization(GenericModelManager manager) {
    List<GJaxbNode> organizations = manager.getNodesByType(new QName("http://fr.emac.gind/crisis_functions", "Actor")); 
    for(GJaxbNode organization: organizations) {
      if(manager.findInputEdgesOfNode(organization).isEmpty()) {
        return organization;
      }
    }
    return null;
  }








  public static GJaxbNode findFirstService(GenericModelManager manager, GJaxbNode organization) {
    List<GJaxbNode> services = manager.getNodesByRoles("function"); 
    for(GJaxbNode service: services) {
      List<GJaxbEdge> edges = manager.findInputEdgesOfNode(service);
      for(GJaxbEdge edge: edges) {
        if(edge.getSource() == organization && edge.getType().equals(new QName("http://fr.emac.gind/core-model", "Provides"))) {
          return service;
        }
      }
    }
    return null;
  }


  public static GJaxbNode createTask(GJaxbNode function, GJaxbNode objective, GJaxbNode organization)
      throws SOAException {
    GJaxbNode task = new GJaxbNode();
    task.setId("task_" + UUID.randomUUID().toString());
    if(function.getRole().contains("human_function")) {
      task.setType(new QName("http://fr.emac.gind/core-model", "HumanTask"));
      if(GenericModelHelper.findProperty("Attributed to", function.getProperty()) != null) {
        task.getProperty().add(GenericModelHelper.findProperty("Attributed to", function.getProperty()));
      }
      if(GenericModelHelper.findProperty("Priority", function.getProperty()) != null) {
        task.getProperty().add(GenericModelHelper.findProperty("Priority", function.getProperty()));
      }
      task.getRole().add("human_task");
    } else if(function.getRole().contains("computer_function")) {
      task.setType(new QName("http://fr.emac.gind/core-model", "ComputerTask"));
      task.getRole().add("computer_task");
    } else if(function.getRole().contains("mediation_function")) {
      task.setType(new QName("http://fr.emac.gind/core-model", "MediationTask"));
      if(GenericModelHelper.findProperty("type", function.getProperty()) != null) {
        task.getProperty().add( GenericModelHelper.findProperty("type", function.getProperty()));
      }
      task.getRole().add("mediation_task");
    } 

    if(organization != null) {
      String color = organizationColors.get(organization);
      if(color == null) {
        color = ColorHelper.randomHexaColor();
        organizationColors.put(organization, color);
      }
      task.setColor(color);
      GenericModelHelper.findProperty("color", task.getProperty(), true).setValue(color);
    }else {
      String rcolor = ColorHelper.randomHexaColor();
      task.setColor(rcolor);
      GenericModelHelper.findProperty("color", task.getProperty(), true).setValue(rcolor);
    }


    task.getRole().add("task");
    task.getRole().add("temp");
    task.getProperty().add(GenericModelHelper.createProperty("monitorable", "true"));
    task.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(function)));

    // data
    if(GenericModelHelper.findProperty("Input Properties", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("Input Properties", function.getProperty()).clone());
    }
    if(GenericModelHelper.findProperty("Output Properties", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("Output Properties", function.getProperty()).clone());
    }
    // resources
    if(GenericModelHelper.findProperty("produces", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("produces", function.getProperty()).clone());
    }
    if(GenericModelHelper.findProperty("consumes", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("consumes", function.getProperty()).clone());
    }
    if(GenericModelHelper.findProperty("requires", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("requires", function.getProperty()).clone());
    }

    // impedance
    // TODO: generate automatically
    if(GenericModelHelper.findProperty("duration", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("duration", function.getProperty()).clone());
    }
    if(GenericModelHelper.findProperty("cost", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("cost", function.getProperty()).clone());
    }
    if(GenericModelHelper.findProperty("fixed cost", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("fixed cost", function.getProperty()).clone());
    }
    if(GenericModelHelper.findProperty("production capacity", function.getProperty()) != null) {
      task.getProperty().add((GJaxbProperty) GenericModelHelper.findProperty("production capacity", function.getProperty()).clone());
    }


    // add associated objective
    if(objective != null) {
      task.getProperty().add(GenericModelHelper.createProperty("associatedObjectives", "[{ 'name': '" + GenericModelHelper.getName(objective) + "', 'id': '" + objective.getId() + "' }]"));
    }

    // add invoked function
    if(function != null) {
      task.getProperty().add(GenericModelHelper.createProperty("invokedFunction", "[{ 'name': '" + GenericModelHelper.getName(function) + "', 'id': '" + function.getId() + "', 'type': '" + function.getType() + "' }]"));
    }

    // add belong to organization
    if(organization != null) {
      task.getProperty().add(GenericModelHelper.createProperty("belongToOrganization", "[{ 'name': '" + GenericModelHelper.getName(organization) + "', 'id': '" + organization.getId() + "', 'type': '" + organization.getType() + "' }]"));
    }
   
    return task;
  }


  public static void insertNodeBefore(GJaxbNode task, GJaxbNode nodeAfter, GenericModelManager mm) {
    List<GJaxbEdge> edgesBeforeTask = mm.findInputEdgesOfNode(nodeAfter);

    GJaxbGenericModel process = mm.getModels().get(0);

    edgesBeforeTask.stream().forEach(e -> { e.setTarget(task); });

    // create edge 2
    GJaxbEdge edge2 = new GJaxbEdge();
    edge2.setId("sequence_" + UUID.randomUUID().toString());
    edge2.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
    edge2.getRole().add("sequence_flow");
    edge2.getRole().add("temp");
    edge2.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(task) + " to " + GenericModelHelper.getName(nodeAfter)));
    edge2.setSource(task);
    edge2.setTarget(nodeAfter);
    process.getEdge().add(edge2);

    if(!containsNode(process, task)) {
      process.getNode().add(task);
    }

  }

  public static void insertNodeAfter(GJaxbNode task, GJaxbNode nodeBefore, GenericModelManager mm) {
    List<GJaxbEdge> edgesAfterTask = mm.findOutputEdgesOfNode(nodeBefore);


    GJaxbGenericModel process = mm.getModels().get(0);

    if(edgesAfterTask != null) {
      edgesAfterTask.stream().forEach(e -> { e.setSource(task); });
    }

    // create edge 1
    GJaxbEdge edge1 = new GJaxbEdge();
    edge1.setId("sequence_" + UUID.randomUUID().toString());
    edge1.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
    edge1.getRole().add("sequence_flow");
    edge1.getRole().add("temp");
    edge1.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(nodeBefore) + " to " + GenericModelHelper.getName(task)));
    edge1.setSource(nodeBefore);
    edge1.setTarget(task);
    process.getEdge().add(edge1);



    if(!containsNode(process, task)) {
      process.getNode().add(task);
    }

  }

  public static boolean containsNode(GJaxbGenericModel process, GJaxbNode node) {
    if(process.getNode().stream().filter(n -> { return n.getId().equals(node.getId()); }).count() > 0) {
      return true;
    }
    return false;
  }




  public static boolean containsEdge(GJaxbGenericModel process, GJaxbEdge edge) {
    for(GJaxbEdge e: process.getEdge()) {
      if(edge.getSource() != null && edge.getTarget() != null) {
        if(e.getSource() != null && e.getSource().getId().equals(edge.getSource().getId()) &&
            e.getTarget() != null && e.getTarget().getId().equals(edge.getTarget().getId())) {
          return true;
        }
      } else if(edge.getSource() == null && edge.getTarget() != null) {
        if(e.getSource() == null &&
            e.getTarget() != null && e.getTarget().getId().equals(edge.getTarget().getId())) {
          return true;
        }
      } else if(edge.getSource() != null && edge.getTarget() == null) {
        if(e.getSource() != null && e.getSource().getId().equals(edge.getSource().getId()) &&
            e.getTarget() == null) {
          return true;
        }
      } else if(edge.getSource() == null && edge.getTarget() == null) {
        if(e.getSource() == null && e.getTarget() == null) {
          return true;
        }
      }
    }
    return false;
  }

  public static GJaxbNode findProject(String collaborationName, String knowledgeSpaceName, CoreGovClient coreClient) throws QueryFault {


    GJaxbQuery request = new GJaxbQuery();
    request.setSelectedKnowledgeSpace(new fr.emac.gind.gov.core_gov.GJaxbSelectedKnowledgeSpace());
    request.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
    request.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);
    String name = knowledgeSpaceName.replace(" KnowledgeSpace", "");
    request.setQuery("Match (n:project:`" + collaborationName + "`:`" + knowledgeSpaceName + "` { property_name : '" + name + "' }) return n;");
    GJaxbQueryResponse response = coreClient.query(request);
    if(response != null && response.getSingle() != null && response.getSingle().getGenericModel() != null && response.getSingle().getGenericModel().getNode().size()>0) {
      return response.getSingle().getGenericModel().getNode().get(0);
    } else {
      return null;
    }
  }

  public static void removeNodeFromProcess(GJaxbNode startEvent,
      GenericModelManager mmp) {
    mmp.findOutputEdgesOfNode(startEvent).clear();
    mmp.getModels().forEach(m ->{
      m.getNode().remove(startEvent);
    });
    //match (n:startEvent)-[r]-(m) delete n,r
  }

  public static void insertNodeInEdge(GJaxbNode task, GJaxbEdge edge,  GenericModelManager merged_mm) {
    GJaxbNode target = edge.getTarget();
    GJaxbGenericModel process = merged_mm.getModels().get(0);

    edge.setTarget(task);
    edge.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(edge.getSource()) + " to " + GenericModelHelper.getName(task)));

    // create edge 1
    GJaxbEdge edge1 = new GJaxbEdge();
    edge1.setId("sequence_" + UUID.randomUUID().toString());
    edge1.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
    edge1.getRole().add("sequence_flow");
    edge1.getRole().add("temp");
    edge1.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(task) + " to " + GenericModelHelper.getName(target)));
    edge1.setSource(task);
    edge1.setTarget(target);
    process.getEdge().add(edge1);

    if(!containsNode(process, task)) {
      process.getNode().add(task);
    }

  }

  public static GJaxbGenericModel extractProcess(CoreGov coreClient,
      String processId, String startEventId, GJaxbEffectiveMetaModel metaModel,
      String collaborationName, String knowledgeSpaceName) throws Exception {

    GJaxbGenericModel process = null;
    String where = " where (";
    String or = "";

    //    for(fr.emac.gind.modeler.metamodel.GJaxbEdge edge: metaModel.getEdge()) {
    //      where = where + or + "r1.type = '" + edge.getType().toString().trim() + "'";
    //      or = " OR ";
    //    }
    //    where = where + ") AND (";
    //    or = "";
    for(fr.emac.gind.modeler.metamodel.GJaxbNode node: metaModel.getNode()) {
      where = where + or + "n1.type = '" + node.getType().toString().trim() + "'";
      or = " OR ";

    }
    where = where + ") AND (";
    or = "";
    for(fr.emac.gind.modeler.metamodel.GJaxbNode node: metaModel.getNode()) {
      where = where + or + "n2.type = '" + node.getType().toString().trim() + "'";
      or = " OR ";
    }
    where = where + " )";

    if(processId != null) {

      LOG.debug("processId = " + processId);

      GJaxbQuery q = new GJaxbQuery();
      q.setQuery("match (n1:startEvent { property_processId : '" + processId + "' }) OPTIONAL MATCH (n1 { node_status : 'ACTIF' })-[r1*]->(n2 { node_status : 'ACTIF' }) " + where + " return distinct n1, r1, n2");
      GJaxbSelectedKnowledgeSpace ck = new GJaxbSelectedKnowledgeSpace();
      ck.setCollaborationName(collaborationName);
      ck.setKnowledgeName(knowledgeSpaceName);
      q.setSelectedKnowledgeSpace(ck);
      q.setViewName("process");

      GJaxbQueryResponse res = coreClient.query(q);
      process = res.getSingle().getGenericModel();

    } else if(startEventId != null) {
      LOG.debug("startEventId = " + startEventId);

      GJaxbQuery q = new GJaxbQuery();
      q.setQuery("match (n1:startEvent { modelNodeId   : '" + startEventId + "_c__" + collaborationName + "_k__" + knowledgeSpaceName + "' }) OPTIONAL MATCH (n1 { node_status : 'ACTIF' })-[r1*]->(n2 { node_status : 'ACTIF' }) " + where + " return distinct n1, r1, n2");
      GJaxbSelectedKnowledgeSpace ck = new GJaxbSelectedKnowledgeSpace();
      ck.setCollaborationName(collaborationName);
      ck.setKnowledgeName(knowledgeSpaceName);
      q.setSelectedKnowledgeSpace(ck);
      q.setViewName("process");

      GJaxbQueryResponse res = coreClient.query(q);
      process = res.getSingle().getGenericModel();

    }


    // get isolated node like risk for example
    GJaxbEffectiveMetaModel metaModelWithoutTasksAndEvents = (GJaxbEffectiveMetaModel) metaModel.clone(); 
    metaModelWithoutTasksAndEvents.getNode().removeIf(metaNode -> { return metaNode.getRole().contains("task") || metaNode.getRole().contains("event") || metaNode.getRole().contains("gateway"); }); 


    GJaxbQuery q = new GJaxbQuery();
    q.setQuery(ModelsGovImpl.createExtractionRuleFromMetaModelByType(metaModelWithoutTasksAndEvents, collaborationName, knowledgeSpaceName));
    GJaxbSelectedKnowledgeSpace ck = new GJaxbSelectedKnowledgeSpace();
    ck.setCollaborationName(collaborationName);
    ck.setKnowledgeName(knowledgeSpaceName);
    q.setSelectedKnowledgeSpace(ck);
    q.setViewName("process");

    GJaxbQueryResponse queryResp = coreClient.query(q);
    if(queryResp != null && queryResp.getSingle() != null) {
      GJaxbGenericModel extra_response = queryResp.getSingle().getGenericModel();

      if(extra_response != null) {
        if(extra_response.getNode() != null) {
          for(GJaxbNode n: extra_response.getNode()) { 
            if(!containsNode(process, n)) {
              process.getNode().add(n);
            }
          }
        }
        if(extra_response.getEdge() != null) {
          for(GJaxbEdge e: extra_response.getEdge()) { 
            if(!containsEdge(process, e)) {
              process.getEdge().add(e);
            }
          }
        }
      }
    }
    process.getEdge().removeIf(edge -> {
      if(edge!= null && edge.getType() != null)
        return edge.getType().toString().equals("isParentOf"); 
      else return false;}
        );


    return process;
  }


  public static Optional<String> findProcessName(GJaxbGenericModel process) throws Exception {

    GenericModelManager mm = new GenericModelManager(process);

    List<GJaxbNode> startEvents = mm.getNodesByRoles("startEvent");
    Optional<String> processName = startEvents.stream().map(startEvent -> {
      GJaxbProperty processNameInSE = GenericModelHelper.findProperty("processName", startEvent.getProperty(), true);
      return processNameInSE.getValue();
    }).filter(processNameInStartEvent -> {
      return processNameInStartEvent != null && !processNameInStartEvent.trim().isEmpty();
    }).findFirst();


    return processName;
  }


  public static Optional<String> findProcessId(GJaxbGenericModel process) throws Exception {

    GenericModelManager mm = new GenericModelManager(process);

    List<GJaxbNode> startEvents = mm.getNodesByRoles("startEvent");
    Optional<String> processId = startEvents.stream().map(startEvent -> {
      GJaxbProperty processIdInSE = GenericModelHelper.findProperty("processId", startEvent.getProperty(), true);
      return processIdInSE.getValue();
    }).filter(processIdInStartEvent -> {
      return processIdInStartEvent != null && !processIdInStartEvent.trim().isEmpty();
    }).findFirst();


    return processId;
  }
}
