package fr.emac.gind.workflow.generator;

/*
 * #%L
 * abstract-workflow-deduction
 * %%
 * Copyright (C) 2014 - 2016 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.Map.Entry;
import java.util.Stack;

import javax.xml.namespace.QName;

import org.json.JSONArray;

import fr.emac.gind.commons.utils.color.ColorHelper;
import fr.emac.gind.commons.utils.jaxb.JSONJAXBContext;
import fr.emac.gind.commons.utils.jaxb.SOAException;
import fr.emac.gind.gov.core.client.CoreGovClient;
import fr.emac.gind.gov.core_gov.GJaxbQuery;
import fr.emac.gind.gov.core_gov.GJaxbQueryResponse;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.models.generic.modeler.GenericModelHelper;
import fr.emac.gind.models.generic.modeler.GenericModelManager;
import fr.emac.gind.workflow.report.GJaxbReport;
import fr.emac.gind.workflow.report.GJaxbStatusType;
import fr.gind.emac.gov.core_gov.QueryFault;

/**
 *
 *
 * @author: Nicolas Salatge (nicolas.salatge@mines-albi.fr)
 */
public class ProcessDeductionHelper {

  private static  Map<GJaxbNode, String> organizationColors = new HashMap<GJaxbNode, String>();

  public static TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>> createMapObjectivesByEffectOrRiskByPriority(
      String currentCollaborationName, String currentKnowledgeSpaceName,
      GJaxbReport report, JSONArray ordering, CoreGovClient coreClient) throws SOAException, QueryFault {
    TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>> objectivesByEffectOrRiskByPiority = new TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>>();
    for(int i = 0; i < ordering.length(); i++) {
      GJaxbNode effectOrRisk = JSONJAXBContext.getInstance().unmarshall("{ \"node\" : "  + ordering.getJSONObject(i).getJSONObject("object").toString() + " }", GJaxbNode.class);

      // find associated objectives (treated or prevent)
      Map<GJaxbNode, List<GJaxbNode>> objectivesByEffectOrRisk = new HashMap<GJaxbNode, List<GJaxbNode>>();
      GJaxbGenericModel assoObj = coreClient.singleQuery(new String("match (n1)-[:`{http://fr.emac.gind/core-model}Concerns`]->(n2 { modelNodeId: '" + effectOrRisk.getId() + "_c__${collaboration}_k__${knowledgeSpace}' }) return n1").replace("${collaboration}", currentCollaborationName).replace("${knowledgeSpace}", currentKnowledgeSpaceName), currentCollaborationName, currentKnowledgeSpaceName);
      if(assoObj == null || assoObj.getNode() == null || assoObj.getNode().size() == 0) {
        report.getResult().getAnalyticReport().setStatus(GJaxbStatusType.INCOMPLETE);
        report.getResult().getAnalyticReport().setReportMessage(report.getResult().getAnalyticReport().getReportMessage() + "Error: Impossible to find at least one objective corresponding to this effect or risk: " + GenericModelHelper.getName(effectOrRisk) + "\n");
      }
      if(assoObj != null && assoObj.getNode() != null) {
        assoObj.getNode().forEach(obj -> { obj.setUserData("associatedEffectOrRisk", effectOrRisk); });
        objectivesByEffectOrRisk.put(effectOrRisk, assoObj.getNode());
      }
      objectivesByEffectOrRiskByPiority.put(new Integer(i), objectivesByEffectOrRisk);
    }
    return objectivesByEffectOrRiskByPiority;
  }

  public static Map<GJaxbNode, GJaxbGenericModel> createMapPathByObjective(String currentCollaborationName,
      String currentKnowledgeSpaceName, GJaxbReport report,
      TreeMap<Integer, Map<GJaxbNode, List<GJaxbNode>>> objectivesByEffectOrRiskByPiority, CoreGovClient coreClient)
          throws QueryFault {
    Map<GJaxbNode, GJaxbGenericModel> pathByObj = new HashMap<GJaxbNode, GJaxbGenericModel>(); 
    for(Map<GJaxbNode, List<GJaxbNode>> effectOrRiskByObjectives: objectivesByEffectOrRiskByPiority.values()) {
      for(List<GJaxbNode> objectives: effectOrRiskByObjectives.values()) {
        for(GJaxbNode obj: objectives) {

          List<GJaxbGenericModel> pathsModel = coreClient.multipleQuery(new String("match p = allShortestPaths((a:organization)-[:`{http://fr.emac.gind/core-model}Provides`|:`{http://fr.emac.gind/core-model}Satisfies`|:`{http://fr.emac.gind/core-model}Concerns`|:near*]-(o:objective { modelNodeId : '" + obj.getId() + "_c__${collaboration}_k__${knowledgeSpace}'} )) " + 
              " where filter(x in nodes(p) where x:`${collaboration}`:`${knowledgeSpace}`) and any(c in nodes(p) where c:function and c.node_status = 'ACTIF') and any(o in nodes(p) where o:objective and o.node_status = 'ACTIF' and o.modelNodeId = '" + obj.getId() + "_c__${collaboration}_k__${knowledgeSpace}') and a.node_status = 'ACTIF' and o.node_status = 'ACTIF' return p  ").replace("${collaboration}", currentCollaborationName).replace("${knowledgeSpace}", currentKnowledgeSpaceName), currentCollaborationName, currentKnowledgeSpaceName);

          if(pathsModel != null && !pathsModel.isEmpty()) {
            Entry<Double, GJaxbGenericModel> bestPath = findBestPath(pathsModel);


            pathByObj.put(obj, bestPath.getValue());
          } else {
            pathByObj.put(obj, null);
          }
        }
      }
    }
    return pathByObj;
  }



  public static Entry<Double, GJaxbGenericModel> findBestPath(
      List<GJaxbGenericModel> pathsModel) {

    TreeMap<Double, List<GJaxbGenericModel>> orderingPathsByPertinenceIndice = new TreeMap<Double, List<GJaxbGenericModel>>();
    double pertinenceIndiceMax = 0;
    GJaxbGenericModel bestPath = null;
    for(GJaxbGenericModel path: pathsModel) {
      double pertinenceIndice = calculatePertinenceIndice(path);


      if(orderingPathsByPertinenceIndice.get(pertinenceIndice) == null) {
        orderingPathsByPertinenceIndice.put(pertinenceIndice, new ArrayList<GJaxbGenericModel>());
      }
      orderingPathsByPertinenceIndice.get(pertinenceIndice).add(path);
    }

    pertinenceIndiceMax = orderingPathsByPertinenceIndice.lastKey();
    List<GJaxbGenericModel> bestPaths = orderingPathsByPertinenceIndice.get(pertinenceIndiceMax);


    int numberOfNodesMin = bestPaths.get(0).getNode().size();
    for(GJaxbGenericModel path: bestPaths) {
      if(path.getNode().size() <= numberOfNodesMin) {
        numberOfNodesMin = path.getNode().size();
        bestPath = path;
      }
    }


    Entry<Double, GJaxbGenericModel> entry = new AbstractMap.SimpleEntry<Double, GJaxbGenericModel>(pertinenceIndiceMax, bestPath);
    return entry;
  }

  public static double calculatePertinenceIndice(GJaxbGenericModel path) {
    double res = 1;
    for(GJaxbEdge edge: path.getEdge()) {
      GJaxbProperty satisfiesProp = GenericModelHelper.findProperty("coverage", edge.getProperty());
      if(satisfiesProp != null && satisfiesProp.getValue() != null) {
        double coverage = Double.parseDouble(satisfiesProp.getValue())/100;
        res = res * coverage;
      } else {
        GJaxbProperty nearProp = GenericModelHelper.findProperty("near at", edge.getProperty());
        if(nearProp != null && nearProp.getValue() != null) {
          double near = Double.parseDouble(nearProp.getValue())/100;
          res = res * near;
        } 
      }
    }
    return res*100;
  }


  public static String printPath(GJaxbGenericModel path) {
    String pathS = "";
    GJaxbNode currentNode = findStartNode(path.getNode()); 
    boolean reverse = false;
    List<GJaxbEdge> edges = new ArrayList<GJaxbEdge>(path.getEdge());
    while(currentNode != null) {
      GJaxbEdge currentEdge = findEdgeWhereSourceIs(edges, currentNode);
      if(currentEdge != null && reverse == false) {
        pathS = pathS + "(" + GenericModelHelper.getName(currentNode) + ")" + "->";
        currentNode = currentEdge.getTarget();
        edges.remove(currentEdge);
      } else {
        currentEdge = findEdgeWhereTargetIs(edges, currentNode);
        if(currentEdge != null) {
          reverse = true;
          pathS = pathS + "(" + GenericModelHelper.getName(currentNode) + ")" + "<-";
          currentNode = currentEdge.getSource();
          edges.remove(currentEdge);
        } else {
          pathS = pathS + "(" + GenericModelHelper.getName(currentNode) + ")";
          currentNode = null;
        }
      } 
    }
    //
    return pathS;
  }

  private static GJaxbEdge findEdgeWhereSourceIs(List<GJaxbEdge> edges,
      GJaxbNode currentNode) {
    for(GJaxbEdge edge: edges) {
      if(edge.getSource().getId().equals(currentNode.getId())) {
        return edge;
      }
    }
    return null;
  }

  private static GJaxbEdge findEdgeWhereTargetIs(List<GJaxbEdge> edges,
      GJaxbNode currentNode) {
    for(GJaxbEdge edge: edges) {
      if(edge.getTarget().getId().equals(currentNode.getId())) {
        return edge;
      }
    }
    return null;
  }

  private static GJaxbNode findStartNode(List<GJaxbNode> nodes) {
    for(GJaxbNode node: nodes) {
      GJaxbProperty first = GenericModelHelper.findProperty("firstInPath", node.getProperty());
      if(first != null && first.getValue() != null && Boolean.valueOf(first.getValue()) == true) {
        return node;
      }
    }
    return null;
  }


  public static void showProcess(GJaxbNode startEvent,
      TreeMap<Integer, Entry<GJaxbNode, List<GJaxbNode>>> orderingGateways, GJaxbNode endEvent) {
    List<Object> process = new ArrayList<Object>();
    process.add(startEvent);
    for(Entry<GJaxbNode, List<GJaxbNode>> gateway: orderingGateways.values()) {
      if(gateway.getValue().size() > 1) {
        process.add(gateway.getValue());
      } else {
        process.add(gateway.getValue().get(0));
      }
    }
    process.add(endEvent);


    int i = 1;
    for(Object step: process) {
      if(step instanceof GJaxbNode) {

      } else {
        System.out.print("step " + i + ": ");
        if(step != null) {
          for(GJaxbNode node: (List<GJaxbNode>)step) {
            System.out.print(GenericModelHelper.getName(node) + ", ");
          }
        }

      }
      i = i + 1;
    }


  }

  public static GJaxbNode findOrganization(GenericModelManager manager) {
    List<GJaxbNode> organizations = manager.getNodesByType(new QName("http://fr.emac.gind/crisis_functions", "Actor")); 
    for(GJaxbNode organization: organizations) {
      if(manager.findInputEdgesOfNode(organization).isEmpty()) {
        return organization;
      }
    }
    return null;
  }








  public static GJaxbNode findFirstService(GenericModelManager manager, GJaxbNode organization) {
    List<GJaxbNode> services = manager.getNodesByRoles("function"); 
    for(GJaxbNode service: services) {
      List<GJaxbEdge> edges = manager.findInputEdgesOfNode(service);
      for(GJaxbEdge edge: edges) {
        if(edge.getSource() == organization && edge.getType().equals(new QName("http://fr.emac.gind/core-model", "Provides"))) {
          return service;
        }
      }
    }
    return null;
  }


  public static GJaxbNode createTask(GJaxbNode function, GJaxbNode objective, GJaxbNode organization)
      throws SOAException {
    GJaxbNode task = new GJaxbNode();
    task.setId("task_" + UUID.randomUUID().toString());
    if(function.getRole().contains("human_function")) {
      task.setType(new QName("http://fr.emac.gind/core-model", "HumanTask"));
      if(GenericModelHelper.findProperty("Attributed to", function.getProperty()) != null) {
        task.getProperty().add(GenericModelHelper.findProperty("Attributed to", function.getProperty()));
      }
      if(GenericModelHelper.findProperty("Priority", function.getProperty()) != null) {
        task.getProperty().add(GenericModelHelper.findProperty("Priority", function.getProperty()));
      }
      task.getRole().add("human_task");
    } else if(function.getRole().contains("computer_function")) {
      task.setType(new QName("http://fr.emac.gind/core-model", "ComputerTask"));
      task.getRole().add("computer_task");
    } else if(function.getRole().contains("mediation_function")) {
      task.setType(new QName("http://fr.emac.gind/core-model", "MediationTask"));
      if(GenericModelHelper.findProperty("type", function.getProperty()) != null) {
        task.getProperty().add( GenericModelHelper.findProperty("type", function.getProperty()));
      }
      task.getRole().add("mediation_task");
    } 

    if(organization != null) {
      String color = organizationColors.get(organization);
      if(color == null) {
        color = ColorHelper.randomHexaColor();
        organizationColors.put(organization, color);
      }
      task.setColor(color);
      GenericModelHelper.findProperty("color", task.getProperty(), true).setValue(color);
    }

    task.getRole().add("task");
    task.getRole().add("temp");
    task.getProperty().add(GenericModelHelper.createProperty("monitorable", "true"));
    task.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(function)));

    // data
    if(GenericModelHelper.findProperty("Input Properties", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("Input Properties", function.getProperty()));
    }
    if(GenericModelHelper.findProperty("Output Properties", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("Output Properties", function.getProperty()));
    }
    // resources
    if(GenericModelHelper.findProperty("produces", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("produces", function.getProperty()));
    }
    if(GenericModelHelper.findProperty("consumes", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("consumes", function.getProperty()));
    }
    if(GenericModelHelper.findProperty("requires", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("requires", function.getProperty()));
    }

    // impedance
    // TODO: generate automatically
    if(GenericModelHelper.findProperty("duration", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("duration", function.getProperty()));
    }
    if(GenericModelHelper.findProperty("cost", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("cost", function.getProperty()));
    }
    if(GenericModelHelper.findProperty("fixed cost", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("fixed cost", function.getProperty()));
    }
    if(GenericModelHelper.findProperty("production capacity", function.getProperty()) != null) {
      task.getProperty().add(GenericModelHelper.findProperty("production capacity", function.getProperty()));
    }


    // add associated objective
    if(objective != null) {
      task.getProperty().add(GenericModelHelper.createProperty("associatedObjectives", "[{ 'name': '" + GenericModelHelper.getName(objective) + "', 'id': '" + objective.getId() + "' }]"));
    }

    // add invoked function
    if(function != null) {
      task.getProperty().add(GenericModelHelper.createProperty("invokedFunction", "[{ 'name': '" + GenericModelHelper.getName(function) + "', 'id': '" + function.getId() + "', 'type': '" + function.getType() + "' }]"));
    }

    // add belong to organization
    if(organization != null) {
      task.getProperty().add(GenericModelHelper.createProperty("belongToOrganization", "[{ 'name': '" + GenericModelHelper.getName(organization) + "', 'id': '" + organization.getId() + "', 'type': '" + organization.getType() + "' }]"));
    }

    return task;
  }


  public static void insertNodeBefore(GJaxbNode task, GJaxbNode nodeAfter, GenericModelManager mm) {
    GJaxbEdge edgeToDelete = mm.findInputEdgesOfNode(nodeAfter).get(0);
    GJaxbNode nodeBefore = edgeToDelete.getSource();

    GJaxbGenericModel process = mm.getModels().get(0);
    process.getEdge().remove(edgeToDelete);


    // create edge 1
    GJaxbEdge edge1 = new GJaxbEdge();
    edge1.setId("sequence_" + UUID.randomUUID().toString());
    edge1.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
    edge1.getRole().add("sequence_flow");
    edge1.getRole().add("temp");
    edge1.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(nodeBefore) + " to " + GenericModelHelper.getName(task)));
    edge1.setSource(nodeBefore);
    edge1.setTarget(task);
    process.getEdge().add(edge1);


    // create edge 2
    GJaxbEdge edge2 = new GJaxbEdge();
    edge2.setId("sequence_" + UUID.randomUUID().toString());
    edge2.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
    edge2.getRole().add("sequence_flow");
    edge2.getRole().add("temp");
    edge2.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(task) + " to " + GenericModelHelper.getName(nodeAfter)));
    edge2.setSource(task);
    edge2.setTarget(nodeAfter);
    process.getEdge().add(edge2);

    if(!process.getNode().contains(task)) {
      process.getNode().add(task);
    }

  }

  public static void insertNodeAfter(GJaxbNode task, GJaxbNode nodeBefore, GenericModelManager mm) {
    GJaxbEdge edgeToDelete = mm.findOutputEdgesOfNode(nodeBefore).size() > 0 ? mm.findOutputEdgesOfNode(nodeBefore).get(0) : null;
    GJaxbNode nodeAfter = null;
    if(edgeToDelete != null) {
      nodeAfter = edgeToDelete.getTarget();
    }

    GJaxbGenericModel process = mm.getModels().get(0);

    if(edgeToDelete != null) {
      process.getEdge().remove(edgeToDelete);
    }

    // create edge 1
    GJaxbEdge edge1 = new GJaxbEdge();
    edge1.setId("sequence_" + UUID.randomUUID().toString());
    edge1.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
    edge1.getRole().add("sequence_flow");
    edge1.getRole().add("temp");
    edge1.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(nodeBefore) + " to " + GenericModelHelper.getName(task)));
    edge1.setSource(nodeBefore);
    edge1.setTarget(task);
    process.getEdge().add(edge1);


    if(nodeAfter != null) {
      // create edge 2
      GJaxbEdge edge2 = new GJaxbEdge();
      edge2.setId("sequence_" + UUID.randomUUID().toString());
      edge2.setType(new QName("http://fr.emac.gind/core-model", "SequenceFlow"));
      edge2.getRole().add("sequence_flow");
      edge2.getRole().add("temp");
      edge2.getProperty().add(GenericModelHelper.createProperty("name", GenericModelHelper.getName(task) + " to " + GenericModelHelper.getName(nodeAfter)));
      edge2.setSource(task);
      edge2.setTarget(nodeAfter);
      process.getEdge().add(edge2);
    }

    if(!process.getNode().contains(task)) {
      process.getNode().add(task);
    }

  }

  public static GJaxbNode findProject(String collaborationName, String knowledgeSpaceName, CoreGovClient coreClient) throws QueryFault {


    GJaxbQuery request = new GJaxbQuery();
    request.setSelectedKnowledgeSpace(new fr.emac.gind.gov.core_gov.GJaxbSelectedKnowledgeSpace());
    request.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
    request.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);
    String name = knowledgeSpaceName.replace(" KnowledgeSpace", "");
    request.setQuery("Match (n:project:`" + collaborationName + "`:`" + knowledgeSpaceName + "` { property_name : '" + name + "' }) return n;");
    GJaxbQueryResponse response = coreClient.query(request);
    if(response != null && response.getSingle() != null && response.getSingle().getGenericModel() != null && response.getSingle().getGenericModel().getNode().size()>0) {
      return response.getSingle().getGenericModel().getNode().get(0);
    } else {
      return null;
    }
  }


}
