/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.bundle;

import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.CaseFormat;

import fr.emac.gind.campaign.manager.server.CampaignManagerWebService;
import fr.emac.gind.commons.utils.io.DirectoryListener;
import fr.emac.gind.commons.utils.net.IPUtil;
import fr.emac.gind.commons.utils.rest.StaticRestServer;
import fr.emac.gind.commons.utils.ws.LocalRegistry;
import fr.emac.gind.commons.utils.ws.SPIWebServicePrimitives;
import fr.emac.gind.commons.utils.ws.StaticJettyServer;
import fr.emac.gind.event.broker.EventBrokerWebService;
import fr.emac.gind.game_master.GameMasterWebService;
import fr.emac.gind.generic.application.DWApplicationService;
import fr.emac.gind.governance.GovernanceWebService;
import fr.emac.gind.h2gis.gis.H2GisWebService;
import fr.emac.gind.humantask.HumantaskWebService;
import fr.emac.gind.io.interpretation.engine.InterpretationEngineContainer;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.mock.endpoints.manager.MockEndpointManagerWebService;
import fr.emac.gind.monitoring.detection.DetectionWebService;
import fr.emac.gind.monitoring.server.MonitoringWebService;
import fr.emac.gind.osm.gis.OsmGisWebService;
import fr.emac.gind.r.iocal.RIOCALService;
import fr.emac.gind.r.ioda.RIODAService;
import fr.emac.gind.r.ioded.RIODEDService;
import fr.emac.gind.r.ioga.RIOGAService;
import fr.emac.gind.r.iome.RIOMEService;
import fr.emac.gind.r.ioplay.RIOPLAYService;
import fr.emac.gind.r.iored.RIOREDService;
import fr.emac.gind.r.iose.RIOSEService;
import fr.emac.gind.r.iosemit.RIOSEMITService;
import fr.emac.gind.r.iosepe.RIOSEPEService;
import fr.emac.gind.r.iota.RIOTAService;
import fr.emac.gind.r.iote.RIOTEService;
import fr.emac.gind.r.iowa.RIOWAService;
import fr.emac.gind.rio.bundle.config.ConfigurationUnifier;
import fr.emac.gind.sensors.controler.SensorControlerWebService;
import fr.emac.gind.sensors.manager.SensorManagerWebService;
import fr.emac.gind.storage.mongodb.EventStorageWebService;
import fr.emac.gind.timeseries.mongodb.server.TimeSeriesWebService;
import fr.emac.gind.workflow.engine.WorkflowContainer;

public class RIOSuiteAbstractBundle {
	
	private static final Logger LOG = LoggerFactory.getLogger(RIOSuiteAbstractBundle.class.getName());

	protected Configuration unifiedConf = null;
	protected Configuration globalConf = null;

	protected List<SPIWebServicePrimitives> webservices = new ArrayList<SPIWebServicePrimitives>();
	protected Map<Class<? extends SPIWebServicePrimitives>, Configuration> mapSpecificConfigSubComps = new HashMap<Class<? extends SPIWebServicePrimitives>, Configuration>();

	private final JSONObject allWebApplications = new JSONObject();

	protected DirectoryListener directoryListener = null;

	protected List<DWApplicationService> webappServicesToActivate = new ArrayList<DWApplicationService>();
	protected List<Class<? extends SPIWebServicePrimitives>> soapServicesToActivate = new ArrayList<Class<? extends SPIWebServicePrimitives>>();

	// Map of SOAP Services to launch by default, organized by wave number (1, 2, 3)
	private static final Map<Integer, List<Class<? extends SPIWebServicePrimitives>>> SOAP_SERVICES = Map.ofEntries(
			Map.entry(1, List.of(EventStorageWebService.class)),
			Map.entry(2,
					Arrays.asList(TimeSeriesWebService.class, CampaignManagerWebService.class,
							GovernanceWebService.class, EventBrokerWebService.class)),
			Map.entry(3,
					Arrays.asList(HumantaskWebService.class, WorkflowContainer.class, MockEndpointManagerWebService.class,
							SensorManagerWebService.class, SensorControlerWebService.class, H2GisWebService.class, OsmGisWebService.class,
							DetectionWebService.class, GameMasterWebService.class)),
			Map.entry(4, Arrays.asList(InterpretationEngineContainer.class, MonitoringWebService.class)));

	protected RIOGAService rioga = null;
	protected RIOWAService riowa = null;

	public RIOSuiteAbstractBundle(List<DWApplicationService> webappServices,
			List<Class<? extends SPIWebServicePrimitives>> wsList) throws Exception {
		if (webappServices.isEmpty()) {
			// No filter on services, we activate all services
			webappServicesToActivate.add(new RIOGAService());
			webappServicesToActivate.add(new RIODAService());
			webappServicesToActivate.add(new RIOTEService());
			webappServicesToActivate.add(new RIOMEService());
			webappServicesToActivate.add(new RIOSEPEService());
			webappServicesToActivate.add(new RIOPLAYService());
			webappServicesToActivate.add(new RIOSEService());
			webappServicesToActivate.add(new RIOSEMITService());
			webappServicesToActivate.add(new RIOWAService());
			webappServicesToActivate.add(new RIOTAService());
			webappServicesToActivate.add(new RIOREDService());
			webappServicesToActivate.add(new RIODEDService());
			webappServicesToActivate.add(new RIOCALService());
		} else {
			webappServicesToActivate = webappServices;
		}

		if (wsList == null || wsList.isEmpty()) {
			for (List<Class<? extends SPIWebServicePrimitives>> wave : SOAP_SERVICES.values()) {
				this.soapServicesToActivate.addAll(wave);
			}
		} else {
			this.soapServicesToActivate = wsList;
		}

	}

	public String getName() {
		return "r-iosuite";
	}

	public Configuration getUnifiedConf() {
		return unifiedConf;
	}

	public void setUnifiedConf(Configuration unifiedConf) throws Exception {
		this.unifiedConf = unifiedConf;
		this.globalConf = ConfigurationUnifier.desunify(null, this, null);
	}

	public List<Class<? extends SPIWebServicePrimitives>> getSoapServicesToActivate() {
		return soapServicesToActivate;
	}

	public void boot(Configuration unified_conf) throws Exception {
		this.testPortsAvailabilities(unified_conf);
		this.setUnifiedConf(unified_conf);

		if (this.unifiedConf == null) {
			this.unifiedConf = ConfigurationUnifier.unify(webappServicesToActivate, soapServicesToActivate);
		}

		// initialize python interpreter
		
		//File pytonhAndMongoConfigFile = new File(Configuration.CONFIG_PATH,
		//		"/generated/" + "python" + "/conf/" + "python_config.properties");
		Configuration mongoConf = ConfigurationUnifier.desunify("all", this, null);
		//PythonManager.getInstance().init(pythonAndMongoConf);
		

		if (this.unifiedConf.getProperties().get("mongodb-database-embedded") != null
				&& Boolean.valueOf(this.unifiedConf.getProperties().get("mongodb-database-embedded"))) {
			LOG.info("Start embedded mongoDB");
			DWApplicationService.startEmbeddedMongDB(true, mongoConf);
		}

		// Start Web Services (SOAP)
		startBackendWebServices();

		// Start Webapps (DW REST Services)
		for (DWApplicationService service : this.webappServicesToActivate) {
			initService(service);
			if (service.getName().equals("r-ioga")) {
				this.rioga = (RIOGAService) service;
			} else if (service.getName().equals("r-iowa")) {
				this.riowa = (RIOWAService) service;
			}

		}
	}

	private void testPortsAvailabilities(Configuration suiteConfig) throws Exception {
		for (Entry<String, String> entry : suiteConfig.getProperties().entrySet()) {
			if (entry.getKey().endsWith("port") && !entry.getKey().equals("python-ml-engine-port")) {
				Integer defaultPort = null;
				try {
					defaultPort = Integer.parseInt(entry.getValue().trim());
				} catch (Exception e) {
					// do nothing
				}
				if (defaultPort != null && defaultPort > 0) {
					if (!IPUtil.available(defaultPort)) {
						throw new Exception("This port is not available '" + entry.getKey() + "' :" + defaultPort);
					}
				}

			}
		}

	}

	private void initService(DWApplicationService service) throws Exception {
		File configFile = new File(Configuration.CONFIG_PATH,
				"/generated/" + service.getName() + "/conf/" + "component_config.properties");
		Configuration serviceConf = ConfigurationUnifier.desunify(service.getName(), this, configFile);
		System.out.println("configFile: " + configFile.getCanonicalPath());
		service.boot(serviceConf);
	}

	public void addWebApplication(String name, String address) {
		this.allWebApplications.put(name, new JSONObject());
		this.allWebApplications.getJSONObject(name).put("address", address);
	}

	private void startBackendWebServices() throws Exception {
		// Start WS BackEnd

		// 1st wave
		LOG.debug("Start of wave 1");
		startWSByWaves(SOAP_SERVICES.get(1));
		Thread.sleep(600);
		LOG.debug("End of wave 1");

		// 2nd wave
		LOG.debug("Start of wave 2");
		startWSByWaves(SOAP_SERVICES.get(2));
		Thread.sleep(400);
		LOG.debug("End of wave 2");

		// 3nd wave
		LOG.debug("Start of wave 3");
		startWSByWaves(SOAP_SERVICES.get(3));
		Thread.sleep(400);
		LOG.debug("End of wave 3");

		// 4nd wave
		LOG.debug("Start of wave 4");
		startWSByWaves(SOAP_SERVICES.get(4));
		Thread.sleep(400);
		LOG.debug("End of wave 4");

	}

	private void startWSByWaves(List<Class<? extends SPIWebServicePrimitives>> wss) throws Exception {
		for (Class<? extends SPIWebServicePrimitives> wsServiceClass : wss) {

			if (soapServicesToActivate.contains(wsServiceClass)) {
				Configuration serviceConf = null;
				if (mapSpecificConfigSubComps.containsKey(wsServiceClass)) {
					serviceConf = mapSpecificConfigSubComps.get(wsServiceClass);
				} else {
					String confName = CaseFormat.LOWER_CAMEL.to(CaseFormat.LOWER_HYPHEN,
							wsServiceClass.getSimpleName());
					File componentConf = new File(Configuration.CONFIG_PATH,
							"/generated/" + confName + "/conf/" + "component_config.properties");

					serviceConf = ConfigurationUnifier.desunify(confName, this, componentConf);
				}

				Constructor<? extends SPIWebServicePrimitives> constructor = wsServiceClass.getConstructor();
				SPIWebServicePrimitives ws = constructor.newInstance();

				this.getWebservices().add(ws);

				try {
					String waiter = "*";
					LOG.debug("Try to start ws spi: " + ws.getClass());
					ExecutorService service = Executors.newSingleThreadExecutor();
					final Map<String, Object> wsConf = (Map)serviceConf.getProperties();
					Future<Boolean> res = service.submit(new Callable<Boolean>() {

						@Override
						public Boolean call() throws Exception {
							try {
								ws.start(wsConf);
								return true;
							} catch (Throwable e) {
								e.printStackTrace();
								throw new RuntimeException(
										"Impossible to start web service: " + wsServiceClass.getSimpleName(), e);
								
							}
						}

					});
					while(!res.isDone()) {
						Thread.sleep(2000);
						LOG.debug("Try to start ws spi: " + ws.getClass() + " -> " + waiter);
						waiter = waiter + "*";
					}
				} catch (Throwable e) {
					e.printStackTrace();
					throw new RuntimeException("Impossible to start web service: " + wsServiceClass.getSimpleName(), e);
				}

			}
		}
	}

	public void run() throws Exception {

		for (DWApplicationService service : this.webappServicesToActivate) {
			try {
				LOG.debug("try to start: " + service.getName());
				service.run("server",
                        new File(service.getYmlConfig().toURI()).getCanonicalFile().toString());
				LOG.debug(service.getName() + " started");
			} catch (Throwable e) {
				LOG.error("Impossible to run dw application: " + service.getName());
				throw new RuntimeException("Impossible to run dw application: " + service.getName(), e);
			}
			this.addWebApplication(service.getName(), service.getApplicationUrl());
		}

		// set all web applications in each components
		for (DWApplicationService service : this.webappServicesToActivate) {
			service.getApplicationContext().setAllWebApplications(allWebApplications);
		}

		this.directoryListener = new DirectoryListener(new File("./target/listener"), this.globalConf);
		ExecutorService service = Executors.newSingleThreadExecutor();
		service.execute(this.directoryListener);

	}

	public List<SPIWebServicePrimitives> getWebservices() {
		return webservices;
	}

	public void setSpecificConfigurationOfSubComponents(Class<? extends SPIWebServicePrimitives> keyConf,
			Configuration conf) {
		this.mapSpecificConfigSubComps.put(keyConf, conf);
	}

	public RIOGAService getRioga() {
		return rioga;
	}

	public RIOWAService getRiowa() {
		return riowa;
	}

	public List<DWApplicationService> getWebappServicesToActivate() {
		return webappServicesToActivate;
	}

	public void stop() throws Exception {

		for (SPIWebServicePrimitives ws : webservices) {
			ws.getAllAddresses().forEach(address -> LOG.debug("stopping ws " + address));
			ws.stop();
		}

		for (DWApplicationService service : webappServicesToActivate) {
			LOG.debug("Stopping " + service.getName());
			service.stop();
		}

		// stop all servers running 
		StaticJettyServer.getInstance().stop(true);
		StaticRestServer.getInstance().stop();
		LocalRegistry.getInstance().clear();
	}

}
