/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.bundle.config;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.CaseFormat;

import fr.emac.gind.commons.utils.io.FileUtil;
import fr.emac.gind.commons.utils.ws.SPIWebServicePrimitives;
import fr.emac.gind.generic.application.DWApplicationService;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.rio.bundle.RIOSuiteAbstractBundle;

public class ConfigurationUnifier {

	private static final Logger LOG = LoggerFactory.getLogger(ConfigurationUnifier.class.getName());

	public enum Type {
		WEBAPP,
		WS,
		COMMON
	}

	public static Configuration desunify(String componentName, RIOSuiteAbstractBundle bundle, File configFile) throws Exception {
		Configuration unifiedConf = bundle.getUnifiedConf();

		List<GJaxbProperty> componentProps = new ArrayList<GJaxbProperty>();
		for(Entry<String, String> prop: unifiedConf.getProperties().entrySet()) {
			if(prop.getKey().contains("...")) {
				if(componentName != null && prop.getKey().contains(componentName + "...")) {
					// specific component property
					componentProps.add(GenericModelHelper.createProperty(prop.getKey(), prop.getValue()));
				} else if(componentName != null && componentName.equalsIgnoreCase("all")) {
					componentProps.add(GenericModelHelper.createProperty(prop.getKey(), prop.getValue()));
				}
			} else {
				// common property
				componentProps.add(GenericModelHelper.createProperty(prop.getKey(), prop.getValue()));
			}
		}

		// replace key by value in property
		replaceKeyByValue(componentProps, unifiedConf);

		Map<String, String> comPropsMap = new HashMap<String, String>();
		if(componentName == null || !componentName.equalsIgnoreCase("all")) {
			for(GJaxbProperty prop: componentProps) {
				if(prop.getName().contains("...")) {
					prop.setName(prop.getName().substring(prop.getName().indexOf("...") + "...".length()));
				}
				if(comPropsMap.get(prop.getName()) == null) {
					comPropsMap.put(prop.getName(), prop.getValue());
				} else {
					LOG.warn("conf prop already exist '" + prop.getName() + "' with value: " + comPropsMap.get(prop.getName()));
				}
			}
		} else if(componentName.equalsIgnoreCase("all")) {
			for(GJaxbProperty prop: componentProps) {
				comPropsMap.put(prop.getName(), prop.getValue()); 
			}
		}

		Map<String, String> globalProps = new HashMap<String, String>(comPropsMap);
		List<PropertiesGroup> groups = sortPropertiesByGroup(comPropsMap);

		StringBuffer buffer = new StringBuffer();
		buffer.append(addDiese(("# " + componentName + " properties #").length()));
		buffer.append("# " + componentName + " properties #" + "\n");
		buffer.append(addDiese(("# " + componentName + " properties #").length()));
		for(PropertiesGroup group: groups) {
			if(group.getGroupName() != null && !group.getGroupName().isBlank()) {
				buffer.append("\n# " + group.getGroupName() + "\n"); 
			}
			for(GJaxbProperty prop: group.getProperties()) {
				buffer.append(prop.getName() + " = " + prop.getValue() + "\n"); 
			}
			buffer.append("\n"); 
		}


		Configuration componentConf = null;
		if(configFile != null) {
			configFile.getParentFile().mkdirs();
			configFile.createNewFile();
			FileUtil.setContents(configFile, buffer.toString());
			componentConf = new Configuration(configFile.toURI().toURL());
		} else {
			componentConf = new Configuration(globalProps);
		}
		return componentConf;
	}


	private static void replaceKeyByValue(List<GJaxbProperty> componentProps, Configuration unifiedConf) {
		for(GJaxbProperty prop: componentProps) {
			Pattern pattern = Pattern.compile("(\\$\\{[^\\}]*\\})");
			Matcher matcher = pattern.matcher(prop.getValue());
			while(matcher.find()) {

				String key = matcher.group();
				String propName = key.replace("${", "").replace("}", "");

				String propValue = unifiedConf.getProperties().get(propName);
				if(propValue != null) {
					prop.setValue(prop.getValue().replace(key, propValue));
				}

			}
		}
	}


	public static Configuration unify(RIOSuiteAbstractBundle bundle) throws Exception {
		return ConfigurationUnifier.unify(bundle.getWebappServicesToActivate(), bundle.getSoapServicesToActivate());
	}


	public static Configuration unify(List<DWApplicationService> webappServices, List<Class<? extends SPIWebServicePrimitives>> wsServices) throws Exception {

		Map<Type, Map<String, Map<String, String>>> allConfMap = ConfigurationUnifier.unifyConf(webappServices, wsServices);

		Map<Type, SortedMap<Integer, Map<String, Map<String, String>>>> allSortedConfMap = priorizeByPortProperty(allConfMap);
		replaceValuesByPropertyKey(allSortedConfMap);

		StringBuffer buffer = new StringBuffer();

		buffer.append("###############################################" + "\n");
		buffer.append("##                                           ##" + "\n");
		buffer.append("##             Common Properties             ##" + "\n");
		buffer.append("##                                           ##" + "\n");
		buffer.append("###############################################" + "\n");
		buffer.append("\n"); 
		for(Entry<Integer, Map<String, Map<String, String>>> common_props_prio: allSortedConfMap.get(Type.COMMON).entrySet()) {
			for(Entry<String, Map<String, String>> common_props: common_props_prio.getValue().entrySet()) {
				List<PropertiesGroup> groups = sortPropertiesByGroup(common_props.getValue());

				for(PropertiesGroup group: groups) {
					if(group.getGroupName() != null && !group.getGroupName().isBlank()) {
						buffer.append("\n# " + group.getGroupName() + "\n"); 
					}
					for(GJaxbProperty prop: group.getProperties()) {
						buffer.append(prop.getName() + " = " + prop.getValue() + "\n"); 
					}
					buffer.append("\n"); 
				}
			}
		}
		buffer.append("\n\n\n"); 


		buffer.append("###############################################" + "\n");
		buffer.append("##                                           ##" + "\n");
		buffer.append("##              Web Applications             ##" + "\n");
		buffer.append("##                                           ##" + "\n");
		buffer.append("###############################################" + "\n");
		buffer.append("\n"); 
		for(Entry<Integer, Map<String, Map<String, String>>> webapp_props_prio: allSortedConfMap.get(Type.WEBAPP).entrySet()) {
			for(Entry<String, Map<String, String>> webapp_props: webapp_props_prio.getValue().entrySet()) {
				buffer.append("# " + webapp_props.getKey() + " properties :" + "\n");
				buffer.append(addDiese(("# " + webapp_props.getKey() + " properties :").length()));

				List<PropertiesGroup> groups = sortPropertiesByGroup(webapp_props.getValue());

				for(PropertiesGroup group: groups) {
					if(group.getGroupName() != null && !group.getGroupName().isBlank()) {
						buffer.append("\n# " + group.getGroupName() + "\n"); 
					}
					for(GJaxbProperty prop: group.getProperties()) {
						buffer.append(prop.getName() + " = " + prop.getValue() + "\n"); 
					}
					buffer.append("\n"); 
				}
			}
			buffer.append("\n\n"); 
		}
		buffer.append("\n\n\n"); 

		buffer.append("###############################################" + "\n");
		buffer.append("##                                           ##" + "\n");
		buffer.append("##                Web Services               ##" + "\n");
		buffer.append("##                                           ##" + "\n");
		buffer.append("###############################################" + "\n");
		buffer.append("\n"); 
		for(Entry<Integer, Map<String, Map<String, String>>> ws_props_prio: allSortedConfMap.get(Type.WS).entrySet()) {
			for(Entry<String, Map<String, String>> ws_props: ws_props_prio.getValue().entrySet()) {
				buffer.append("# " + ws_props.getKey() + " properties :" + "\n");
				buffer.append(addDiese(("# " + ws_props.getKey() + " properties :").length()));
				List<PropertiesGroup> groups = sortPropertiesByGroup(ws_props.getValue());

				for(PropertiesGroup group: groups) {
					if(group.getGroupName() != null && !group.getGroupName().isBlank()) {
						buffer.append("\n# " + group.getGroupName() + "\n"); 
					}
					for(GJaxbProperty prop: group.getProperties()) {
						buffer.append(prop.getName() + " = " + prop.getValue() + "\n"); 
					}
					buffer.append("\n"); 
				}
			}
			buffer.append("\n\n"); 
		}
		buffer.append("\n\n\n"); 


		LOG.debug("conf buffer:\n" + buffer);
		File unifiedConfFile = new File(Configuration.CONFIG_PATH, Configuration.CONFIG_NAME);
		unifiedConfFile.getParentFile().mkdirs();
		unifiedConfFile.createNewFile();
		FileUtil.setContents(unifiedConfFile, buffer.toString());

		Configuration unifiedConf = new Configuration(unifiedConfFile.toURI().toURL());

		return unifiedConf;
	}


	private static Map<Type, SortedMap<Integer, Map<String, Map<String, String>>>> priorizeByPortProperty(Map<Type, Map<String, Map<String, String>>> allConfMap) {
		Map<Type, SortedMap<Integer, Map<String, Map<String, String>>>> priorityMap = new HashMap<Type, SortedMap<Integer, Map<String, Map<String, String>>>>();

		List<GJaxbProperty> appsByPort = new ArrayList<GJaxbProperty>();
		for(Entry<Type, Map<String, Map<String, String>>> entry: allConfMap.entrySet()) {
			for(Entry<String, Map<String, String>> propsByApp: entry.getValue().entrySet()) {

				String appName = propsByApp.getKey();
				Optional<Entry<String, String>> portEntry = propsByApp.getValue().entrySet().stream().filter(e -> e.getKey().endsWith("...port")).findFirst();

				if(portEntry.isPresent()) {
					String port = portEntry.get().getValue();
					appsByPort.add(GenericModelHelper.createProperty(appName, port));
				} else {
					appsByPort.add(GenericModelHelper.createProperty(appName, "0"));
				}

			}
		}

		// sort list
		Collections.sort(appsByPort, new Comparator<GJaxbProperty>() {

			@Override
			public int compare(GJaxbProperty p1, GJaxbProperty p2) {
				Integer port1 = getInt(p1);
				Integer port2 = getInt(p2);

				return port1.compareTo(port2);
			}

			private int getInt(GJaxbProperty p) {
				try {
					return Integer.parseInt(p.getValue());
				} catch(NumberFormatException e) {
					return 0;
				}
			}
		});


		for(Entry<Type, Map<String, Map<String, String>>> entry: allConfMap.entrySet()) {
			Type type = entry.getKey();
			priorityMap.put(type, new TreeMap<Integer, Map<String, Map<String, String>>>());

			int priority = 0;
			for(GJaxbProperty appByPort: appsByPort) {
				Map<String, String> propsOfApp = entry.getValue().get(appByPort.getName());
				if(propsOfApp != null) {
					priorityMap.get(type).put(priority, new HashMap<String, Map<String, String>>());
					priorityMap.get(type).get(priority).put(appByPort.getName(), propsOfApp);
					priority = priority + 1;
				}
			}

		}

		return priorityMap;
	}


	private static void replaceValuesByPropertyKey(Map<Type, SortedMap<Integer, Map<String, Map<String, String>>>> allSortedConfMap) {
		// Map<Type: webapp or ws or common, Map<String:webapp name or common, Map<String: property name, String: property value>>> allConfMap
		for(Entry<Type, SortedMap<Integer, Map<String, Map<String, String>>>> entry1: allSortedConfMap.entrySet()) {
			for(Entry<Integer, Map<String, Map<String, String>>> entryPrio1: entry1.getValue().entrySet()) {
				for(Entry<String, Map<String, String>> propsByApp1: entryPrio1.getValue().entrySet()) {
					for(Entry<String, String> prop1: propsByApp1.getValue().entrySet()) {
						String propKey1 = prop1.getKey();
						String propValue1 = prop1.getValue();

						for(Entry<Type, SortedMap<Integer, Map<String, Map<String, String>>>> entry2: allSortedConfMap.entrySet()) {
							for(Entry<Integer, Map<String, Map<String, String>>> entryPrio2: entry2.getValue().entrySet()) {
								for(Entry<String, Map<String, String>> propsByApp2: entryPrio2.getValue().entrySet()) {
									for(Entry<String, String> prop2: propsByApp2.getValue().entrySet()) {
										String propKey2 = prop2.getKey();
										String propValue2 = prop2.getValue();

										if(!propKey1.equals(propKey2) && !propValue1.contains("$") && propValue2.contains(propValue1)
												&& !propValue2.equals("true") && !propValue2.equals("false")
												&& !propKey2.contains("-login") && !propKey2.contains("-pwd")
												&& !propKey2.contains("_login") && !propKey2.contains("_password")
												&& !propKey1.equals("publish-lot-size") && !propValue1.equals("gind")) {
											String newPropValue2 = propValue2.replace(propValue1, "${" + propKey1 + "}");
											propsByApp2.getValue().put(propKey2, newPropValue2);
										}

									}
								}
							}
						}

					}
				}
			}
		}
	}


	private static List<PropertiesGroup> sortPropertiesByGroup(Map<String, String> properties) {
		List<PropertiesGroup> groups = new ArrayList<PropertiesGroup>();

		PropertiesGroup defaultGroup = new PropertiesGroup();
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("host", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("protocol", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("generation-date", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("application", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("proxy-port", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("port", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("version", properties));
		defaultGroup.addPropertyIfExist(removeIfPropertiesContains("defaultAdmin", properties));
		if(defaultGroup.getProperties().size() > 0) {
			groups.add(defaultGroup);
		}

		PropertiesGroup neo4jGroup = new PropertiesGroup("Neo4j Database (see neo4j_clustering.conf or neo4j_embedded.conf for more informations)");
		neo4jGroup.addPropertyIfExist(removeIfPropertiesContains("neo4j-database-embedded", properties));
		neo4jGroup.addPropertyIfExist(removeIfPropertiesContains("neo4j-database-login", properties));
		neo4jGroup.addPropertyIfExist(removeIfPropertiesContains("neo4j-database-pwd", properties));
		if(neo4jGroup.getProperties().size() > 0) {
			groups.add(neo4jGroup);
		}

		PropertiesGroup mongoGroup = new PropertiesGroup("Mongo Database");
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-embedded", properties));
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-name", properties));
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-host", properties));
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-port", properties));
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-auth_enabled", properties));
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-login", properties));
		mongoGroup.addPropertyIfExist(removeIfPropertiesContains("mongodb-database-pwd", properties));
		if(mongoGroup.getProperties().size() > 0) {
			groups.add(mongoGroup);
		}

		PropertiesGroup h2GisGroup = new PropertiesGroup("H2Gis Database");
		h2GisGroup.addPropertyIfExist(removeIfPropertiesContains("h2gis-database-url", properties));
		h2GisGroup.addPropertyIfExist(removeIfPropertiesContains("h2gis-database-login", properties));
		h2GisGroup.addPropertyIfExist(removeIfPropertiesContains("h2gis-database-pwd", properties));
		h2GisGroup.addPropertyIfExist(removeIfPropertiesContains("h2gis-database-repository", properties));
		if(h2GisGroup.getProperties().size() > 0) {
			groups.add(h2GisGroup);
		}

		PropertiesGroup pythonGroup = new PropertiesGroup("Python");
		pythonGroup.addPropertyIfExist(removeIfPropertiesContains("python-activated", properties));
		pythonGroup.addPropertyIfExist(removeIfPropertiesContains("python-ml-engine-port", properties));
		if(pythonGroup.getProperties().size() > 0) {
			groups.add(pythonGroup);
		}

		PropertiesGroup portGroup = new PropertiesGroup();
		List<Entry<String, String>> portProps = properties.entrySet().stream().filter(e -> e.getKey().endsWith("-port")).collect(Collectors.toList());

		Collections.sort(portProps, new Comparator<Entry<String, String>>() {

			@Override
			public int compare(Entry<String, String> p1, Entry<String, String> p2) {
				Integer port1 = getInt(p1);
				Integer port2 = getInt(p2);
				return port1.compareTo(port2);
			}

			private int getInt(Entry<String, String> p1) {
				try {
					return Integer.parseInt(p1.getValue());
				} catch(NumberFormatException e) {
					return 0;
				}
			}
		});

		for(Entry<String, String> portP: portProps) {
			portGroup.addPropertyIfExist(removeIfPropertiesContains(portP.getKey(), properties));
		}
		if(portGroup.getProperties().size() > 0) {
			groups.add(portGroup);
		}


		PropertiesGroup publishGroup = new PropertiesGroup();
		List<Entry<String, String>> publishProps = properties.entrySet().stream().filter(e -> e.getKey().startsWith("publish-")).collect(Collectors.toList());
		for(Entry<String, String> publishP: publishProps) {
			publishGroup.addPropertyIfExist(removeIfPropertiesContains(publishP.getKey(), properties));
		}
		if(publishGroup.getProperties().size() > 0) {
			groups.add(publishGroup);
		}

		PropertiesGroup httpGroup = new PropertiesGroup();
		List<Entry<String, String>> httpProps = properties.entrySet().stream().filter(e -> e.getValue().startsWith("http://") || e.getValue().startsWith("https://")).collect(Collectors.toList());
		for(Entry<String, String> httpP: httpProps) {
			httpGroup.addPropertyIfExist(removeIfPropertiesContains(httpP.getKey(), properties));
		}
		if(httpGroup.getProperties().size() > 0) {
			groups.add(httpGroup);
		}


		// add remaining properties
		PropertiesGroup othersGroup = new PropertiesGroup();
		for(Entry<String, String> prop: properties.entrySet()) {
			othersGroup.addPropertyIfExist(GenericModelHelper.createProperty(prop.getKey(), prop.getValue()));
		}
		if(othersGroup.getProperties().size() > 0) {
			groups.add(othersGroup);
		}

		return groups;
	}


	private static GJaxbProperty removeIfPropertiesContains(String key, Map<String, String> properties) {
		GJaxbProperty prop = null;
		String value = properties.remove(key);
		if(value == null) {
			String complete_key = null; 
			for(Entry<String, String> p: properties.entrySet()) {
				if(p.getKey().endsWith("..." + key)) {
					complete_key = p.getKey();
					break;
				}
			}
			value = properties.remove(complete_key);
			prop = GenericModelHelper.createProperty(complete_key, value);
		} else {
			prop = GenericModelHelper.createProperty(key, value);
		}
		return prop;
	}


	private static String addDiese(int length) {
		StringBuffer buffer = new StringBuffer() ;
		for(int i = 0; i < length; i++) {
			buffer.append("#");
		}
		buffer.append("\n");
		return buffer.toString();
	}


	private static Map<Type, Map<String, Map<String, String>>> unifyConf(List<DWApplicationService> webappServices, List<Class<? extends SPIWebServicePrimitives>> wsServices) throws Exception {

		// Get all properties
		// Map<String:property key, Map<Type: type App, Map<String: webapp or ws name, String: property value>>> allPropsMap
		Map<String, Map<Type, Map<String, String>>> allPropsMap = new HashMap<String, Map<Type, Map<String, String>>>();
		for(DWApplicationService service: webappServices) {
			Configuration default_serviceConf = getConfigurationOfComponent(service.getName());

			for(Entry<String, String> prop: default_serviceConf.getProperties().entrySet()) {
				if(allPropsMap.get(prop.getKey()) == null) {
					allPropsMap.put(prop.getKey(), new HashMap<Type, Map<String, String>>());
				}
				if(allPropsMap.get(prop.getKey()).get(Type.WEBAPP) == null) {
					allPropsMap.get(prop.getKey()).put(Type.WEBAPP, new HashMap<String, String>());
				}

				allPropsMap.get(prop.getKey()).get(Type.WEBAPP).put(service.getName(), prop.getValue());
			}
		}

		for(Class<? extends SPIWebServicePrimitives> wsServiceClass: wsServices) {

			String confName = CaseFormat.LOWER_CAMEL.to(CaseFormat.LOWER_HYPHEN, wsServiceClass.getSimpleName());
			Configuration default_serviceConf = getConfigurationOfComponent(confName);


			for(Entry<String, String> prop: default_serviceConf.getProperties().entrySet()) {
				if(allPropsMap.get(prop.getKey()) == null) {
					allPropsMap.put(prop.getKey(), new HashMap<Type, Map<String, String>>());
				}
				if(allPropsMap.get(prop.getKey()).get(Type.WS) == null) {
					allPropsMap.get(prop.getKey()).put(Type.WS, new HashMap<String, String>());
				}

				allPropsMap.get(prop.getKey()).get(Type.WS).put(confName, prop.getValue());
			}

		}


		// Filter common properties
		// Map<Type: webapp or ws or common, Map<String:webapp name or common, Map<String: property name, String: property value>>> sortedPropsMap
		Map<Type, Map<String, Map<String, String>>> sortedPropsMap = new HashMap<Type, Map<String, Map<String, String>>>();
		sortedPropsMap.put(Type.COMMON, new HashMap<String, Map<String, String>>());
		sortedPropsMap.put(Type.WEBAPP, new HashMap<String, Map<String, String>>());
		sortedPropsMap.put(Type.WS, new HashMap<String, Map<String, String>>());
		for(Entry<String, Map<Type, Map<String, String>>> prop: allPropsMap.entrySet()) {

			if(numberOfValues(prop.getValue()) > 1) {
				// verify that the values are the same
				List<String> values = getDifferentValues(prop.getValue());

				if(values.size() == 1) {
					if(sortedPropsMap.get(Type.COMMON).get("common") == null) {
						sortedPropsMap.get(Type.COMMON).put("common", new HashMap<String, String>());
					}
					sortedPropsMap.get(Type.COMMON).get("common").put(prop.getKey(), values.get(0));
				} else {
					LOG.warn("Several values for the same property '" + prop.getKey() + "': " + values);

					for(Entry<Type, Map<String, String>> entry: prop.getValue().entrySet()) {
						Type t = entry.getKey();
						for(Entry<String, String> p : entry.getValue().entrySet()) {
							String serviceName = p.getKey();
							String value = p.getValue();
							if(sortedPropsMap.get(t).get(serviceName) == null) {
								sortedPropsMap.get(t).put(serviceName, new HashMap<String, String>());
							}
							sortedPropsMap.get(t).get(serviceName).put(serviceName + "..." + prop.getKey(), value);
						}
					}
				}

			} else {
				for(Entry<Type, Map<String, String>> entry: prop.getValue().entrySet()) {
					Type t = entry.getKey();
					for(Entry<String, String> p : entry.getValue().entrySet()) {
						String serviceName = p.getKey();
						String value = p.getValue();
						if(sortedPropsMap.get(t).get(serviceName) == null) {
							sortedPropsMap.get(t).put(serviceName, new HashMap<String, String>());
						}
						sortedPropsMap.get(t).get(serviceName).put(serviceName + "..." + prop.getKey(), value);
					}
				}
			}
		}


		return sortedPropsMap;
	}



	private static int numberOfValues(Map<Type, Map<String, String>> valuesByApps) {
		int cpt = 0;
		for(Entry<Type, Map<String, String>> valuesByApp: valuesByApps.entrySet()) {
			cpt = cpt + valuesByApp.getValue().size();
		}
		return cpt;
	}


	private static List<String> getDifferentValues(Map<Type, Map<String, String>> map) {
		List<String> differentValues = new ArrayList<String>();
		for(Entry<Type, Map<String, String>> entry1: map.entrySet()) { 
			for(Entry<String, String> entry2: entry1.getValue().entrySet()) {
				if(!differentValues.contains(entry2.getValue())) {
					differentValues.add(entry2.getValue());
				}
			}
		}
		return differentValues;
	}

	public static Configuration getConfigurationOfComponent(String componentName) throws Exception {
		Configuration serviceConf = null;
		File confFile = new File("../conf/generated/" + componentName + "/conf/config.properties");
		if(confFile.exists()) {
			serviceConf = new Configuration(confFile.toURI().toURL());
			LOG.debug(componentName + ": Configuration file exist");
		} else if(Thread.currentThread().getContextClassLoader().getResource("test/conf/generated/" + componentName + "/conf/config.properties") != null){
			serviceConf = new Configuration(Thread.currentThread().getContextClassLoader().getResource("test/conf/generated/" + componentName + "/conf/config.properties"));
			LOG.debug(componentName + ": Configuration found in class loader");
		} else if(Thread.currentThread().getContextClassLoader().getResource("conf/generated/" + componentName + "/conf/config.properties") != null){
			serviceConf = new Configuration(Thread.currentThread().getContextClassLoader().getResource("conf/generated/" + componentName + "/conf/config.properties"));
			LOG.debug(componentName + ": Configuration found in class loader");
		} 

		if(serviceConf == null) {
			throw new Exception("Impossible to find configuration file for: " + componentName);
		}

		return serviceConf;
	}

}
