/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.suite;

/*
 * #%L
 * storage-mongodb-impl
 * %%
 * Copyright (C) 2014 - 2017 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import javax.xml.namespace.QName;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import fr.emac.gind.commons.utils.RIOConstant;
import fr.emac.gind.commons.utils.io.AbstractFileLoader;
import fr.emac.gind.commons.utils.io.UploadUtil;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.generic.application.utils.LoadedUsecase;
import fr.emac.gind.gov.ai.chatbot.client.AIChatbotClient;
import fr.emac.gind.gov.ai_chatbot.AiChatbot;
import fr.emac.gind.gov.core.client.CoreGovClient;
import fr.emac.gind.gov.core.client.SystemGovClient;
import fr.emac.gind.gov.core_gov.CoreGov;
import fr.emac.gind.gov.models.client.ModelsGovClient;
import fr.emac.gind.gov.models_gov.ModelsGov;
import fr.emac.gind.gov.system_gov.GJaxbConnectUserToCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbUpdateUser;
import fr.emac.gind.gov.system_gov.GJaxbUpdateUserResponse;
import fr.emac.gind.gov.system_gov.SystemGov;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveConceptType;
import fr.emac.gind.modeler.metamodel.GJaxbMetaModel;
import fr.emac.gind.modeler.metamodel.GJaxbRelation;
import fr.emac.gind.models.generic.modeler.effective_meta_model.EffectiveMetaModelHelper;
import fr.emac.gind.models.generic.modeler.effective_meta_model.EffectiveMetaModelManager;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.rio.dw.resources.FileResource;
import fr.emac.gind.rio.dw.resources.gov.ProjectResource;
import fr.emac.gind.rio.dw.resources.gov.SystemResource;

/**
 *
 *
 * @author Nicolas Salatge
 */
public class PublishUserLoader extends AbstractFileLoader {

	public static Logger LOG = LoggerFactory.getLogger(PublishUserLoader.class);

	private Configuration conf = null;

	private SystemGov systemClient = null;
	private ModelsGov modelsClient = null;
	private CoreGov coreClient = null;
	private AiChatbot aiChatBotClient = null;

	private static final QName NAME = new QName("http://fr.emac.gind/system", "User");

	private GJaxbEffectiveConceptType collaborationMetaModel = null;
	private GJaxbRelation connectedToMetaModel = null;

	public PublishUserLoader() throws Exception {
		GJaxbMetaModel sysMM = XMLJAXBContext.getInstance().unmarshallDocument(Thread.currentThread().getContextClassLoader().getResource("model/system/System_MetaModel.xml"), GJaxbMetaModel.class);

		EffectiveMetaModelManager manager = new EffectiveMetaModelManager(EffectiveMetaModelHelper.generateEffectiveMetaModelFromMetaModel(sysMM));
		collaborationMetaModel = manager.getConceptByType(new QName("http://fr.emac.gind/system", "Collaboration"));
		connectedToMetaModel = manager.getRelationByType(new QName("http://fr.emac.gind/system", "Connected_To"));
	}

	@Override
	public void setConfiguration(Object conf) throws Exception {
		this.conf = (Configuration) conf;
		this.systemClient = SystemGovClient.createClient(((Configuration)conf).getProperties().get("governance").replace("/gov", "/GovSystem"));
		this.modelsClient = ModelsGovClient.createClient(((Configuration)conf).getProperties().get("governance").replace("/gov", "/GovModels"));
		this.coreClient = CoreGovClient.createClient(((Configuration)conf).getProperties().get("governance").replace("/gov", "/GovCore"));
		this.aiChatBotClient = AIChatbotClient
				.createClient(((Configuration)conf).getProperties().get("governance").replace("/gov", "/GovAIChatbot"));

	}

	@Override
	public QName getQName() {
		return NAME;
	}

	@Override
	public synchronized void load(File userDirectory) throws Exception {
		if(userDirectory.isFile()) {
			userDirectory = userDirectory.getParentFile();
		}
		LOG.info("load on USER LOADER => " + userDirectory);
		Optional<File> opFile = FileUtils.listFiles(userDirectory, new String[] { "xml" }, true).stream().findFirst();
		int retryCpt = 0;
		while(!opFile.isPresent() && retryCpt < 10) {
			LOG.warn("user file not present !!! Retry...");
			opFile = FileUtils.listFiles(userDirectory, new String[] { "xml" }, true).stream().findFirst();
			Thread.sleep(300);
			retryCpt++;
		}
		if(opFile.isPresent()) {
			File userfile = opFile.get();
			LOG.warn("user file detected: " + userfile);
			Optional<File> picturefile = FileUtils.listFiles(userDirectory, UploadUtil.AUTHORIZED_IMAGE_EXTENSIONS.toArray(new String[UploadUtil.AUTHORIZED_IMAGE_EXTENSIONS.size()]), true).stream().findFirst();

			// publish user in governance
			Document userDoc = DOMUtil.getInstance().parse(userfile);
			GJaxbNode userNode = XMLJAXBContext.getInstance().unmarshallDocument(userDoc, GJaxbNode.class);

			GJaxbUpdateUser req = new GJaxbUpdateUser();
			req.setNode(userNode);
			req.setCreateIfNotExist(true);
			GJaxbUpdateUserResponse resp = this.systemClient.updateUser(req);
			userNode.setId(resp.getId());
			

			if(ProjectResource.REALWORLD_UC_NODE != null && ProjectResource.WORLD_COLLABORATION_NODE != null) {
				// publish associated person/cell phone/wath of user if not exist in real world
				SystemResource.publishPersonCellPhoneWatchToAssociateUserInRealWorld(userNode, this.coreClient, this.modelsClient, this.systemClient, this.aiChatBotClient);
			}


			// move picture
			File rootDirectory = new File(RIOConstant.RESOURCES_FOLDER);
			rootDirectory = new File(rootDirectory, "users");
			rootDirectory = new File(rootDirectory, RegExpHelper.toRegexFriendlyName(userNode.getId()));
			rootDirectory = new File(rootDirectory, "pictures");
			if(picturefile.isPresent()) {
				FileUtils.copyFileToDirectory(picturefile.get(), rootDirectory, true);
			}
			
			// attach user in default collaboration (predefined-usecase in conf)
			if(!ProjectResource.USECASES_ALREADY_LOADED.isEmpty()) {
				LoadedUsecase uc = ProjectResource.USECASES_ALREADY_LOADED.values().iterator().next();
				GJaxbNode colladNode = uc.getCollaboration();
				GJaxbNode kownledgeNode = uc.getKnowledgeSpace();
				
				GenericModelHelper.findProperty("Presence", userNode.getProperty(), true).setValue("Available");
				
				GenericModelHelper.findProperty("currentCollaborationName", userNode.getProperty(), true).setValue(GenericModelHelper.getName(colladNode));
				GenericModelHelper.findProperty("currentKnowledgeSpaceName", userNode.getProperty(), true).setValue(GenericModelHelper.getName(kownledgeNode));

				GJaxbUpdateUser upReq = new GJaxbUpdateUser();
				upReq.setNode(userNode);
				upReq.setCreateIfNotExist(true);
				this.systemClient.updateUser(upReq);
				
				GJaxbConnectUserToCollaboration reqConnColl = new GJaxbConnectUserToCollaboration();
				reqConnColl.setCollaborationId(colladNode.getId());
				reqConnColl.setUserId(userNode.getId());
				this.systemClient.connectUserToCollaboration(reqConnColl);
			}

			
			
			LOG.info("User added: " + GenericModelHelper.getName(userNode));
		} else {
			LOG.warn("File in user directory not found: " + userDirectory);
		}
	}

	@Override
	public List<File> listenedDirectories() {
		return List.of(new File("./target/listener"));
	}

}
