/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: josepe
 */
package fr.emac.gind.usecases.cep.helper;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.net.URL;
import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import javax.xml.namespace.QName;

import org.junit.jupiter.api.Assertions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import fr.emac.gind.commons.utils.io.FileUtil;
import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.commons.utils.xml.XMLPrettyPrinter;
import fr.emac.gind.event.consumer.AbstractNotifierClient;
import fr.emac.gind.event.consumer.NotificationConsumerWebService;
import fr.emac.gind.event.interpretation.config.InterpretationConfigDeployHelper;
import fr.emac.gind.event.interpretation.config.InterpretationConfigsManagerCommandClient;
import fr.emac.gind.event.producer.NotificationManagerImpl;
import fr.emac.gind.event.producer.NotificationProducerWebService;
import fr.emac.gind.governance.GovernanceWebService;
import fr.emac.gind.interpretationconfigs.InterpretationCommand;
import fr.emac.gind.io.interpretation.engine.InterpretationEngineContainer;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.model.interpretation.config.GJaxbDeploy;
import fr.emac.gind.model.interpretation.config.GJaxbDeployResponse;
import fr.emac.gind.model.interpretation.config.GJaxbGetInterpretationConfigs;
import fr.emac.gind.model.interpretation.config.GJaxbGetInterpretationConfigsResponse;
import fr.emac.gind.model.interpretation.config.GJaxbInterpretationConfig;
import fr.emac.gind.model.interpretation.config.GJaxbInterpretationConfigs;
import fr.emac.gind.storage.mongodb.EventStorageWebService;
import fr.emac.gind.storage.mongodb.StorageImpl;
import gind.org.oasis_open.docs.wsn.b_2.GJaxbNotify;
import gind.org.oasis_open.docs.wsn.t_1.GJaxbTopicSetType;
import jakarta.xml.bind.JAXBElement;

public class CEPClientRunner {
	
	private static final Logger LOG = LoggerFactory.getLogger(CEPClientRunner.class.getName());


  
  private final GovernanceWebService gov;
  private final InterpretationEngineContainer ieEngine;
  private final EventStorageWebService wsStorage;
  
  public CEPClientRunner(GovernanceWebService gov, InterpretationEngineContainer ieEngine, EventStorageWebService wsStorage) {
    this.gov = gov;
    this.ieEngine = ieEngine;
    this.wsStorage = wsStorage;
    
  }
  
  
  public static Entry<QName, List<Document>> createEntry(QName topic, String... msg) {
    List<Document> docs = Arrays.asList(msg).stream().map(m -> { try {
      return DOMUtil.getInstance().parse(new ByteArrayInputStream(m.getBytes()));
    } catch (Exception e) {
      e.printStackTrace();
      throw new RuntimeException(e);
    } }).collect(Collectors.toList());
    Entry<QName, List<Document>> e = new SimpleEntry<QName, List<Document>>(topic, docs);
    return e;
  }
  
  
  public List<GJaxbNotify> runComplexTest(
      URL ruleFile,
      int assertCpt,
      Entry<QName, List<Document>>... inputEventsPerTopic
      ) throws Exception{
    NotificationProducerWebService producer = new NotificationProducerWebService();
    NotificationConsumerWebService consumer = new NotificationConsumerWebService();

    try {

      GJaxbTopicSetType topicSet = new GJaxbTopicSetType();
      JAXBElement<GJaxbTopicSetType> ts = new gind.org.oasis_open.docs.wsn.t_1.ObjectFactory().createTopicSet(topicSet);

      for(Entry<QName, List<Document>> e : inputEventsPerTopic) {
        Document doc = DOMUtil.getInstance().newDocument();
        Element newChild = doc.createElementNS(e.getKey().getNamespaceURI(), e.getKey().getLocalPart());
        newChild.setAttributeNS("http://docs.oasis-open.org/wsn/t-1", "wstop:topic", "true");
        doc.adoptNode(newChild);
        doc.appendChild(newChild);
        ts.getValue().getAny().add(doc.getDocumentElement());      
      }

      File topicSetFile = File.createTempFile("temp_", java.util.UUID.randomUUID().toString(), new File("target"));
      LOG.debug(XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(ts)));
      FileUtil.setContents(topicSetFile, XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(ts)));

      InterpretationCommand client = InterpretationConfigsManagerCommandClient.createClient(ieEngine.getAddress(InterpretationConfigsManagerCommandClient.class));// "http://localhost:8190/cepRulesManager");
     
      GJaxbInterpretationConfig rule = XMLJAXBContext.getInstance().unmarshallDocument(ruleFile, GJaxbInterpretationConfig.class);
      GJaxbInterpretationConfigs configs = new GJaxbInterpretationConfigs();
      configs.getInterpretationConfig().add(rule);
      GJaxbDeploy deployRequest = InterpretationConfigDeployHelper.buildRequest(configs, "a", "b");

      String pa = rule.getContext().getSubscriptionsRequired().getEntry().get(0).getEndpointAddressToSubscribe();
      URL url = new URL(pa);
      String producerAddress = url.getPath().substring(1);
      producer.start(new HashMap<String, Object>(){{
        put("host","localhost"); 
        put("port",8391); 
        put("serviceName", producerAddress); 
        put("storage", wsStorage.getAddress(StorageImpl.class));// "http://localhost:10100/StorageService"); 
        put("topicSet", topicSetFile.toURI().toURL().toString());}});

      GJaxbDeployResponse response = client.deploy(deployRequest);

      Assertions.assertNotNull(response);

      GJaxbGetInterpretationConfigs list =  new GJaxbGetInterpretationConfigs();
      list.setCollaborationName("a");
      list.setKnowledgeSpaceName("b");
      GJaxbGetInterpretationConfigsResponse res = client.getInterpretationConfigs(list);

      Assertions.assertEquals(1, res.getInterpretationConfig().size());



      // subscribe on
      final List<GJaxbNotify> notifyList = Collections.synchronizedList(new ArrayList<GJaxbNotify>());
      final AbstractNotifierClient notifier = new AbstractNotifierClient("http://localhost:9501/NotifierClient") {
        public void notify(GJaxbNotify notify) {
          try {
            LOG.debug("rule output notification received: ");
            String n = XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(notify));
            LOG.debug(n);
          } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            Assertions.fail();
          }
          notifyList.add(notify);
        }
      };

      //WARNING WE CONSIDER ONE AND ONLY ONE EventAction
      QName outputTopic = rule.getEventAction().get(0).getTopic();
      LOG.debug("output topic "+outputTopic);

      consumer.start(new HashMap<String, Object>(){{put("host","localhost"); put("port",9501); put("serviceName", "NotifierClient"); put("notifierClient", notifier); }});
    
      
      String subscriptionId = consumer.getImplementation(notifier.getClass()).subscribeOn(ieEngine.getAddress(NotificationManagerImpl.class), outputTopic);// "http://localhost:8190/cepRulesSubscriber", outputTopic);//new QName("http://fr.emac.gind/c2net/PO","POTopic"));
      Assertions.assertNotNull(subscriptionId);


      // send primitive events

      Arrays.asList(inputEventsPerTopic).forEach((entry)->{

        entry.getValue().stream().forEach((d)->{
          try {
            producer.getImplementation(NotificationManagerImpl.class).getNotifier().sendNotificationOnTopic(d, entry.getKey(), null, false);
          } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            Assertions.fail();
          }

        });

      });
      while(notifyList.size() < assertCpt) {
        Thread.sleep(1000);
        System.out.print("-");
      }

      return notifyList;

    } finally {
      if(consumer != null) {
        consumer.stop();
      }
      if(producer != null) {
        producer.stop();
      }

    }
  }

  
}
