/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URI;
import java.net.URL;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.jetty.servlets.CrossOriginFilter;
import org.eclipse.jetty.websocket.server.config.JettyWebSocketServletContainerInitializer;
//import org.atmosphere.cpr.AtmosphereServlet;
import org.glassfish.jersey.media.multipart.MultiPartFeature;
import org.glassfish.jersey.server.filter.RolesAllowedDynamicFeature;
import org.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import com.fasterxml.jackson.databind.ObjectMapper;

/*
 * #%L
 * dw-generic-application
 * %%
 * Copyright (C) 2014 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */



import fr.emac.gind.application.model.GJaxbApplication;
import fr.emac.gind.commons.utils.RIOConstant;
import fr.emac.gind.commons.utils.io.FileUtil;
import fr.emac.gind.commons.utils.lang.UncheckedException;
import fr.emac.gind.commons.utils.net.IPUtil;
import fr.emac.gind.commons.utils.net.IPUtil.TRANSPORT_PROTOCOL;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.uri.URIHelper;
import fr.emac.gind.commons.utils.ws.SPIWebServicePrimitives;
import fr.emac.gind.commons.utils.ws.StaticJettyServer;
import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.commons.utils.xml.XMLPrettyPrinter;
import fr.emac.gind.commons.utils.yaml.YMLConfigGenerator;
import fr.emac.gind.database.mongodb.embedded.EmbeddedMongoDb;
import fr.emac.gind.event.consumer.AbstractNotifierClient;
import fr.emac.gind.event.consumer.NotificationConsumerWebService;
import fr.emac.gind.generic.application.auth.BasicAuthenticator;
import fr.emac.gind.generic.application.auth.BasicCredentialAuthFilter;
import fr.emac.gind.generic.application.auth.UserRoleAuthorizer;
import fr.emac.gind.generic.application.bundles.AssetsInterceptorBundle;
import fr.emac.gind.generic.application.bundles.modifier.ApplicationNameModifierResource;
import fr.emac.gind.generic.application.bundles.modifier.ExternalModifierResource;
import fr.emac.gind.generic.application.configuration.GenericConfiguration;
import fr.emac.gind.generic.application.soap.WebsocketCommandImpl;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.generic.application.utils.RestResourceManager;
import fr.emac.gind.generic.application.websocket.chat.ChatServerServlet;
import fr.emac.gind.generic.application.websocket.pubsub.PubSubServerServlet;
import fr.emac.gind.generic.application.websocket.sockjs.SockJSServerServlet;
import fr.emac.gind.gov.ai.chatbot.service.builder.model.ollama.OllamaContainerConfig;
import fr.emac.gind.gov.ai.chatbot.service.builder.model.ollama.OllamaManager;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.modeler.metamodel.GJaxbEdgeMetaModeling;
import fr.emac.gind.modeler.metamodel.GJaxbModelingEdgeViewDefinition;
import fr.emac.gind.modeler.metamodel.GJaxbRelation;
import fr.emac.gind.models.generic.modeler.utils.svg.SVGGenerator;
import fr.emac.gind.rio.dw.resources.DWApplicationContextResource;
import fr.emac.gind.rio.dw.resources.FileResource;
import fr.emac.gind.rio.dw.resources.HumanTaskResource;
import fr.emac.gind.rio.dw.resources.StorageResource;
import fr.emac.gind.rio.dw.resources.gov.AIChatbotResource;
import fr.emac.gind.rio.dw.resources.gov.CoreResource;
import fr.emac.gind.rio.dw.resources.gov.MetaModelsResource;
import fr.emac.gind.rio.dw.resources.gov.ModelsResource;
import fr.emac.gind.rio.dw.resources.gov.Neo4JResource;
import fr.emac.gind.rio.dw.resources.gov.SystemResource;
import fr.emac.gind.websocket.command.WebsocketCommand;
import io.dropwizard.auth.AuthDynamicFeature;
import io.dropwizard.auth.AuthValueFactoryProvider;
import io.dropwizard.configuration.EnvironmentVariableSubstitutor;
import io.dropwizard.configuration.SubstitutingSourceProvider;
import io.dropwizard.core.Application;
import io.dropwizard.core.Configuration;
import io.dropwizard.core.setup.Bootstrap;
import io.dropwizard.core.setup.Environment;
import io.dropwizard.jersey.jackson.JsonProcessingExceptionMapper;
import io.swagger.v3.jaxrs2.integration.resources.OpenApiResource;
import io.swagger.v3.oas.integration.SwaggerConfiguration;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.info.Info;
import jakarta.servlet.DispatcherType;
import jakarta.servlet.FilterRegistration;
import jakarta.servlet.ServletRegistration;
import jakarta.xml.ws.Endpoint;

/**
 *
 *
 * @author Nicolas Salatge
 */
public abstract class DWApplicationService extends Application<GenericConfiguration> {

	private static Logger LOG = LoggerFactory.getLogger(DWApplicationService.class.getName());


	protected GJaxbApplication application = null;
	protected ApplicationContext applicationContext = null;

	protected fr.emac.gind.launcher.Configuration conf = null;

	protected URL ymlConfig = null;


	protected Map<String, Object> context = new HashMap<String, Object>();


	private List<Class<?>> webSocketResources = new ArrayList<Class<?>>();


	private ServletRegistration.Dynamic pubSubWebsocket = null;

	private static EmbeddedMongoDb embeddedMongoDB = null;
	private static boolean SHARED_RESOURCES_LOADED = false;

	private Environment environment = null;
	private RestResourceManager resourcesManager = null;

	protected AssetsInterceptorBundle interceptorBundle = null;

	protected static WebsocketCommand WEB_SOCKET_COMMAND = null;
	protected static Endpoint WEB_SOCKET_COMMAND_SERVER = null;


	protected SystemResource systemResource = null;
	protected HumanTaskResource humanTaskResource = null;
	protected CoreResource coreResource = null;
	protected MetaModelsResource metaModelsResource = null;
	protected ModelsResource modelsResource = null;
	
	protected AIChatbotResource aiChatbotResource = null;
	
	protected FileResource fileResource = null;
	protected DWApplicationContextResource contextResource = null;

	private String queryPath = null;

	private String applicationUrl = null;

	protected static NotificationConsumerWebService HUMAN_TASK_CONSUMER = null;

	public DWApplicationService() throws Exception {
		this(new HashMap<String, Object> ());
	}

	public DWApplicationService(Map<String, Object> context) throws Exception {
		this.context = context;
	}



	public WebsocketCommand getWebsocketCommand() {
		return WEB_SOCKET_COMMAND;
	}

	@Override
	protected void bootstrapLogging() {
	}

	public String getName() {
		return "generic-application";
	}

	public String getShortPath() {
		return "/webjars/" + this.getName() + "/generated/index.html";
	}


	public String getRedirection() {
		return "";
	}

	public String getQueryPath() {
		return queryPath;
	}

	public void setQueryPath(String queryPath) {
		this.queryPath = queryPath;
	}

	public void boot(fr.emac.gind.launcher.Configuration conf) throws Exception {
		try {
			this.conf = conf;

			this.ymlConfig = YMLConfigGenerator.generate(conf, this.getName());
			LOG.info("ymlConfig: " + this.ymlConfig);
			LOG.debug("ymlConfig exist: " + new File(this.ymlConfig.toURI()).exists());
			
			// Embedded mongoDB
			if(this.conf.getProperties().get("mongodb-database-embedded") != null && Boolean.valueOf(this.conf.getProperties().get("mongodb-database-embedded").toString())) {
				LOG.info("Start embedded mongoDB");
				startEmbeddedMongDB(false, conf);
			}
			
			// Ollama manager
			if(this.conf.getProperties().get("ollama") != null && !this.conf.getProperties().get("ollama").isBlank()) {
				List<OllamaContainerConfig> containerConfigs = OllamaContainerConfig.parse(new JSONArray(this.conf.getProperties().get("ollama")));
				OllamaManager.getInstance().addContainers(containerConfigs);
				
				OllamaManager.getInstance().startEmbeddedContainer();
			}

			Integer proxify =  null;
			if(context.get("proxy-port") != null) {
				proxify = Integer.parseInt((String)context.get("proxy-port").toString());
			}
			TRANSPORT_PROTOCOL protocol =  TRANSPORT_PROTOCOL.HTTP;
			if(context.get("protocol") != null) {
				protocol = TRANSPORT_PROTOCOL.fromString((String)context.get("protocol").toString());
			}

			this.applicationUrl = new URI(IPUtil.createPrettyHost(this.conf.getHost(), this.conf.getPort(), protocol, proxify) + "/" + this.getName()  + this.getShortPath()).toString();
			if(this.queryPath != null) {
				this.applicationUrl = this.applicationUrl + "?" + queryPath;
			}

			this.application = generateJaxbApplication(this.getName(), conf, this.context);

			if(this.application != null) {
				this.applicationContext = this.createApplicationContext();
				this.initializeApplicationContext(applicationContext);
			}

		} catch (Exception e) {
			throw new UncheckedException(e);
		}
	}

	public ApplicationContext getApplicationContext() {
		return applicationContext;
	}

	public CoreResource getCoreResource() {
		return coreResource;
	}

	public ModelsResource getModelsResource() {
		return modelsResource;
	}

	public SystemResource getUserResource() {
		return systemResource;
	}

	public DWApplicationContextResource getContextResource() {
		return contextResource;
	}
	
	public static GJaxbApplication generateJaxbApplication(String name, fr.emac.gind.launcher.Configuration conf, Map<String, Object> context)
			throws Exception {
		URL applicationModel = null;
		if(conf.getUrl().toString().startsWith("jar:")) {
			applicationModel = URIHelper.resolve(conf.getUrl().toURI(), (String)conf.getProperties().get("application")).toURL();
		} else if(conf.getProperties().get("application") != null) {
			applicationModel = new File(new File(conf.getUrl().toURI()).getParentFile().getCanonicalFile(), conf.getProperties().get("application")).getCanonicalFile().toURI().toURL();
		}

		GJaxbApplication application = null;
		if(applicationModel != null) {

			if(applicationModel.getProtocol().equals("file")) {
				File appliFile = new File(applicationModel.toURI());

				if(!appliFile.exists()) {
					// try to found test file
					applicationModel = Thread.currentThread().getContextClassLoader().getResource("test/conf/generated/" + name + "/conf/Application.xml");
				}

				System.out.println("applicationModel: " + applicationModel);
				String bufferApplication = FileUtil.getContents(applicationModel.openStream());
				bufferApplication = bufferApplication.replace("##application_name##", name);
				Document doc = DOMUtil.getInstance().parse(new ByteArrayInputStream(bufferApplication.getBytes()));

				application = XMLJAXBContext.getInstance().unmarshallDocument(doc, GJaxbApplication.class);
				if(application.getExtensions() == null) {
					application.setExtensions(new GJaxbApplication.Extensions());
				}
				context.put("application", application);

			}

		}
		return application;
	}

	public URL getYmlConfig() {
		return ymlConfig;

	}

	@Override
	public void initialize(Bootstrap<GenericConfiguration> bootstrap) {

		ObjectMapper mapper = bootstrap.getObjectMapper();
		mapper.registerModule(new GJaxbSerialierModule());

		// Enable variable substitution with environment variables
		bootstrap.setConfigurationSourceProvider(
				new SubstitutingSourceProvider(
						bootstrap.getConfigurationSourceProvider(),
						new EnvironmentVariableSubstitutor(false)
						)
				);

		// bootstrap.addBundle(new AssetsBundle());

		if(this.getName().equals("generic-application")) {
			interceptorBundle = new AssetsInterceptorBundle("/META-INF/resources/webjars", "/" + "generic-application" + "/webjars", new ExternalModifierResource(this.getName(), this.conf), new ApplicationNameModifierResource(this.getName()));
			bootstrap.addBundle(interceptorBundle);
		}
		else { 
			interceptorBundle = new AssetsInterceptorBundle("/META-INF/resources/webjars", "/" + this.getName() + "/webjars", new ExternalModifierResource(this.getName(), this.conf), new ApplicationNameModifierResource(this.getName()));
			bootstrap.addBundle(interceptorBundle);
		}

	}

	public GJaxbApplication getApplication() {
		return application;
	}

	public fr.emac.gind.launcher.Configuration getConfiguration() {
		return conf;
	}

	public void setConfiguration(fr.emac.gind.launcher.Configuration conf) {
		this.conf = conf;
	}

	public <T extends ApplicationContext> T createApplicationContext() throws Exception {
		ApplicationContext applicationContext = new ApplicationContext(this);
		return (T) applicationContext;
	}

	public <T extends ApplicationContext> void initializeApplicationContext(T applicationContext) throws Exception {
		applicationContext.setApplication(application);
		applicationContext.getApplication().setBuildDate((String) conf.getProperties().get("generation-date"));
	}

	@Override
	public void run(GenericConfiguration configuration, Environment env) throws Exception {
		this.environment = env;

		OpenAPI oas = new OpenAPI();
		Info info = new Info()
				.title("R-IOSuite API")
				.description("RESTful R-IOSuite Framework")
				.termsOfService("https://r-iosuite.com/")
				.contact(new Contact().email("gind-riosuite@mines-albi.fr"));

		oas.info(info);
		SwaggerConfiguration oasConfig = new SwaggerConfiguration()
				.openAPI(oas)
				.prettyPrint(true)
				.resourcePackages(Stream.of("fr.emac.gind.rio.dw.resources")
						.collect(Collectors.toSet()));
		environment.jersey().register(new OpenApiResource()
				.openApiConfiguration(oasConfig));
		
		System.out.println("rsc config: " + this.environment.jersey().getResourceConfig());
		
		environment.jersey().register(new RIOExceptionMapper());
		
		if(!SHARED_RESOURCES_LOADED) {
			copySharedResources();
			SHARED_RESOURCES_LOADED = true;
		}

		if(WEB_SOCKET_COMMAND == null && this.conf.getProperties().get("websocket-command-service-port") != null) {
			WEB_SOCKET_COMMAND = new WebsocketCommandImpl(conf.getPort());
			WEB_SOCKET_COMMAND_SERVER = StaticJettyServer.getInstance().publishJAXWSEndpoint(SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP, this.conf.getHost(), Integer.parseInt(this.conf.getProperties().get("websocket-command-service-port")), Integer.parseInt(conf.getProperties().get("proxy-port")), "/websocketCommandService"), WEB_SOCKET_COMMAND);
			LOG.info("WebsocketCommandService started at: " + SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP, this.conf.getHost(), Integer.parseInt(this.conf.getProperties().get("websocket-command-service-port")), Integer.parseInt(conf.getProperties().get("proxy-port")), "/websocketCommandService"));
		}

		this.resourcesManager = new RestResourceManager(env);
		if(this.application != null) {
			this.fileResource = new FileResource(this.getName());
			this.resourcesManager.addResource(this.fileResource);

			this.resourcesManager.addResource(new StorageResource(conf));

			this.coreResource = new CoreResource(conf);
			this.metaModelsResource = new MetaModelsResource(conf, WEB_SOCKET_COMMAND);
			this.resourcesManager.addResource(new Neo4JResource(conf));
			this.resourcesManager.addResource(coreResource);

			this.modelsResource = new ModelsResource(conf, WEB_SOCKET_COMMAND, this.metaModelsResource);
			this.resourcesManager.addResource(this.modelsResource);
			this.resourcesManager.addResource(this.metaModelsResource);

			this.contextResource = new DWApplicationContextResource(this, applicationContext, conf, coreResource);
			this.resourcesManager.addResource(contextResource);

			this.systemResource = new SystemResource(applicationContext, conf);
			this.resourcesManager.addResource(this.systemResource);

			this.humanTaskResource = new HumanTaskResource(conf, this.modelsResource);
			this.resourcesManager.addResource(this.humanTaskResource);
			
			
			this.aiChatbotResource = new AIChatbotResource(conf, this.fileResource);
			this.resourcesManager.addResource(this.aiChatbotResource);
			
		}


		this.environment.jersey().register(new JsonProcessingExceptionMapper(true));
		this.environment.jersey().register(MultiPartFeature.class); 
		this.environment.healthChecks().register("dummy", new DummyHealthCheck());

		this.environment.jersey().register(RolesAllowedDynamicFeature.class);
		this.environment.jersey().register(new AuthDynamicFeature(
				new BasicCredentialAuthFilter.Builder<DWUser>()
				.setAuthenticator(new BasicAuthenticator(systemResource))
				.setAuthorizer(new UserRoleAuthorizer())
				.setRealm("SUPER SECRET STUFF")
				.buildAuthFilter()));

		//If you want to use @Auth to inject a custom Principal type into your resource
		this.environment.jersey().register(new AuthValueFactoryProvider.Binder<DWUser>(DWUser.class));

		environment.getApplicationContext().addServletContainerInitializer(new JettyWebSocketServletContainerInitializer());
		final ServletRegistration.Dynamic websocket = env.servlets().addServlet("ws_chat", new ChatServerServlet());
		websocket.setAsyncSupported(true);
		websocket.addMapping("/ws/chat/*");


		this.doRun(configuration, env, this.resourcesManager);

		this.resourcesManager.registerAllResources();


		if(conf.getProperties().get("humantask-event-notifier-port") != null) {  
			createNotifierForHumanTaskEvent();
		}

		// Enable CORS headers
		final FilterRegistration.Dynamic cors =
				environment.servlets().addFilter("CORS", CrossOriginFilter.class);

		// Configure CORS parameters
		cors.setInitParameter(CrossOriginFilter.ALLOWED_ORIGINS_PARAM, "*");
		cors.setInitParameter(CrossOriginFilter.ACCESS_CONTROL_ALLOW_ORIGIN_HEADER, "*");
		// cors.setInitParameter(CrossOriginFilter.ALLOWED_HEADERS_PARAM, "X-Requested-With,Content-Type,Accept,Origin,Authorization");
		cors.setInitParameter(CrossOriginFilter.ALLOWED_METHODS_PARAM, "OPTIONS,GET,PUT,POST,DELETE,HEAD");
		cors.setInitParameter(CrossOriginFilter.ALLOW_CREDENTIALS_PARAM, "true");
		cors.setInitParameter(CrossOriginFilter.ALLOWED_HEADERS_PARAM,
				"Cache-Control,If-Modified-Since,Pragma,Content-Type,Authorization,X-Requested-With,Content-Length,Accept,Origin");

		// Add URL mapping
		cors.addMappingForUrlPatterns(EnumSet.allOf(DispatcherType.class), true, "/*");

		LOG.info("\n\nApplication started at : \n\t" + this.applicationUrl + "\n\n");

	}


	public abstract void doRun(Configuration configuration, Environment env, RestResourceManager resourcesManager) throws Exception;



	public static void startEmbeddedMongDB(boolean testMode, fr.emac.gind.launcher.Configuration conf) throws Exception {
		if(embeddedMongoDB == null) {
			LOG.info("Start Embedded MongoDB");
			embeddedMongoDB = new EmbeddedMongoDb(conf);
			embeddedMongoDB.start(testMode);
		}
	}

	public RestResourceManager getResourcesManager() {
		return resourcesManager;
	}

	public static void stopEmbeddedMongDB() throws Exception {
		if(embeddedMongoDB != null) {
			LOG.info("Stop Embedded MongoDB");
			embeddedMongoDB.stop();
			embeddedMongoDB = null;
		}
	}

	public void activatePubSubServerServlet(Environment environment) throws Exception {
		if(pubSubWebsocket == null) {
			pubSubWebsocket =  environment.servlets().addServlet("ws_pubsub", new PubSubServerServlet());
			pubSubWebsocket.setAsyncSupported(true);
			pubSubWebsocket.addMapping("/ws/pubsub/*");

			final ServletRegistration.Dynamic websocket = environment.servlets().addServlet("ws_sockjs", new SockJSServerServlet());
			websocket.setAsyncSupported(true);
			websocket.addMapping("/sockjs-node/*");
		}
	}


	public void addWebSocketResources(Class<?> clazz) {
		webSocketResources.add(clazz);
	}


	public void stop() throws Exception {
		LOG.debug(this.getName() + "is going to stop.");
		if(this.environment != null &&  this.environment.getAdminContext() != null &&  this.environment.getAdminContext().getServer() != null) {
			this.environment.getAdminContext().getServer().stop();
			this.environment.getAdminContext().getServer().destroy();
		}
		if(this.environment != null &&  this.environment.getApplicationContext() != null &&  this.environment.getApplicationContext().getServer() != null) {
			this.environment.getApplicationContext().getServer().stop();
			this.environment.getApplicationContext().getServer().destroy();
		}
		if(WEB_SOCKET_COMMAND_SERVER != null) {
			StaticJettyServer.getInstance().unPublishJAXWSEndpoint(WEB_SOCKET_COMMAND_SERVER);
			WEB_SOCKET_COMMAND_SERVER.stop();
			WEB_SOCKET_COMMAND_SERVER = null;
		}
		if(ModelsResource.ASYNC_PUBLISH_EXTRACT_CONSUMER != null) {
			ModelsResource.ASYNC_PUBLISH_EXTRACT_CONSUMER.stop();
			ModelsResource.ASYNC_PUBLISH_EXTRACT_CONSUMER = null;
		}
		if(HUMAN_TASK_CONSUMER != null) {
			HUMAN_TASK_CONSUMER.stop();
			HUMAN_TASK_CONSUMER = null;
		}
		stopEmbeddedMongDB();
		
		OllamaManager.getInstance().stopEmbeddedContainer();
	}

	public String getApplicationUrl() {
		return applicationUrl;
	}


	private void createNotifierForHumanTaskEvent() throws Exception {
		if(HUMAN_TASK_CONSUMER == null) {
			final String notifierClientAddress = SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP, conf.getHost(), Integer.parseInt(conf.getProperties().get("humantask-event-notifier-port")), Integer.parseInt(conf.getProperties().get("proxy-port")), "/HumanTaskEventNotifierClient");
			final AbstractNotifierClient notifier = new RIOHumanTaskNotifierClient(notifierClientAddress, WEB_SOCKET_COMMAND, getName());
			HUMAN_TASK_CONSUMER = new NotificationConsumerWebService();
			HUMAN_TASK_CONSUMER.start(new HashMap<String, Object>(){{put("host","0.0.0.0"); put("port", conf.getProperties().get("humantask-event-notifier-port")); put("serviceName", "HumanTaskEventNotifierClient"); put("notifierClient", notifier); }});

		}
	}

	public static String generateIcon(SVGGenerator svgGen, GJaxbRelation edge,
			String collaborationName, String knowledgeSpaceName) throws Exception {
		Document doc = svgGen.createSVGFromEdge(edge);

		String cn = "";
		if(collaborationName != null) {
			cn = cn + RegExpHelper.toRegexFriendlyName(collaborationName) + "/";
		}
		if(knowledgeSpaceName != null) {
			cn = cn + RegExpHelper.toRegexFriendlyName(knowledgeSpaceName) + "/";
		}

		File target = new File(RIOConstant.RESOURCES_FOLDER);
		target = new File(target, "edges_icons/" + cn);


		String webName = null;
		if(edge.getType() != null) {
			String fileName = RegExpHelper.toRegexFriendlyName(edge.getType().getLocalPart()) + ".svg";

			File svgFile = new File(target, fileName).getCanonicalFile();
			if(!svgFile.exists()) {
				target.mkdirs();
				svgFile.createNewFile();
				FileUtil.setContents(svgFile, XMLPrettyPrinter.print(doc));
			}


			webName = "/" + "##application_name##" + "/webjars/" + "resourcesFolder/edges_icons/" + cn + fileName;
			if(!edge.isSetEdgeMetaModeling()) {
				edge.setEdgeMetaModeling(new GJaxbEdgeMetaModeling());
			}
			if(!edge.getEdgeMetaModeling().isSetModelingEdgeViewDefinition()) {
				edge.getEdgeMetaModeling().setModelingEdgeViewDefinition(new GJaxbModelingEdgeViewDefinition());
			}
			edge.getEdgeMetaModeling().getModelingEdgeViewDefinition().setIcon(webName);
		}
		return webName;
	}


	public static void copySharedResources() throws Exception {

		File target = new File(RIOConstant.RESOURCES_FOLDER);
		
		String shareDirectory = "META-INF/resources/webjars/app/share/";
		
		URL root = Thread.currentThread().getContextClassLoader().getResource(shareDirectory);
		LOG.debug("root - " + root);
		if(root == null) {
			throw new Exception("Shared resources not found!!!");
		}
		
		Path myPath = null;
		if (root.getProtocol().equals("jar")) {
			LOG.debug("get mypath from jar");
			FileSystem fileSystem = null;
			try {
				fileSystem = FileSystems.getFileSystem(root.toURI());
			} catch(FileSystemNotFoundException e) {
				fileSystem = FileSystems.newFileSystem(root.toURI(), java.util.Collections.<String, Object>emptyMap());
			}

			myPath = fileSystem.getPath(shareDirectory);
			
		} else {
			LOG.debug("get mypath from file");
			myPath = Paths.get(root.toURI());
		}
		
        List<URI> collect = Files.walk(myPath)
                .filter(Files::isRegularFile)
                .map(x -> x.toUri())
                .collect(Collectors.toList());
        
        for(URI shareFileInJar: collect) {
        	String pathAndFileName = shareFileInJar.toString().substring(shareFileInJar.toString().indexOf("/share/"));
        	LOG.debug("Copy share resource: " + pathAndFileName);
        	File shareFileInFolder = new File(target, pathAndFileName);
        	shareFileInFolder.getParentFile().mkdirs();
        	FileOutputStream out = new FileOutputStream(shareFileInFolder);
        	FileUtil.copy(shareFileInJar.toURL().openStream(), out);
        	out.close();
        }

	}



}
