/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application.auth;

/*
 * #%L
 * dw-users
 * %%
 * Copyright (C) 2014 - 2017 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */




import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.Principal;

import io.dropwizard.auth.AuthFilter;
import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import io.dropwizard.auth.basic.BasicCredentials;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.container.ContainerRequestContext;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.SecurityContext;
import java.util.Base64;

public class BasicCredentialAuthFilter<P extends Principal> extends AuthFilter<BasicCredentials, P> {



  @Override
  public void filter(ContainerRequestContext requestContext) throws IOException {
    final BasicCredentials credentials =
        getCredentials(requestContext.getHeaders().getFirst(HttpHeaders.AUTHORIZATION));
    Boolean authenticate = true;
    try {
    	authenticate = authenticate(requestContext, credentials, SecurityContext.BASIC_AUTH);
    } catch(Throwable e) {
    	throw new WebApplicationException("Authentication Failed!!! Perhaps that the user is not registered !!!", Response.Status.UNAUTHORIZED);
    }
    if (!authenticate) {
      throw new WebApplicationException(unauthorizedHandler.buildResponse(prefix, realm));
    }
  }

  /**
   * Parses a Base64-encoded value of the `Authorization` header
   * in the form of `Basic dXNlcm5hbWU6cGFzc3dvcmQ=`.
   *
   * @param header the value of the `Authorization` header
   * @return a username and a password as {@link BasicCredentials}
   */
  private BasicCredentials getCredentials(String header) {
	  if (header == null) {
          return null;
      }

      final int space = header.indexOf(' ');
      if (space <= 0) {
          return null;
      }

      final String method = header.substring(0, space);
      if (!prefix.equalsIgnoreCase(method)) {
          return null;
      }

      final String decoded;
      try {
          decoded = new String(Base64.getDecoder().decode(header.substring(space + 1)), StandardCharsets.UTF_8);
      } catch (IllegalArgumentException e) {
          logger.warn("Error decoding credentials", e);
          return null;
      }

      // Decoded credentials is 'username:password'
      final int i = decoded.indexOf(':');
      if (i <= 0) {
          return null;
      }

      final String username = decoded.substring(0, i);
      final String password = decoded.substring(i + 1);
      return new BasicCredentials(username, password);
  }

  /**
   * Builder for {@link BasicCredentialAuthFilter}.
   * <p>An {@link Authenticator} must be provided during the building process.</p>
   *
   * @param <P> the principal
   */
  public static class Builder<P extends Principal> extends
  AuthFilterBuilder<BasicCredentials, P, BasicCredentialAuthFilter<P>> {

    @Override
    protected BasicCredentialAuthFilter<P> newInstance() {
      return new BasicCredentialAuthFilter<>();
    }
  }
}
