/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application.bundles;

import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.CRC32;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.net.HttpHeaders;

import io.dropwizard.servlets.assets.ByteRange;
import io.dropwizard.servlets.assets.ResourceURL;
import io.dropwizard.util.Resources;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Invocation;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status.Family;

public class AssetInterceptorServlet extends HttpServlet {
	private static final long serialVersionUID = 6393345594784987908L;

	private Map<String, AbstractModifierResource> mapModifiers = null;

	private static Logger LOG = LoggerFactory.getLogger(AssetInterceptorServlet.class.getName());


	// HTTP header names
	private static final String IF_MODIFIED_SINCE = "If-Modified-Since";
	private static final String IF_NONE_MATCH = "If-None-Match";
	private static final String IF_RANGE = "If-Range";
	private static final String RANGE = "Range";
	private static final String ACCEPT_RANGES = "Accept-Ranges";
	private static final String CONTENT_RANGE = "Content-Range";
	private static final String ETAG = "ETag";
	private static final String LAST_MODIFIED = "Last-Modified";

	private static class CachedAsset {
		private final byte[] resource;
		private final String eTag;
		private final long lastModifiedTime;

		private CachedAsset(byte[] resource, long lastModifiedTime) {
			this.resource = resource;
			this.eTag = '"' + hash(resource) + '"';
			this.lastModifiedTime = lastModifiedTime;
		}

		private static String hash(byte[] resource) {
			final CRC32 crc32 = new CRC32();
			crc32.update(resource);
			return Long.toHexString(crc32.getValue());
		}

		public byte[] getResource() {
			return resource;
		}

		public String getETag() {
			return eTag;
		}

		public long getLastModifiedTime() {
			return lastModifiedTime;
		}
	}

	private static final String DEFAULT_MEDIA_TYPE = "text/html";

	private final String resourcePath;
	private final String uriPath;

	@Nullable
	private final String indexFile;

	private final String defaultMediaType;

	@Nullable
	private final Charset defaultCharset;

	/**
	 * Creates a new {@code AssetServlet} that serves static assets loaded from {@code resourceURL}
	 * (typically a file: or jar: URL). The assets are served at URIs rooted at {@code uriPath}. For
	 * example, given a {@code resourceURL} of {@code "file:/data/assets"} and a {@code uriPath} of
	 * {@code "/js"}, an {@code AssetServlet} would serve the contents of {@code
	 * /data/assets/example.js} in response to a request for {@code /js/example.js}. If a directory
	 * is requested and {@code indexFile} is defined, then {@code AssetServlet} will attempt to
	 * serve a file with that name in that directory. If a directory is requested and {@code
	 * indexFile} is null, it will serve a 404.
	 *
	 * @param resourcePath   the base URL from which assets are loaded
	 * @param uriPath        the URI path fragment in which all requests are rooted
	 * @param indexFile      the filename to use when directories are requested, or null to serve no
	 *                       indexes
	 * @param defaultCharset the default character set
	 */
	public AssetInterceptorServlet(String resourcePath,
			String uriPath,
			@Nullable String indexFile,
			@Nullable Charset defaultCharset, Map<String, AbstractModifierResource> mapModifiers) {
		this(resourcePath, uriPath, indexFile, DEFAULT_MEDIA_TYPE, defaultCharset, mapModifiers);
	}

	/**
	 * Creates a new {@code AssetServlet} that serves static assets loaded from {@code resourceURL}
	 * (typically a file: or jar: URL). The assets are served at URIs rooted at {@code uriPath}. For
	 * example, given a {@code resourceURL} of {@code "file:/data/assets"} and a {@code uriPath} of
	 * {@code "/js"}, an {@code AssetServlet} would serve the contents of {@code
	 * /data/assets/example.js} in response to a request for {@code /js/example.js}. If a directory
	 * is requested and {@code indexFile} is defined, then {@code AssetServlet} will attempt to
	 * serve a file with that name in that directory. If a directory is requested and {@code
	 * indexFile} is null, it will serve a 404.
	 *
	 * @param resourcePath     the base URL from which assets are loaded
	 * @param uriPath          the URI path fragment in which all requests are rooted
	 * @param indexFile        the filename to use when directories are requested, or null to serve no
	 *                         indexes
	 * @param defaultMediaType the default media type
	 * @param defaultCharset   the default character set
	 * @since 2.0
	 */
	public AssetInterceptorServlet(String resourcePath,
			String uriPath,
			@Nullable String indexFile,
			@Nullable String defaultMediaType,
			@Nullable Charset defaultCharset, Map<String, AbstractModifierResource> mapModifiers) {
		final String trimmedPath = trimSlashes(resourcePath);
		this.resourcePath = trimmedPath.isEmpty() ? trimmedPath : trimmedPath + '/';
		final String trimmedUri = trimTrailingSlashes(uriPath);
		this.uriPath = trimmedUri.isEmpty() ? "/" : trimmedUri;
		this.indexFile = indexFile;
		this.defaultMediaType = defaultMediaType == null ? DEFAULT_MEDIA_TYPE : defaultMediaType;
		this.defaultCharset = defaultCharset;
		this.mapModifiers = mapModifiers;
	}

	private static String trimSlashes(String s) {
		final Matcher matcher = Pattern.compile("^/*(.*?)/*$").matcher(s);
		if (matcher.find()) {
			return matcher.group(1);
		} else {
			return s;
		}
	}

	private static String trimTrailingSlashes(String s) {
		final Matcher matcher = Pattern.compile("(.*?)/*$").matcher(s);
		if (matcher.find()) {
			return matcher.group(1);
		} else {
			return s;
		}
	}


	public String getUriPath() {
		return uriPath;
	}

	@Nullable
	public String getIndexFile() {
		return indexFile;
	}

	/**
	 * @since 2.0
	 */
	public String getDefaultMediaType() {
		return defaultMediaType;
	}


	/**
	 * @since 2.0
	 */
	@Nullable
	public Charset getDefaultCharset() {
		return defaultCharset;
	}

	@Override
	protected void doGet(HttpServletRequest req,
			HttpServletResponse resp) throws ServletException, IOException {
		try {
			final StringBuilder builder = new StringBuilder(req.getServletPath());
			if (req.getPathInfo() != null) {
				builder.append(req.getPathInfo());
			}
			final CachedAsset cachedAsset = loadAsset(builder.toString(), req);
			if (cachedAsset == null) {
				resp.sendError(HttpServletResponse.SC_NOT_FOUND);
				return;
			}

			if (isCachedClientSide(req, cachedAsset)) {
				resp.sendError(HttpServletResponse.SC_NOT_MODIFIED);
				return;
			}

			final String rangeHeader = req.getHeader(RANGE);

			final int resourceLength = cachedAsset.getResource().length;
			List<ByteRange> ranges = Collections.emptyList();

			boolean usingRanges = false;
			// Support for HTTP Byte Ranges
			// http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html
			if (rangeHeader != null) {

				final String ifRange = req.getHeader(IF_RANGE);

				if (ifRange == null || cachedAsset.getETag().equals(ifRange)) {
					ranges = parseRangeHeader(rangeHeader, resourceLength);

					if (ranges.isEmpty()) {
						resp.sendError(HttpServletResponse.SC_REQUESTED_RANGE_NOT_SATISFIABLE);
						return;
					}

					resp.setStatus(HttpServletResponse.SC_PARTIAL_CONTENT);
					usingRanges = true;

					final String byteRanges = ranges.stream()
							.map(ByteRange::toString)
							.collect(Collectors.joining(","));
					resp.addHeader(CONTENT_RANGE, "bytes " + byteRanges + "/" + resourceLength);
				}
			}

			resp.setDateHeader(LAST_MODIFIED, cachedAsset.getLastModifiedTime());
			resp.setHeader(ETAG, cachedAsset.getETag());

			final String requestUri = req.getRequestURI();
			final String mediaType = Optional.ofNullable(req.getServletContext().getMimeType(
					indexFile != null && requestUri.endsWith("/") ? requestUri + indexFile : requestUri))
					.orElse(defaultMediaType);
			if (mediaType.startsWith("video") || mediaType.startsWith("audio") || usingRanges) {
				resp.addHeader(ACCEPT_RANGES, "bytes");
			}

			resp.setContentType(mediaType);
			if (defaultCharset != null) {
				resp.setCharacterEncoding(defaultCharset.toString());
			}

			try (ServletOutputStream output = resp.getOutputStream()) {
				if (usingRanges) {
					for (ByteRange range : ranges) {
						output.write(cachedAsset.getResource(), range.getStart(),
								range.getEnd() - range.getStart() + 1);
					}
				} else {
					output.write(cachedAsset.getResource());
				}
			}
		} catch (RuntimeException | URISyntaxException ignored) {
			if (!resp.isCommitted()) {
				resp.reset();
				resp.setStatus(HttpServletResponse.SC_NOT_FOUND);
			}
		}
	}

	@Nullable
	private CachedAsset loadAsset(String key, HttpServletRequest req) throws URISyntaxException, IOException {
		if (!key.startsWith(uriPath) || key.startsWith("/generic-application")) {
			throw new IllegalArgumentException("Cache key must start with " + uriPath + " or " + "/generic-application");
		}

		if(key.startsWith("/generic-application")) {
			key = key.replace("/generic-application/webjars", uriPath);
		}
		if(key.contains("(") && key.contains(")")) {
			key = key.substring(0, key.indexOf("(")) + "index.html";
		}


		final String requestedResourcePath = trimSlashes(key.substring(uriPath.length()));
		final String absoluteRequestedResourcePath = trimSlashes(this.resourcePath + requestedResourcePath);

		

		List<AbstractModifierResource> modifiers = findModifiers(absoluteRequestedResourcePath);;
		LOG.info("absoluteRequestedResourcePath = " + absoluteRequestedResourcePath);

		byte[] buffer = null;
		long lastModified = System.currentTimeMillis();


		if(!absoluteRequestedResourcePath.contains("/webjars/resourcesFolder") && !absoluteRequestedResourcePath.contains("/generatePages/") && !absoluteRequestedResourcePath.contains("/generatePagesWithMultipleContext/")) {

			
			
			if(!absoluteRequestedResourcePath.contains("/NEXUS")) {
				
				URL requestedResourceURL = getResourceURL(absoluteRequestedResourcePath);
				lastModified = ResourceURL.getLastModified(requestedResourceURL);
				if (lastModified < 1) {
					// Something went wrong trying to get the last modified time: just use the current time
					lastModified = System.currentTimeMillis();
				}

				// zero out the millis since the date we get back from If-Modified-Since will not have them
				lastModified = (lastModified / 1000) * 1000;
				
				requestedResourceURL = Resources.getResource(absoluteRequestedResourcePath);


				if (ResourceURL.isDirectory(requestedResourceURL)) {
					if (indexFile != null) {
						requestedResourceURL = getResourceURL(absoluteRequestedResourcePath + '/' + indexFile);
					} else {
						// directory requested but no index file defined
						return null;
					}
				}


				// return new CachedAsset(readResource(requestedResourceURL), lastModified);
				buffer = readResource(requestedResourceURL);
			} else {
				buffer = this.getNexusResource(absoluteRequestedResourcePath, req.getQueryString());
				
			}

		} 

		try {
			if(modifiers != null && !modifiers.isEmpty()) {
				for(AbstractModifierResource modifier: modifiers) {
					if(buffer != null) {
						byte[] newBuffer = modifier.replace(key, buffer);
						if(newBuffer != null) {
							buffer = newBuffer;
						}
					} else {
						byte[] newBuffer = modifier.replace(key, null);
						if(newBuffer != null) {
							buffer = newBuffer;
						}
					}
				}
			} else {
				LOG.warn("Impossible to find modifier for: " + requestedResourcePath);
			}
		} catch(Throwable e) {
			// do nothing
			e.printStackTrace();
			throw new IOException(e);
		}
		return new CachedAsset(buffer, lastModified);
	}

	/**
	@Nullable
	private CachedAsset loadAsset(String key, HttpServletRequest req) throws URISyntaxException, IOException {
		Preconditions.checkArgument(key.startsWith(uriPath) || key.startsWith("/generic-application"));

		if(key.startsWith("/generic-application")) {
			key = key.replace("/generic-application/webjars", uriPath);
		}
		if(key.contains("(") && key.contains(")")) {
			key = key.substring(0, key.indexOf("(")) + "index.html";
		}


		final String requestedResourcePath = CharMatcher.is('/').trimFrom(key.substring(uriPath.length()));
		final String absoluteRequestedResourcePath = CharMatcher.is('/').trimFrom(
				this.resourcePath + requestedResourcePath);

		URL requestedResourceURL = null;
		long lastModified = System.currentTimeMillis();
		byte[] buffer = null;
		List<AbstractModifierResource> modifiers = null;

		if(key.endsWith("/ws")) {
			return null;
		}

		try {

			LOG.info("absoluteRequestedResourcePath = " + absoluteRequestedResourcePath);
			if(!absoluteRequestedResourcePath.contains("/webjars/resourcesFolder") && !absoluteRequestedResourcePath.contains("/generatePages/") && !absoluteRequestedResourcePath.contains("/generatePagesWithMultipleContext/")) {

				if(!absoluteRequestedResourcePath.contains("/NEXUS")) {
					requestedResourceURL = Resources.getResource(absoluteRequestedResourcePath);

					if (ResourceURL.isDirectory(requestedResourceURL)) {
						if (indexFile != null) {
							requestedResourceURL = Resources.getResource(absoluteRequestedResourcePath + '/' + indexFile);
						} else {
							// directory requested but no index file defined
							return null;
						}
					}


					lastModified = ResourceURL.getLastModified(requestedResourceURL);
					if (lastModified < 1) {
						// Something went wrong trying to get the last modified time: just use the current time
						lastModified = System.currentTimeMillis();
					}


					// zero out the millis since the date we get back from If-Modified-Since will not have them
					lastModified = (lastModified / 1000) * 1000;

					buffer = FileUtil.getContents(requestedResourceURL.openStream()).getBytes();
					modifiers = findModifiers(requestedResourceURL.toString());

				} else {
					buffer = this.getNexusResource(absoluteRequestedResourcePath, req.getQueryString());
					modifiers = findModifiers(absoluteRequestedResourcePath);
				}

			} else {
				modifiers = findModifiers(absoluteRequestedResourcePath);
			}



			if(modifiers != null && !modifiers.isEmpty()) {
				for(AbstractModifierResource modifier: modifiers) {
					if(buffer != null) {
						byte[] newBuffer = modifier.replace(key, buffer);
						if(newBuffer != null) {
							buffer = newBuffer;
						}
					} else {
						byte[] newBuffer = modifier.replace(key, null);
						if(newBuffer != null) {
							buffer = newBuffer;
						}
					}
				}
			} else {
				LOG.warn("Impossible to find modifier for: " + requestedResourcePath);
			}
		} catch(Throwable e) {
			// do nothing
			e.printStackTrace();
		}

		return new CachedAsset(buffer, lastModified);
	}
	 */


	private byte[] getNexusResource(String requestedResourcePath, String queryString) throws IOException {
		byte[] buffer = null;
		if(requestedResourcePath.contains("/NEXUS")) {
			try {
				String nexusUrl = queryString.substring(queryString.indexOf("private_resource=") + "private_resource=".length());
				String login = System.getenv().get("RIO_MEDIA_LOGIN");
				String pwd = System.getenv().get("RIO_MEDIA_PWD");

				if(login == null || pwd == null) {
					throw new IOException("RIO_MEDIA_LOGIN and/or RIO_MEDIA_PWD are not correctly set !!!");
				}

				String encoding = Base64.getEncoder().encodeToString((login + ":" + pwd).getBytes("UTF-8"));

				Client client = ClientBuilder.newClient();
				WebTarget webTarget = client.target(nexusUrl);

				Invocation.Builder invocationBuilder = webTarget.request(jakarta.ws.rs.core.MediaType.WILDCARD);
				invocationBuilder.header(HttpHeaders.AUTHORIZATION, "Basic " + encoding);
				Response response = invocationBuilder.get();

				if (response.getStatusInfo().getFamily() == Family.SUCCESSFUL) {
					LOG.debug("Success! " + response.getStatus());
					LOG.debug("" + response.getEntity());
					InputStream downloadFile = response.readEntity(InputStream.class);
					buffer = downloadFile.readAllBytes();
				} else {
					LOG.debug("ERROR! " + response.getStatus());    
					LOG.debug("" + response.getEntity());
					InputStream downloadFile = response.readEntity(InputStream.class);
					buffer = downloadFile.readAllBytes();
				}
			} catch(Throwable e) {
				e.printStackTrace();
				throw new IOException(e);
			}
		}
		return buffer;
	}

	private List<AbstractModifierResource> findModifiers(String requestedResourceURL) {

		List<AbstractModifierResource> acceptedModifiers = new ArrayList<AbstractModifierResource>();

		if(this.mapModifiers != null) {
			for(AbstractModifierResource modifier: this.mapModifiers.values()) {
				if(modifier.accept(requestedResourceURL)) {
					acceptedModifiers.add(modifier);
				}

			}
		}
		return acceptedModifiers;
	}

	protected URL getResourceURL(String absoluteRequestedResourcePath) {
		return Resources.getResource(absoluteRequestedResourcePath);
	}

	protected byte[] readResource(URL requestedResourceURL) throws IOException {
		try (InputStream inputStream = requestedResourceURL.openStream()) {
			return inputStream.readAllBytes();
		}
	}

	private boolean isCachedClientSide(HttpServletRequest req, CachedAsset cachedAsset) {
		// https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/If-Modified-Since
		// Indicates that with the presense of If-None-Match If-Modified-Since should be ignored.
		String ifNoneMatchHeader = req.getHeader(IF_NONE_MATCH);
		if (ifNoneMatchHeader != null) {
			return cachedAsset.getETag().equals(ifNoneMatchHeader);
		} else {
			return req.getDateHeader(IF_MODIFIED_SINCE) >= cachedAsset.getLastModifiedTime();
		}
	}

	/**
	 * Parses a given Range header for one or more byte ranges.
	 *
	 * @param rangeHeader    Range header to parse
	 * @param resourceLength Length of the resource in bytes
	 * @return List of parsed ranges
	 */
	private List<ByteRange> parseRangeHeader(final String rangeHeader, final int resourceLength) {
		try {
			final List<ByteRange> byteRanges;
			if (rangeHeader.contains("=")) {
				final String[] parts = rangeHeader.split("=", -1);
				if (parts.length > 1) {
					byteRanges = Arrays.stream(parts[1].split(",", -1))
							.map(String::trim)
							.map(s -> ByteRange.parse(s, resourceLength))
							.collect(Collectors.toList());
				} else {
					byteRanges = Collections.emptyList();
				}
			} else {
				byteRanges = Collections.emptyList();
			}
			return byteRanges;
		} catch (NumberFormatException e) {
			return Collections.emptyList();
		}
	}
}
