/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application.bundles;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.dropwizard.core.Configuration;
import io.dropwizard.core.ConfiguredBundle;
import io.dropwizard.core.setup.Environment;

/**
 * A bundle for serving static asset files from the classpath.
 */
/**
 *
 *
 * @author Nicolas Salatge
 */
public class AssetsInterceptorBundle implements ConfiguredBundle<Configuration> {
  private static final Logger LOGGER = LoggerFactory.getLogger(AssetsInterceptorBundle.class);

  private static final String DEFAULT_ASSETS_NAME = "assets";
  private static final String DEFAULT_INDEX_FILE = "index.html";
  private static final String DEFAULT_PATH = "/assets";
  private static final String DEFAULT_MEDIA_TYPE = "text/html";

  private final String resourcePath;
  private final String uriPath;
  private final String indexFile;
  private final String assetsName;
  private final String defaultMediaType;
  
  private Map<String, AbstractModifierResource> mapModifiers = new HashMap<String, AbstractModifierResource>();
  /**
   * Creates a new AssetsBundle which serves up static assets from
   */
  public AssetsInterceptorBundle(AbstractModifierResource... abstractModifierResources) {
      this(DEFAULT_PATH, DEFAULT_PATH, DEFAULT_INDEX_FILE, DEFAULT_ASSETS_NAME, DEFAULT_MEDIA_TYPE, abstractModifierResources);
  }

  /**
   * Creates a new AssetsBundle which will configure the application to serve the static files
   * located in {@code src/main/resources/${path}} as {@code /${path}}. For example, given a
   * {@code path} of {@code "/assets"}, {@code src/main/resources/assets/example.js} would be
   * served up from {@code /assets/example.js}.
   *
   */
  public AssetsInterceptorBundle(String path, AbstractModifierResource... abstractModifierResources) {
      this(path, path, DEFAULT_INDEX_FILE, DEFAULT_ASSETS_NAME, DEFAULT_MEDIA_TYPE, abstractModifierResources);
  }

  /**
   * Creates a new AssetsBundle which will configure the application to serve the static files
   * located in {@code src/main/resources/${resourcePath}} as {@code /${uriPath}}. For example, given a
   * {@code resourcePath} of {@code "/assets"} and a uriPath of {@code "/js"},
   * {@code src/main/resources/assets/example.js} would be served up from {@code /js/example.js}.
   *
   */
  public AssetsInterceptorBundle(String resourcePath, String uriPath, AbstractModifierResource... abstractModifierResources) {
      this(resourcePath, uriPath, DEFAULT_INDEX_FILE, DEFAULT_ASSETS_NAME, DEFAULT_MEDIA_TYPE, abstractModifierResources);
  }

  /**
   * Creates a new AssetsBundle which will configure the application to serve the static files
   * located in {@code src/main/resources/${resourcePath}} as {@code /${uriPath}}. If no file name is
   * in ${uriPath}, ${indexFile} is appended before serving. For example, given a
   * {@code resourcePath} of {@code "/assets"} and a uriPath of {@code "/js"},
   * {@code src/main/resources/assets/example.js} would be served up from {@code /js/example.js}.
   *
   */
  public AssetsInterceptorBundle(String resourcePath, String uriPath, String indexFile, AbstractModifierResource... abstractModifierResources) {
      this(resourcePath, uriPath, indexFile, DEFAULT_ASSETS_NAME, DEFAULT_MEDIA_TYPE, abstractModifierResources);
  }

  /**
   * Creates a new AssetsBundle which will configure the application to serve the static files
   * located in {@code src/main/resources/${resourcePath}} as {@code /${uriPath}}. If no file name is
   * in ${uriPath}, ${indexFile} is appended before serving. For example, given a
   * {@code resourcePath} of {@code "/assets"} and a uriPath of {@code "/js"},
   * {@code src/main/resources/assets/example.js} would be served up from {@code /js/example.js}.
   *
   * @param resourcePath the resource path (in the classpath) of the static asset files
   * @param uriPath      the uri path for the static asset files
   * @param indexFile    the name of the index file to use
   * @param assetsName   the name of servlet mapping used for this assets bundle
   */
  public AssetsInterceptorBundle(String resourcePath, String uriPath, String indexFile, String assetsName, AbstractModifierResource... abstractModifierResources) {
      this(resourcePath, uriPath, indexFile, assetsName, DEFAULT_MEDIA_TYPE, abstractModifierResources);
  }

  /**
   * Creates a new AssetsBundle which will configure the application to serve the static files
   * located in {@code src/main/resources/${resourcePath}} as {@code /${uriPath}}. If no file name is
   * in ${uriPath}, ${indexFile} is appended before serving. For example, given a
   * {@code resourcePath} of {@code "/assets"} and a uriPath of {@code "/js"},
   * {@code src/main/resources/assets/example.js} would be served up from {@code /js/example.js}.
   *
   * @param resourcePath     the resource path (in the classpath) of the static asset files
   * @param uriPath          the uri path for the static asset files
   * @param indexFile        the name of the index file to use
   * @param assetsName       the name of servlet mapping used for this assets bundle
   * @param defaultMediaType the default media type for unknown file extensions
   * @since 2.0
   */
  public AssetsInterceptorBundle(String resourcePath, String uriPath, String indexFile, String assetsName,
                      String defaultMediaType, AbstractModifierResource... abstractModifierResources) {
      if (!resourcePath.startsWith("/")) {
          throw new IllegalArgumentException(resourcePath + " is not an absolute path");
      }

      if ("/".equals(resourcePath)) {
          throw new IllegalArgumentException(resourcePath + " is the classpath root");
      }

      this.resourcePath = resourcePath.endsWith("/") ? resourcePath : (resourcePath + '/');
      this.uriPath = uriPath.endsWith("/") ? uriPath : (uriPath + '/');
      this.indexFile = indexFile;
      this.assetsName = assetsName;
      this.defaultMediaType = defaultMediaType;
      for(AbstractModifierResource modifier: abstractModifierResources) {
        addModifierResource(modifier);
      }
  }
  
  public void addModifierResource(AbstractModifierResource modifier) {
    mapModifiers.put(modifier.getClass().getName(), modifier);
  }

  @Override
  public void run(Configuration configuration, Environment environment) {
      final String pathPattern = uriPath + '*';
      LOGGER.info("Registering AssetBundle with name: {} for path {}", assetsName, pathPattern);
      environment.servlets().addServlet(assetsName, createServlet()).addMapping(pathPattern);
  }

  public String getResourcePath() {
      return resourcePath;
  }

  public String getUriPath() {
      return uriPath;
  }

  public String getIndexFile() {
      return indexFile;
  }

  /**
   * @since 2.0
   */
  public String getDefaultMediaType() {
      return defaultMediaType;
  }

  protected AssetInterceptorServlet createServlet() {
      return new AssetInterceptorServlet(resourcePath, uriPath, indexFile, defaultMediaType, StandardCharsets.UTF_8, mapModifiers);
  }

 
}
