/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application.configuration;

import java.io.IOException;

/*
 * #%L
 * dw-generic-application
 * %%
 * Copyright (C) 2014 - 2017 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import com.fasterxml.jackson.databind.ObjectMapper;

import io.dropwizard.core.Configuration;
import io.dropwizard.core.cli.Cli;
import io.dropwizard.core.cli.Command;
import io.dropwizard.configuration.ConfigurationException;
import io.dropwizard.configuration.ConfigurationFactory;
import io.dropwizard.configuration.ConfigurationFactoryFactory;
import io.dropwizard.configuration.ConfigurationSourceProvider;
import io.dropwizard.core.setup.Bootstrap;
import io.dropwizard.util.Generics;
import jakarta.validation.Validator;
import net.sourceforge.argparse4j.inf.Argument;
import net.sourceforge.argparse4j.inf.Namespace;
import net.sourceforge.argparse4j.inf.Subparser;

/**
 * A command whose first parameter is the location of a YAML configuration file. That file is parsed
 * into an instance of a {@link Configuration} subclass, which is then validated. If the
 * configuration is valid, the command is run.
 *
 * @param <T> the {@link Configuration} subclass which is loaded from the configuration file
 * @see Configuration
 */
public class ConfiguredCommandWithoutLog<T extends Configuration> extends Command {
    private boolean asynchronous;

    private T configuration;

    protected ConfiguredCommandWithoutLog(String name, String description) {
      super(name, description);
      this.asynchronous = false;
  }

    /**
     * Returns the {@link Class} of the configuration type.
     *
     * @return the {@link Class} of the configuration type
     */
    protected Class<T> getConfigurationClass() {
        return Generics.getTypeParameter(getClass(), Configuration.class);
    }

    /**
     * Configure the command's {@link Subparser}. <p><strong> N.B.: if you override this method, you
     * <em>must</em> call {@code super.override(subparser)} in order to preserve the configuration
     * file parameter in the subparser. </strong></p>
     *
     * @param subparser the {@link Subparser} specific to the command
     */
    @Override
    public void configure(Subparser subparser) {
        addFileArgument(subparser);
    }

    /**
     * Adds the configuration file argument for the configured command.
     * @param subparser The subparser to register the argument on
     * @return the register argument
     */
    protected Argument addFileArgument(Subparser subparser) {
        return subparser.addArgument("file")
                        .nargs("?")
                        .help("application configuration file");
    }

    @Override
    @SuppressWarnings("unchecked")
    public void run(Bootstrap<?> wildcardBootstrap, Namespace namespace) throws Exception {
        final Bootstrap<T> bootstrap = (Bootstrap<T>) wildcardBootstrap;
        configuration = parseConfiguration(bootstrap.getConfigurationFactoryFactory(),
                                           bootstrap.getConfigurationSourceProvider(),
                                           bootstrap.getValidatorFactory().getValidator(),
                                           namespace.getString("file"),
                                           getConfigurationClass(),
                                           bootstrap.getObjectMapper());

        try {
//            if (configuration != null) {
//                configuration.getLoggingFactory().configure(bootstrap.getMetricRegistry(),
//                                                            bootstrap.getApplication().getName());
//            }

            run(bootstrap, namespace, configuration);
        } catch(Throwable e) {
            throw e;
        } finally {
            if (!asynchronous) {
                cleanup();
            }
        }
    }

    protected void cleanupAsynchronously() {
        this.asynchronous = true;
    }

    protected void cleanup() {
        if (configuration != null) {
            configuration.getLoggingFactory().stop();
        }
    }

    /**
     * Runs the command with the given {@link Bootstrap} and {@link Configuration}.
     *
     * @param bootstrap     the bootstrap bootstrap
     * @param namespace     the parsed command line namespace
     * @param configuration the configuration object
     * @throws Exception if something goes wrong
     */
    protected  void run(Bootstrap<T> bootstrap,
                                Namespace namespace,
                                T configuration) throws Exception {
      
    }

    private T parseConfiguration(ConfigurationFactoryFactory<T> configurationFactoryFactory,
                                 ConfigurationSourceProvider provider,
                                 Validator validator,
                                 String path,
                                 Class<T> klass,
                                 ObjectMapper objectMapper) throws IOException, ConfigurationException {
        final ConfigurationFactory<T> configurationFactory = configurationFactoryFactory
                .create(klass, validator, objectMapper, "dw");
        if (path != null) {
            return configurationFactory.build(provider, path);
        }
        return configurationFactory.build();
    }

    @Override
    public void onError(Cli cli, Namespace namespace, Throwable e) {
      super.onError(cli, namespace, e);
      throw new RuntimeException(e.getMessage(), e); 
    }
    
    
}
