/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application.configuration;

import org.eclipse.jetty.http.HttpCompliance;
import org.eclipse.jetty.server.HttpConnectionFactory;

/*
 * #%L
 * dw-generic-application
 * %%
 * Copyright (C) 2014 - 2017 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.util.component.AbstractLifeCycle;
import org.eclipse.jetty.util.component.LifeCycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.dropwizard.core.Application;
import io.dropwizard.core.Configuration;
import io.dropwizard.core.setup.Environment;
import net.sourceforge.argparse4j.inf.Namespace;

public class ServerCommandWithoutLog<T extends Configuration> extends EnvironmentCommandWithoutLog<T> {
  private static final Logger LOGGER = LoggerFactory.getLogger(ServerCommandWithoutLog.class);

  private final Class<T> configurationClass;

  public ServerCommandWithoutLog(Application<T> application) {
      this(application, "serverWithoutLog", "Runs the Dropwizard application wihtout as an HTTP server");
  }

  /**
   * A constructor to allow reuse of the server command as a different name
   * @param application the application using this command
   * @param name the argument name to invoke this command
   * @param description a summary of what the command does
   */
  protected ServerCommandWithoutLog(final Application<T> application, final String name, final String description) {
      super(application, name, description);
      this.configurationClass = application.getConfigurationClass();
  }

  /*
       * Since we don't subclass ServerCommand, we need a concrete reference to the configuration
       * class.
       */
  @Override
  protected Class<T> getConfigurationClass() {
      return configurationClass;
  }

  @Override
  protected void run(Environment environment, Namespace namespace, T configuration) throws Exception {
      final Server server = configuration.getServerFactory().build(environment);
      try {
          server.addEventListener(new LifeCycleListener());
          
          cleanupAsynchronously();
          server.start();
      } catch (Exception e) {
          LOGGER.error("Unable to start server, shutting down", e);
          try {
              server.stop();
          } catch (Exception e1) {
              LOGGER.warn("Failure during stop server", e1);
          }
          try {
              cleanup();
          } catch (Exception e2) {
              LOGGER.warn("Failure during cleanup", e2);
          }
          throw e;
      }
  }

  private class LifeCycleListener extends AbstractLifeCycle.AbstractLifeCycleListener {
      @Override
      public void lifeCycleStopped(LifeCycle event) {
          cleanup();
      }
  }
}
