/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources;

import java.util.List;

import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.emac.gind.commons.utils.net.IPUtil;
import fr.emac.gind.commons.utils.net.IPUtil.TRANSPORT_PROTOCOL;
import fr.emac.gind.commons.utils.ws.SPIWebServicePrimitives;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.gov.core.client.util.Neo4JId;
import fr.emac.gind.gov.core.client.util.Neo4JReqConstant;
import fr.emac.gind.humantask.GJaxbGetTodoList;
import fr.emac.gind.humantask.GJaxbGetTodoListResponse;
import fr.emac.gind.humantask.GJaxbSubscribeTasksByAssignedToResource;
import fr.emac.gind.humantask.GJaxbSubscribeTasksByAssignedToResourceResponse;
import fr.emac.gind.humantask.GJaxbTask;
import fr.emac.gind.humantask.GJaxbTask.PlansEdge;
import fr.emac.gind.humantask.GJaxbTodoList;
import fr.emac.gind.humantask.GJaxbUpdateTask;
import fr.emac.gind.humantask.GJaxbUpdateTaskResponse;
import fr.emac.gind.humantask.client.HumanTaskClient;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelManager;
import fr.emac.gind.rio.dw.resources.gov.ModelsResource;
import fr.emac.gind.humantask.Humantask;
import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/generic-application/persons/humantask")
//@Api("/humantask")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class HumanTaskResource {

	private static Logger LOG = LoggerFactory.getLogger(HumanTaskResource.class.getName());

	private Configuration conf = null;
	private Humantask htClient = null;
	private ModelsResource modelsResource = null;

	public HumanTaskResource(Configuration conf, ModelsResource modelsResource) throws Exception {
		this.conf = conf;
		this.htClient = HumanTaskClient.createClient(conf.getProperties().get("humantask"));
		this.modelsResource = modelsResource;
	}

	@GET
	@Path("/todoList/{assignedToId}")
	public GJaxbGetTodoListResponse todoList(@Auth DWUser user, @PathParam("assignedToId") String userId,
			@QueryParam("qAssignedToId") String qAssignedToId, @QueryParam("addPersonalTask") Boolean addPersonalTask)
			throws Exception {
		GJaxbGetTodoListResponse response = null;
		assert user != null;

		try {
			LOG.debug("qId = " + qAssignedToId);

			GJaxbGetTodoList request = new GJaxbGetTodoList();
			request.setAssignedToId(qAssignedToId);
			response = this.htClient.getTodoList(request);

			// add personal tasks
			if (addPersonalTask != null && addPersonalTask == true) {
				String query = null;
				if (qAssignedToId == null || qAssignedToId.equals("null")) {
					query = "match (r:`" + GenericModelHelper.collab("Resource") + "`)-[pl:`"
							+ GenericModelHelper.collab("Plans") + "`]->(pt:`"
							+ GenericModelHelper.collab("Personal_Task") + "`) return distinct r, pl, pt";
				} else {
					query = "match (r:`" + GenericModelHelper.collab("Resource") + "` { modelNodeId: '"
							+ Neo4JId.createIdUsingCollaboration(qAssignedToId, user.getCurrentCollaborationName(),
									user.getCurrentKnowledgeSpaceName())
							+ "' })-[pl:`" + GenericModelHelper.collab("Plans") + "`]->(pt:`"
							+ GenericModelHelper.collab("Personal_Task") + "`) return distinct r, pl, pt";
				}
				GJaxbGenericModel modelPersoTasks = this.modelsResource.query(user, query, null);
				GenericModelManager man = new GenericModelManager(modelPersoTasks);
				for (GJaxbNode personalTaskNode : man.getNodesByType(GenericModelHelper.collab("Personal_Task"))) {
					GJaxbTask personalTask = new GJaxbTask();
					personalTask.setDeployedTaskId("null");
					personalTask.setProcessInstanceId("null");

					List<GJaxbEdge> plansEdges = man.findInputEdgesOfNode(personalTaskNode);
					if (!plansEdges.isEmpty()) {
						GJaxbEdge plansEdge = plansEdges.get(0);
						personalTask.setPlansEdge(new PlansEdge());
						personalTask.getPlansEdge().setEdge(plansEdge);
						
						personalTask.getAssignedToIds().add(plansEdge.getSource().getId());
					}
					personalTask.setNode(personalTaskNode);
					if(response.getTodoList() == null) {
						response.setTodoList(new GJaxbTodoList());
					}
					response.getTodoList().getTask().add(personalTask);
				}
			}

			// set source and target of brokenEdge at null
			if (response.getTodoList() != null && response.getTodoList().getTask() != null) {
				for(GJaxbTask t: response.getTodoList().getTask()) {
					if(t.getPlansEdge() != null && t.getPlansEdge().getEdge() != null) {
						t.getPlansEdge().getEdge().setSource(null);
						t.getPlansEdge().getEdge().setTarget(null);
					}
					if (t.getNode().isSetBrokenEdges() && !t.getNode().getBrokenEdges().getBrokenEdge().isEmpty()) {
						t.getNode().getBrokenEdges().getBrokenEdge().forEach(be -> {
							be.getEdge().setSource(null);
							be.getEdge().setTarget(null);
						});
					}
				}
			}

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@PUT
	@Path("/updateTask")
	public GJaxbUpdateTaskResponse updateTask(@Auth DWUser user, String requestJson) throws Exception {
		GJaxbUpdateTaskResponse response = null;
		assert user != null;

		try {
			GJaxbUpdateTask request = JSONJAXBContext.getInstance()
					.unmarshall("{ \"updateTask\" : " + requestJson + " }", GJaxbUpdateTask.class);
			response = this.updateTask(user, request);
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	public GJaxbUpdateTaskResponse updateTask(@Auth DWUser user, GJaxbUpdateTask request) throws Exception {
		GJaxbUpdateTaskResponse response = null;
		assert user != null;

		try {
			assert request.getDeployedTaskId() != null;
			LOG.debug("human request.getId(): " + request.getDeployedTaskId());
			response = this.htClient.updateTask(request);
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/subscribeToAddTasksByAssignedToResource/{assignedToId}")
	public GJaxbSubscribeTasksByAssignedToResourceResponse subscribeToAddTasksByAssignedToResource(@Auth DWUser user,
			@PathParam("assignedToId") String assignedToId, @QueryParam("qAssignToId") String qAssignToId) throws Exception {
		int port = Integer.parseInt(conf.getProperties().get("humantask-event-notifier-port"));

		String callbackAddress = SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP, this.conf.getHost(),
				port, Integer.parseInt(conf.getProperties().get("proxy-port")), "/" + "HumanTaskEventNotifierClient");
		return subscribeToAddTasksByAssignedToResource(user, assignedToId, qAssignToId, callbackAddress, this.htClient);
	}

	public GJaxbSubscribeTasksByAssignedToResourceResponse subscribeToAddTasksByAssignedToResource(DWUser user,
			String assignedToId, String qAssignToId, String callbackAddress, Humantask client) throws Exception {
		GJaxbSubscribeTasksByAssignedToResourceResponse response = null;
		assert user != null;

		try {
			LOG.debug("assignedToId = " + assignedToId);
			LOG.debug("qAssignToId = " + qAssignToId);

			QName topic = new QName("http://www.gind.emac.fr/HumanTaskTopic", "allHumanTaskTopic");

			GJaxbSubscribeTasksByAssignedToResource request = new GJaxbSubscribeTasksByAssignedToResource();
			request.getAssignedToIds().add(qAssignToId);
			request.setTopic(topic);
			request.setCallbackAddress(callbackAddress);
			response = client.subscribeTasksByAssignedToResource(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}
}
