/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.ByteArrayInputStream;
import java.net.http.HttpResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.rest.HTTPClientUtil;
import fr.emac.gind.commons.utils.rest.HTTPClientUtil.HTTPProtocol;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.gov.ai.chatbot.client.AIChatbotClient;
import fr.emac.gind.gov.ai_chatbot.AiChatbot;
import fr.emac.gind.gov.ai_chatbot.FaultMessage;
import fr.emac.gind.gov.ai_chatbot.GJaxbExportDocumentsFromRAG;
import fr.emac.gind.gov.ai_chatbot.GJaxbExportDocumentsFromRAGResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbExternalResourceType;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventActualities;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventActualitiesFromPictures;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventActualitiesFromPicturesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventActualitiesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventBuildings;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventBuildingsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventCodeExplanation;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventCodeExplanationResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventFunctions;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventFunctionsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventGoods;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventGoodsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventGroups;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventGroupsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventIots;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventIotsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPersons;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPersonsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPictures;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPicturesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPicturesResponse.PictureOutput;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPositionFromTerritory;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPositionFromTerritoryResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPotential;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPotentialResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPotentialities;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPotentialitiesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventRoads;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventRoadsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventRoles;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventRolesFromObjectives;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventRolesFromObjectivesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventRolesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventSensors;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventSensorsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventTerritories;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventTerritoriesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventTweets;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventTweetsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbLoadDocumentsForRAG;
import fr.emac.gind.gov.ai_chatbot.GJaxbLoadDocumentsForRAGResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbQuery;
import fr.emac.gind.gov.ai_chatbot.GJaxbQueryOnPictures;
import fr.emac.gind.gov.ai_chatbot.GJaxbQueryOnPicturesResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbQueryResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbRestoreRAGFromDirectory;
import fr.emac.gind.gov.ai_chatbot.GJaxbRestoreRAGFromDirectoryResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbSimilarityLevel;
import fr.emac.gind.gov.ai_chatbot.GJaxbSimilarityLevelResponse;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbNode.Geolocation.ItemView.SpecificGeolocation;
import fr.emac.gind.modeler.genericmodel.GJaxbNode.Modeling;
import fr.emac.gind.modeler.genericmodel.GJaxbNode.Modeling.ItemView;
import fr.emac.gind.modeler.genericmodel.GJaxbNode.Modeling.ItemView.SpecificModeling;
import fr.emac.gind.modeler.genericmodel.GJaxbPosition;
import fr.emac.gind.modeler.metamodel.GJaxbModelingViewDefinition;
import fr.emac.gind.modeler.metamodel.GJaxbModelingViewDefinition.View;
import fr.emac.gind.modeler.metamodel.GJaxbNodeMetaGeolocation;
import fr.emac.gind.modeler.metamodel.GJaxbNodeMetaGeolocation.Point;
import fr.emac.gind.modeler.metamodel.GJaxbNodeMetaModeling;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.rio.dw.resources.FileResource;
import fr.emac.gind.tweet.GJaxbMediaObject;
import fr.emac.gind.tweet.GJaxbTweetObject;
import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/generic-application/aichatbot")
@Produces(MediaType.APPLICATION_JSON)
public class AIChatbotResource implements AiChatbot {

	private static Logger LOG = LoggerFactory.getLogger(AIChatbotResource.class.getName());

	private AiChatbot chatbotClient = null;
	private FileResource fileResource = null;

	private String riopiotAddress = null;

	public AIChatbotResource(Configuration conf, FileResource fileResource) throws Exception {
		this.chatbotClient = AIChatbotClient
				.createClient(conf.getProperties().get("governance").replace("/gov", "/GovAIChatbot"));
		this.fileResource = fileResource;

		if (conf.getMapObjects().containsKey("tools_r-iopiot")) {
			this.riopiotAddress = conf.getMapObjects().get("tools_r-iopiot").toString();
		}
	}

	@POST
	@Path("/query")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbQueryResponse query(@Auth DWUser user, GJaxbQuery parameters) throws FaultMessage {
		return this.query(parameters);
	}

	@POST
	@Path("/inventFunctions")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventFunctions(@Auth DWUser user, GJaxbInventFunctions parameters) throws Exception {
		GJaxbInventFunctionsResponse res = this.inventFunctions(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventFunctionsResponse")
				.toString();

	}

	@POST
	@Path("/inventRoles")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventRoles(@Auth DWUser user, GJaxbInventRoles parameters) throws Exception {
		GJaxbInventRolesResponse res = this.inventRoles(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventRolesResponse")
				.toString();

	}

	@POST
	@Path("/inventRolesFromObjectives")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventRolesFromObjectives(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventRolesFromObjectives parameters = JSONJAXBContext.getInstance().unmarshall(
				"{ \"inventRolesFromObjectives\": " + jsonRequest + " }", GJaxbInventRolesFromObjectives.class);
		GJaxbInventRolesFromObjectivesResponse res = this.inventRolesFromObjectives(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res))
				.get("inventRolesFromObjectivesResponse").toString();
	}

	@POST
	@Path("/inventActualities")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventActualities(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventActualities parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventActualities\": " + jsonRequest + " }", GJaxbInventActualities.class);
		GJaxbInventActualitiesResponse res = this.inventActualities(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventActualitiesResponse")
				.toString();
	}

	@POST
	@Path("/inventPotentialities")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventPotentialities(@Auth DWUser user, GJaxbInventPotentialities parameters) throws Exception {
		GJaxbInventPotentialitiesResponse res = this.inventPotentialities(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventPotentialitiesResponse")
				.toString();
	}

	@POST
	@Path("/inventPotential")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventPotential(@Auth DWUser user, GJaxbInventPotential parameters) throws Exception {
		GJaxbInventPotentialResponse res = this.inventPotential(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventPotentialResponse")
				.toString();
	}

	@POST
	@Path("/inventPersons")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventPersons(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventPersons parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventPersons\":" + jsonRequest + "}", GJaxbInventPersons.class);
		GJaxbInventPersonsResponse res = this.inventPersons(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventPersonsResponse")
				.toString();
	}
	
	@POST
	@Path("/inventGoods")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventGoods(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventGoods parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventGoods\":" + jsonRequest + "}", GJaxbInventGoods.class);
		GJaxbInventGoodsResponse res = this.inventGoods(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventGoodsResponse")
				.toString();
	}

	@POST
	@Path("/inventGroups")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventGroups(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventGroups parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventGroups\":" + jsonRequest + "}", GJaxbInventGroups.class);
		GJaxbInventGroupsResponse res = this.inventGroups(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventGroupsResponse")
				.toString();
	}

	@POST
	@Path("/inventTerritories")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventTerritories(@Auth DWUser user, GJaxbInventTerritories parameters) throws Exception {
		GJaxbInventTerritoriesResponse res = this.inventTerritories(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventTerritoriesResponse")
				.toString();
	}

	@POST
	@Path("/inventSensors")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventSensors(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventSensors parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventSensors\":" + jsonRequest + "}", GJaxbInventSensors.class);

		GJaxbInventSensorsResponse res = this.inventSensors(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventSensorsResponse")
				.toString();
	}

	@POST
	@Path("/inventIots")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventIots(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventIots parameters = JSONJAXBContext.getInstance().unmarshall("{ \"inventIots\":" + jsonRequest + "}",
				GJaxbInventIots.class);

		GJaxbInventIotsResponse res = this.inventIots(parameters);

		uploadResources(user, res.getGenericModel(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventIotsResponse")
				.toString();
	}

	@POST
	@Path("/inventPositionFromTerritory")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventPositionFromTerritory(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventPositionFromTerritory parameters = JSONJAXBContext.getInstance().unmarshall(
				"{ \"inventPositionFromTerritory\":" + jsonRequest + "}", GJaxbInventPositionFromTerritory.class);

		GJaxbInventPositionFromTerritoryResponse res = this.inventPositionFromTerritory(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res))
				.get("inventPositionFromTerritoryResponse").toString();
	}

	@POST
	@Path("/inventTweets")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventTweets(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventTweets parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventTweets\":" + jsonRequest + "}", GJaxbInventTweets.class);

		GJaxbInventTweetsResponse res = this.inventTweets(parameters);

		uploadTweetResources(user, res.getTweet(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventTweetsResponse")
				.toString();
	}

	@POST
	@Path("/inventPictures")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventPictures(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventPictures parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventPictures\":" + jsonRequest + "}", GJaxbInventPictures.class);

		GJaxbInventPicturesResponse res = this.inventPictures(parameters);

		uploadPictureResources(user, res.getPictureOutput(), res.getExternalResource());

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("inventPicturesResponse")
				.toString();
	}

	
	@POST
	@Path("/similarityLevel")
	@Consumes(MediaType.APPLICATION_JSON)
	public String similarityLevel(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbSimilarityLevel parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"similarityLevel\":" + jsonRequest + "}", GJaxbSimilarityLevel.class);

		GJaxbSimilarityLevelResponse res = this.similarityLevel(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("similarityLevelResponse")
				.toString();
	}
	
	@POST
	@Path("/loadDocumentsForRAG")
	@Consumes(MediaType.APPLICATION_JSON)
	public String loadDocumentsForRAG(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbLoadDocumentsForRAG parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"loadDocumentsForRAG\":" + jsonRequest + "}", GJaxbLoadDocumentsForRAG.class);

		GJaxbLoadDocumentsForRAGResponse res = this.loadDocumentsForRAG(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("loadDocumentsForRAGResponse")
				.toString();
	}
	
	@POST
	@Path("/exportDocumentsFromRAG")
	@Consumes(MediaType.APPLICATION_JSON)
	public String exportDocumentsFromRAG(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbExportDocumentsFromRAG parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"exportDocumentsFromRAG\":" + jsonRequest + "}", GJaxbExportDocumentsFromRAG.class);

		GJaxbExportDocumentsFromRAGResponse res = this.exportDocumentsFromRAG(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("exportDocumentsFromRAGResponse")
				.toString();
	}
	
	@POST
	@Path("/restoreRAGFromDirectory")
	@Consumes(MediaType.APPLICATION_JSON)
	public String restoreRAGFromDirectory(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbRestoreRAGFromDirectory parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"restoreRAGFromDirectory\":" + jsonRequest + "}", GJaxbRestoreRAGFromDirectory.class);

		GJaxbRestoreRAGFromDirectoryResponse res = this.restoreRAGFromDirectory(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res)).get("restoreRAGFromDirectoryResponse")
				.toString();
	}

	@Override
	public GJaxbQueryResponse query(GJaxbQuery parameters) throws FaultMessage {
		return this.chatbotClient.query(parameters);
	}

	@Override
	public GJaxbInventRolesResponse inventRoles(GJaxbInventRoles parameters) throws FaultMessage {
		return this.chatbotClient.inventRoles(parameters);
	}

	@Override
	public GJaxbInventActualitiesResponse inventActualities(GJaxbInventActualities parameters) throws FaultMessage {
		return this.chatbotClient.inventActualities(parameters);
	}

	@Override
	public GJaxbInventPersonsResponse inventPersons(GJaxbInventPersons parameters) throws FaultMessage {
		return this.chatbotClient.inventPersons(parameters);
	}

	@Override
	public GJaxbInventTerritoriesResponse inventTerritories(GJaxbInventTerritories parameters) throws FaultMessage {
		return this.chatbotClient.inventTerritories(parameters);
	}

	@Override
	public GJaxbSimilarityLevelResponse similarityLevel(GJaxbSimilarityLevel parameters) throws FaultMessage {
		return this.chatbotClient.similarityLevel(parameters);
	}

	@Override
	public GJaxbInventGroupsResponse inventGroups(GJaxbInventGroups parameters) throws FaultMessage {
		return this.chatbotClient.inventGroups(parameters);
	}

	@Override
	public GJaxbInventPotentialitiesResponse inventPotentialities(GJaxbInventPotentialities parameters)
			throws FaultMessage {
		return this.chatbotClient.inventPotentialities(parameters);
	}

	@Override
	public GJaxbInventPotentialResponse inventPotential(GJaxbInventPotential parameters) throws FaultMessage {
		return this.chatbotClient.inventPotential(parameters);
	}

	@Override
	public GJaxbInventFunctionsResponse inventFunctions(GJaxbInventFunctions parameters) throws FaultMessage {
		return this.chatbotClient.inventFunctions(parameters);
	}

	@Override
	public GJaxbInventSensorsResponse inventSensors(GJaxbInventSensors parameters) throws FaultMessage {
		return this.chatbotClient.inventSensors(parameters);
	}

	@Override
	public GJaxbInventActualitiesFromPicturesResponse inventActualitiesFromPictures(
			GJaxbInventActualitiesFromPictures parameters) throws FaultMessage {
		return this.chatbotClient.inventActualitiesFromPictures(parameters);
	}

	@Override
	public GJaxbQueryOnPicturesResponse queryOnPictures(GJaxbQueryOnPictures parameters) throws FaultMessage {
		return this.chatbotClient.queryOnPictures(parameters);
	}

	@Override
	public GJaxbInventPositionFromTerritoryResponse inventPositionFromTerritory(
			GJaxbInventPositionFromTerritory parameters) throws FaultMessage {
		return this.chatbotClient.inventPositionFromTerritory(parameters);
	}

	@Override
	public GJaxbInventTweetsResponse inventTweets(GJaxbInventTweets parameters) throws FaultMessage {
		return this.chatbotClient.inventTweets(parameters);
	}

	@Override
	public GJaxbInventPicturesResponse inventPictures(GJaxbInventPictures parameters) throws FaultMessage {
		return this.chatbotClient.inventPictures(parameters);
	}

	@Override
	public GJaxbInventIotsResponse inventIots(GJaxbInventIots parameters) throws FaultMessage {
		return this.chatbotClient.inventIots(parameters);
	}

	@Override
	public GJaxbInventRolesFromObjectivesResponse inventRolesFromObjectives(GJaxbInventRolesFromObjectives parameters)
			throws FaultMessage {
		return this.chatbotClient.inventRolesFromObjectives(parameters);
	}

	private void uploadPictureResources(DWUser user, List<PictureOutput> pictureOutputs,
			List<GJaxbExternalResourceType> rscs) throws Exception {
		for (GJaxbExternalResourceType extRsc : rscs) {
			for (PictureOutput pictureOutput : pictureOutputs) {

				if (extRsc.getId().equals(pictureOutput.getPictureId())) {

					String fileName = pictureOutput.getPictureId() + extRsc.getContentType().replace("image/", ".");

					JSONObject data = new JSONObject();
					data.put("nodeId", pictureOutput.getPictureId());
					data.put("category", "pictures");
					data.put("collaborationName", user.getCurrentCollaborationName());
					data.put("knowledgeSpaceName", user.getCurrentKnowledgeSpaceName());
					data.put("fileName", fileName);

					String url = this.fileResource.upload(user, new ByteArrayInputStream(extRsc.getData()),
							FormDataContentDisposition.name(fileName).build(), data.toString());

					pictureOutput.setPictureUrl(url);

				}
			}
		}
		rscs.clear();
	}

	private void uploadTweetResources(DWUser user, List<GJaxbTweetObject> tweets, List<GJaxbExternalResourceType> rscs)
			throws Exception {

		List<PictureOutput> pictureOutputs = new ArrayList<PictureOutput>();
		for (GJaxbTweetObject tweet : tweets) {
			for (GJaxbMediaObject media : tweet.getExtendedEntities().getMedia()) {
				PictureOutput po = new PictureOutput();
				po.setPictureId(media.getIdStr());
				po.setPictureUrl(media.getMediaUrl());
				pictureOutputs.add(po);
			}
		}

		this.uploadPictureResources(user, pictureOutputs, rscs);

		for (PictureOutput po : pictureOutputs) {
			for (GJaxbTweetObject tweet : tweets) {
				for (GJaxbMediaObject media : tweet.getExtendedEntities().getMedia()) {
					if (po.getPictureId().equals(media.getIdStr())) {
						media.setMediaUrl(po.getPictureUrl());
					}
				}
			}
		}

		rscs.clear();
	}

	private void uploadResources(DWUser user, GJaxbGenericModel model, List<GJaxbExternalResourceType> rscs)
			throws Exception {
		for (GJaxbExternalResourceType extRsc : rscs) {

			String nodeId = extRsc.getId().split("::")[0];
			String propertyName = extRsc.getId().split("::")[1];
			GJaxbNode node = GenericModelHelper.findNodeById(nodeId, model.getNode());

			String instanceName = GenericModelHelper.findProperty("name", node.getProperty()).getValue();
			String fileName = "photo_" + UUID.randomUUID().toString()
					+ RegExpHelper.toRegexFriendlyName(instanceName).hashCode()
					+ extRsc.getContentType().replace("image/", ".");

			JSONObject data = new JSONObject();
			data.put("nodeId", nodeId);
			data.put("category", "node");
			data.put("collaborationName", user.getCurrentCollaborationName());
			data.put("knowledgeSpaceName", user.getCurrentKnowledgeSpaceName());
			data.put("fileName", fileName);

			String url = this.fileResource.upload(user, new ByteArrayInputStream(extRsc.getData()),
					FormDataContentDisposition.name(fileName).build(), data.toString());

			if(!node.isSetModeling()) {
				node.setModeling(new Modeling());
			}
			if(node.getModeling().getItemView().isEmpty()) {
				ItemView iv = new ItemView();
				iv.setViewId("view_" + UUID.randomUUID().toString());
				iv.setPosition(new GJaxbPosition());
				iv.setPackageName("UNDEFINED");
				node.getModeling().getItemView().add(iv);
			}
			
			if (node.isSetModeling()) {
				if (node.getModeling().isSetItemView()) {
					node.getModeling().getItemView().forEach(iv -> {
						if (iv != null) {
							if (!iv.isSetSpecificModeling()) {
								iv.setSpecificModeling(new SpecificModeling());
							}
							if (!iv.getSpecificModeling().isSetNodeMetaModeling()) {
								iv.getSpecificModeling().setNodeMetaModeling(new GJaxbNodeMetaModeling());
							}
							if (!iv.getSpecificModeling().getNodeMetaModeling().isSetModelingViewDefinition()) {
								iv.getSpecificModeling().getNodeMetaModeling()
										.setModelingViewDefinition(new GJaxbModelingViewDefinition());
							}
							if (!iv.getSpecificModeling().getNodeMetaModeling().getModelingViewDefinition()
									.isSetView()) {
								iv.getSpecificModeling().getNodeMetaModeling().getModelingViewDefinition()
										.setView(new View());
							}
							iv.getSpecificModeling().getNodeMetaModeling().getModelingViewDefinition().getView()
									.setUrl(url);
						}
					});
				}
			}

			if (node.isSetGeolocation()) {
				if (node.getGeolocation().isSetItemView()) {
					node.getGeolocation().getItemView().forEach(iv -> {
						if (iv != null && iv.getPoint() != null) {
							if (!iv.isSetSpecificGeolocation()) {
								iv.setSpecificGeolocation(new SpecificGeolocation());
							}
							if (!iv.getSpecificGeolocation().isSetNodeMetaGeolocation()) {
								iv.getSpecificGeolocation().setNodeMetaGeolocation(new GJaxbNodeMetaGeolocation());
							}
							if (!iv.getSpecificGeolocation().getNodeMetaGeolocation().isSetPoint()) {
								iv.getSpecificGeolocation().getNodeMetaGeolocation().setPoint(new Point());
							}
							iv.getSpecificGeolocation().getNodeMetaGeolocation().getPoint().setImage(url);
						}
					});
				}
			}
		}
		rscs.clear();
	}

	@GET
	@Path("/getRiopiotContext")
	@Consumes(MediaType.APPLICATION_JSON)
	public String getRiopiotContext(@Auth DWUser user) throws Exception {
		String riopiotConfBuffer = null;
		try {
			HttpResponse<String> response = HTTPClientUtil.sendHTTPRequest(HTTPProtocol.GET,
					this.riopiotAddress + "/api/context/applicationContext", null, null, null, null);

			riopiotConfBuffer = response.body();
			System.out.println("riopiotConfBuffer = " + riopiotConfBuffer);
		} catch (Throwable e) {
			e.printStackTrace();
			LOG.warn("RIOPIOT IS NOT CONNECTED !!!");
			return null;
		}
		return riopiotConfBuffer;
	}

	@POST
	@Path("/inventCodeExplanation")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventCodeExplanation(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventCodeExplanation parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventCodeExplanation\":" + jsonRequest + "}", GJaxbInventCodeExplanation.class);

		GJaxbInventCodeExplanationResponse res = this.inventCodeExplanation(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res))
				.get("inventCodeExplanationResponse").toString();
	}

	@Override
	public GJaxbInventCodeExplanationResponse inventCodeExplanation(GJaxbInventCodeExplanation parameters)
			throws FaultMessage {
		return this.chatbotClient.inventCodeExplanation(parameters);
	}
	
	@POST
	@Path("/inventBuildings")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventBuildings(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventBuildings parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventBuildings\":" + jsonRequest + "}", GJaxbInventBuildings.class);

		GJaxbInventBuildingsResponse res = this.inventBuildings(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res))
				.get("inventBuildingsResponse").toString();
	}

	@Override
	public GJaxbInventBuildingsResponse inventBuildings(GJaxbInventBuildings parameters) throws FaultMessage {
		return this.chatbotClient.inventBuildings(parameters);
	}
	
	@POST
	@Path("/inventRoads")
	@Consumes(MediaType.APPLICATION_JSON)
	public String inventRoads(@Auth DWUser user, String jsonRequest) throws Exception {
		GJaxbInventRoads parameters = JSONJAXBContext.getInstance()
				.unmarshall("{ \"inventRoads\":" + jsonRequest + "}", GJaxbInventRoads.class);

		GJaxbInventRoadsResponse res = this.inventRoads(parameters);

		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res))
				.get("inventBuildingsResponse").toString();
	}

	@Override
	public GJaxbInventRoadsResponse inventRoads(GJaxbInventRoads parameters) throws FaultMessage {
		return this.chatbotClient.inventRoads(parameters);
	}

	@Override
	public GJaxbInventGoodsResponse inventGoods(GJaxbInventGoods parameters) throws FaultMessage {
		return this.chatbotClient.inventGoods(parameters);
	}

	@Override
	public GJaxbLoadDocumentsForRAGResponse loadDocumentsForRAG(GJaxbLoadDocumentsForRAG parameters) throws FaultMessage {
		return this.chatbotClient.loadDocumentsForRAG(parameters);
	}

	@Override
	public GJaxbExportDocumentsFromRAGResponse exportDocumentsFromRAG(GJaxbExportDocumentsFromRAG parameters) throws FaultMessage {
		return this.chatbotClient.exportDocumentsFromRAG(parameters);
	}

	@Override
	public GJaxbRestoreRAGFromDirectoryResponse restoreRAGFromDirectory(GJaxbRestoreRAGFromDirectory parameters) throws FaultMessage {
		return this.chatbotClient.restoreRAGFromDirectory(parameters);
	}

}