/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.util.List;

import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataParam;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.emac.gind.campaign.manager.CampaignManagerItf;
import fr.emac.gind.campaign.manager.client.CampaignManagerClient;
import fr.emac.gind.campaign.manager.client.data.GJaxbAddSolution;
import fr.emac.gind.campaign.manager.client.data.GJaxbCreateCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbCreateCampaignResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbDeleteCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbDeleteCampaignResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbFindSolutionById;
import fr.emac.gind.campaign.manager.client.data.GJaxbFindSolutionByIdResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbFindSolutionsByMetrics;
import fr.emac.gind.campaign.manager.client.data.GJaxbFindSolutionsByMetricsResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetAllCampaignsWithoutResources;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetAllCampaignsWithoutResourcesResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetCampaignResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetCampaignsWithoutResourcesByUser;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetCampaignsWithoutResourcesByUserResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetFailureSolutions;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetFailureSolutionsResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetSolutionsFromCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetSolutionsFromCampaignResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbUpdateCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbUpdateCampaignResponse;
import fr.emac.gind.campaign.manager.data.model.GJaxbCampaign;
import fr.emac.gind.campaign.manager.data.model.GJaxbCampaignSolution;
import fr.emac.gind.campaign.manager.data.model.GJaxbGlobalCampaign;
import fr.emac.gind.campaign.manager.data.model.GJaxbStatusType;
import fr.emac.gind.commons.utils.io.FileUtil;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace;
import fr.emac.gind.workflow.deduction.AbstractDeductionStrategy;
import fr.emac.gind.workflow.deduction.ProcessDeductionHelper;

/*
 * #%L
 * dw-users
 * %%
 * Copyright (C) 2014 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/r-iosepe/campaign")
@Produces(MediaType.APPLICATION_JSON)
public class CampaignResource {

	private static Logger LOG = LoggerFactory.getLogger(CampaignResource.class.getName());

	private CampaignManagerItf campaignClient = null;
	private CoreResource core = null;

	public CampaignResource(Configuration conf, CoreResource core) throws Exception {
		this.campaignClient = CampaignManagerClient.createClient((String) conf.getProperties().get("campaign-manager"));
		this.core = core;
	}

	@POST
	@Path("/publishCampaign")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Produces(MediaType.APPLICATION_JSON)
	public String publishCampaign(@Auth DWUser user, @FormDataParam("file") InputStream is,
			@FormDataParam("file") FormDataContentDisposition fileDetail, @QueryParam("data") String data)
			throws Exception {
		String json = null;
		try {

			String jsonFile = FileUtil.getContents(new BufferedInputStream(is));
			GJaxbGlobalCampaign campaign = JSONJAXBContext.getInstance().unmarshall(jsonFile,
					GJaxbGlobalCampaign.class);
			campaign.getCampaign().setCampaignId(null);

			// verify if not an restart on failure
			if (campaign.getCampaign().getStatus().equals(GJaxbStatusType.RUNNING)) {
				// improve consistency
				campaign.getCampaign().setNumberOfSolutionAchieved(campaign.getCampaignSolution().size());
				campaign.getCampaign().setNumberOfFailureSolution((int) campaign.getCampaignSolution().stream().filter(sc -> {
					return !sc.isSuccess();
				}).count());
			}

			GJaxbCreateCampaign request = new GJaxbCreateCampaign();
			request.setCampaign(campaign.getCampaign());
			request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
			request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			GJaxbCreateCampaignResponse response = this.campaignClient.createCampaign(request);

			for (GJaxbCampaignSolution campaignSolution : campaign.getCampaignSolution()) {
				campaignSolution.setCampaignId(response.getCampaignId());

				GJaxbAddSolution addSc = new GJaxbAddSolution();
				addSc.setCampaignSolution(campaignSolution);
				addSc.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
				addSc.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
				addSc.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());

				this.campaignClient.addSolution(addSc);
			}

			GJaxbGetCampaign requestGet = new GJaxbGetCampaign();
			requestGet.setCampaignId(response.getCampaignId());
			GJaxbGetCampaignResponse responseGet = this.campaignClient.getCampaign(requestGet);

			json = JSONJAXBContext.getInstance().marshallAnyElement(responseGet);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return json;
	}

	@GET
	@Path("/campaigns/{id}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String getCampaignString(@Auth DWUser user, @PathParam("id") String id, @QueryParam("qId") String qId)
			throws Exception {
		GJaxbCampaign jsonCampaign = this.getCampaign(user, id, qId);
		if (jsonCampaign != null) {
			return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(jsonCampaign))
					.getJSONObject("campaign").toString();
		}
		return null;
	}

	public GJaxbCampaign getCampaign(@Auth DWUser user, @PathParam("id") String id, @QueryParam("qId") String qId)
			throws Exception {
		GJaxbCampaign jsonCampaign = null;
		try {
			GJaxbGetCampaign requestGet = new GJaxbGetCampaign();
			requestGet.setCampaignId(id);
			GJaxbGetCampaignResponse responseGet = this.campaignClient.getCampaign(requestGet);

			jsonCampaign = responseGet.getCampaign();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}

		return jsonCampaign;
	}

	@GET
	@Path("/campaignSolutionsFromCampaign/{id}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public List<GJaxbCampaignSolution> getCampaignSolutionsFromCampaign(@Auth DWUser user, @PathParam("id") String id,
			@QueryParam("qId") String qId) throws Exception {
		List<GJaxbCampaignSolution> campaignSolutions = null;
		try {

			GJaxbGetSolutionsFromCampaign parameters = new GJaxbGetSolutionsFromCampaign();
			parameters.setCampaignId(id);
			GJaxbGetSolutionsFromCampaignResponse responseGet = this.campaignClient
					.getSolutionsFromCampaign(parameters);
			campaignSolutions = responseGet.getCampaignSolution();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return campaignSolutions;
	}

	@POST
	@Path("/getFailureSolutions")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public List<GJaxbCampaignSolution> getFailureSolutions(@Auth DWUser user, String campaignId) throws Exception {
		GJaxbGetFailureSolutionsResponse responseGet = null;
		try {

			GJaxbGetFailureSolutions requestGet = new GJaxbGetFailureSolutions();
			requestGet.setCampaignId(campaignId);
			responseGet = this.campaignClient.getFailureSolutions(requestGet);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return responseGet.getCampaignSolution();
	}

	@POST
	@Path("/getAllCampaigns")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String getAllCampaigns(@Auth DWUser user) throws Exception {
		assert user != null;
		String res = null;
		try {
			GJaxbGetAllCampaignsWithoutResources request = new GJaxbGetAllCampaignsWithoutResources();
			GJaxbGetAllCampaignsWithoutResourcesResponse response = this.campaignClient
					.getAllCampaignsWithoutResources(request);

			res = JSONJAXBContext.getInstance().marshallAnyElement(response);
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return res;
	}

	@GET
	@Path("/getCampaignsByUser")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String getCampaignsByUser(@Auth DWUser user) throws Exception {
		assert user != null;
		List<GJaxbCampaign> res = null;
		try {
			if (user.isAdmin()) {
				GJaxbGetAllCampaignsWithoutResources request = new GJaxbGetAllCampaignsWithoutResources();
				GJaxbGetAllCampaignsWithoutResourcesResponse response = this.campaignClient
						.getAllCampaignsWithoutResources(request);
				res = response.getCampaign();
			} else {
				GJaxbGetCampaignsWithoutResourcesByUser request = new GJaxbGetCampaignsWithoutResourcesByUser();
				request.setUserId(user.getUserId());
				GJaxbGetCampaignsWithoutResourcesByUserResponse response = this.campaignClient
						.getCampaignsWithoutResourcesByUser(request);
				res = response.getCampaign();
			}

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}

		JSONArray array = new JSONArray();
		res.forEach(c -> {
			try {
				array.put(
						new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(c)).getJSONObject("campaign"));
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException(e.getMessage(), e);
			}
		});

		return array.toString();
	}

	@GET
	@Path("/campaigns/{campaignId}/{campaignSolutionId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String findSolutionByIdString(@Auth DWUser user, @PathParam("campaignId") String campaignid,
			@PathParam("campaignSolutionId") String campaignSolutionId) throws Exception {
		System.out.println("*********************** campaignid: " + campaignid);
		System.out.println("*********************** campaignSolutionId: " + campaignSolutionId);
		GJaxbCampaignSolution campaignSolution = this.findSolutionById(user, campaignid, campaignSolutionId);
		if (campaignSolution != null) {
			System.out.println("*********************** : " + campaignSolution);
			return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(campaignSolution)).getJSONObject("campaignSolution")
					.toString();
		}
		return null;
	}

	public GJaxbCampaignSolution findSolutionById(@Auth DWUser user, @PathParam("campaignId") String campaignid,
			@PathParam("campaignSolutionId") String campaignSolutionId) throws Exception {
		GJaxbCampaignSolution campaignSolution = null;
		try {
			GJaxbFindSolutionById requestGet = new GJaxbFindSolutionById();
			requestGet.setCampaignId(campaignid);
			requestGet.setCampaignSolutionId(campaignSolutionId);
			GJaxbFindSolutionByIdResponse responseGet = this.campaignClient.findSolutionById(requestGet);
			campaignSolution = responseGet.getCampaignSolution();

			// String outputBuffer = campaignSolution.getOutputMessage().replace("<![CDATA[",
			// "").replace("]]>", "");
			// Document doc = DOMUtil.getInstance().parse(new
			// ByteArrayInputStream(outputBuffer.getBytes()));
			// GJaxbRunSyncResponse resp =
			// XMLJAXBContext.getInstance().unmarshallDocument(doc,
			// GJaxbRunSyncResponse.class);
			//
			//
			// LOG.debug("output: \n" + XMLPrettyPrinter.print(doc));
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return campaignSolution;
	}

//	@SuppressWarnings("unused")
//	@POST
//	@Path("/findSolutionsByMetrics")
//	@Consumes(MediaType.APPLICATION_JSON)
//	@Produces(MediaType.APPLICATION_JSON)
//	public String findSolutionsByMetricsString(@Auth DWUser user, GJaxbFindSolutionsByMetrics request) throws Exception {
//		List<GJaxbCampaignSolution> campaignSolutions = this.findSolutionsByMetrics(user, request);
//
//		JSONArray array = new JSONArray();
//		campaignSolutions.forEach(c -> {
//			try {
//				array.put(
//						new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(c)).getJSONObject("campaignSolution"));
//			} catch (Exception e) {
//				e.printStackTrace();
//				throw new RuntimeException(e.getMessage(), e);
//			}
//		});
//
//		return array.toString();
//
//	}

	@POST
	@Path("/findSolutionsByMetrics")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbFindSolutionsByMetricsResponse findSolutionsByMetrics(@Auth DWUser user, GJaxbFindSolutionsByMetrics request)
			throws Exception {
		GJaxbFindSolutionsByMetricsResponse responseFindSolutions = null;
		try {
			responseFindSolutions = this.campaignClient
					.findSolutionsByMetrics(request);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return responseFindSolutions;
	}


	@POST
	@Path("/createCampaign")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String createCampaign(@Auth DWUser user, String campaignName) throws Exception {
		String jsonCampaign = null;
		try {

			GJaxbNode project = ProcessDeductionHelper.getInstance().findProject(user.getCurrentCollaborationName(),
					user.getCurrentKnowledgeSpaceName(), core.getCoreClient());

			GJaxbCampaign campaign = AbstractDeductionStrategy.createCampaignIfNotExist(campaignClient, campaignName,
					project, 0, user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());

			GJaxbCreateCampaignResponse response = new GJaxbCreateCampaignResponse();
			response.setCampaignId(campaign.getCampaignId());

			jsonCampaign = JSONJAXBContext.getInstance().marshallAnyElement(response);
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return jsonCampaign;
	}
	
	@POST
	@Path("/updateCampaign")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String updateCampaign(@Auth DWUser user, GJaxbCampaign campaign) throws Exception {
		String jsonCampaign = null;
		try {
			
			GJaxbUpdateCampaign upReq = new GJaxbUpdateCampaign();
			upReq.setCampaign(campaign);
			upReq.setCampaignId(campaign.getCampaignId());
			upReq.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			upReq.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
			upReq.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			GJaxbUpdateCampaignResponse resp = this.campaignClient.updateCampaign(upReq);

			jsonCampaign = JSONJAXBContext.getInstance().marshallAnyElement(resp);
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return jsonCampaign;
	}

	@POST
	@Path("/deleteCampaign")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String deleteCampaign(@Auth DWUser user, String campaignId) throws Exception {
		String jsonCampaign = null;
		try {

			GJaxbDeleteCampaign request = new GJaxbDeleteCampaign();
			request.setCampaignId(campaignId);
			GJaxbProperty param = new GJaxbProperty();
			param.setName("runner");
			param.setValue("PRIO");
			request.getProperty().add(param);
			GJaxbDeleteCampaignResponse response = this.campaignClient.deleteCampaign(request);

			jsonCampaign = JSONJAXBContext.getInstance().marshallAnyElement(response);
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return jsonCampaign;
	}

}
