/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.gov.core.client.CoreGovClient;
import fr.emac.gind.gov.core_gov.CoreGov;
import fr.emac.gind.gov.core_gov.GJaxbAddEdge;
import fr.emac.gind.gov.core_gov.GJaxbAddEdgeResponse;
import fr.emac.gind.gov.core_gov.GJaxbAddNode;
import fr.emac.gind.gov.core_gov.GJaxbAddNodeResponse;
import fr.emac.gind.gov.core_gov.GJaxbGetEdge;
import fr.emac.gind.gov.core_gov.GJaxbGetEdgeResponse;
import fr.emac.gind.gov.core_gov.GJaxbGetNode;
import fr.emac.gind.gov.core_gov.GJaxbGetNodeResponse;
import fr.emac.gind.gov.core_gov.GJaxbQuery;
import fr.emac.gind.gov.core_gov.GJaxbQueryResponse;
import fr.emac.gind.gov.core_gov.GJaxbRemoveEdge;
import fr.emac.gind.gov.core_gov.GJaxbRemoveEdgeResponse;
import fr.emac.gind.gov.core_gov.GJaxbRemoveNode;
import fr.emac.gind.gov.core_gov.GJaxbRemoveNodeResponse;
import fr.emac.gind.gov.core_gov.GJaxbUpdateEdge;
import fr.emac.gind.gov.core_gov.GJaxbUpdateEdgeResponse;
import fr.emac.gind.gov.core_gov.GJaxbUpdateNode;
import fr.emac.gind.gov.core_gov.GJaxbUpdateNodeResponse;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.rio.dw.resources.gov.bo.FindNodesParameters;
import fr.emac.gind.rio.dw.resources.gov.bo.FindParameters;
import fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace;
import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/generic-application/core")
//@Api("/core")
@Produces(MediaType.APPLICATION_JSON)
public class CoreResource {

	private static Logger LOG = LoggerFactory.getLogger(CoreResource.class.getName());

	private CoreGov coreClient = null;

	public CoreResource(Configuration conf) throws Exception {
		this.coreClient = CoreGovClient
				.createClient(conf.getProperties().get("governance").replace("/gov", "/GovCore"));
	}

	@POST
	@Path("/nodes")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbNode addNode(@Auth DWUser user, GJaxbNode node) throws Exception {
		GJaxbAddNodeResponse response = null;
		// assert user != null;
		try {

			GJaxbAddNode request = new GJaxbAddNode();
			if (!node.isSetId()) {
				node.setId("node_" + UUID.randomUUID().toString());
			}
			request.setNode(node);
			request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			if (user != null) {
				request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
				request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			}
			if (node.getType().equals(new QName("http://fr.emac.gind/system", "User"))) {
				request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			}
			response = this.coreClient.addNode(request);
			node.setId(response.getId());

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return node;
	}

	@PUT
	@Path("/nodes")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbNode updateNode(@Auth DWUser user, GJaxbNode node) throws Exception {
		GJaxbUpdateNodeResponse response = null;
		try {

			GJaxbUpdateNode request = new GJaxbUpdateNode();
			request.setNode(node);
			request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			if (user != null) {
				request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
				request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			}
			request.setCreateIfNotExist(true);

			if (node.getType().equals(new QName("http://fr.emac.gind/system", "User"))) {
				request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
				request.setCreateIfNotExist(false);
			}
			response = this.coreClient.updateNode(request);
			node.setId(response.getId());

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return node;
	}

	@POST
	@Path("/nodes/getNode")
	public GJaxbGetNodeResponse getNode(@Auth DWUser user, GJaxbGetNode request) throws Exception {
		GJaxbGetNodeResponse response = null;

		try {
			LOG.debug("id = " + request.getId());

			if (user != null) {
				request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
				request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
				request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			}
			response = this.coreClient.getNode(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/nodes/findNodesByTypesAndOrProperties")
	public List<GJaxbNode> findNodesByTypesAndOrPropertiesRest(@Auth DWUser user, FindNodesParameters params)
			throws Exception {
		return this.findNodesByTypesAndOrProperties(user, params.getTypes(), params.getProperties(), params.getStatus(),
				params.getCollaborationName(), params.getKnowledgeSpaceName());
	}

	public List<GJaxbNode> findNodesByTypesAndOrProperties(@Auth DWUser user, @QueryParam("types") List<QName> types,
			@QueryParam("properties") List<FindParameters> properties,
			@QueryParam("node_status") List<String> node_status, String collaborationName, String knowlegeSpaceName)
			throws Exception {
		List<GJaxbNode> nodes = new ArrayList<GJaxbNode>();
		try {

			if (user != null && !user.isAdmin() && user.getCurrentCollaborationName() == null) {
				throw new Exception("CollaborationName cannot be null for a user");
			}

			String queryMsg = "";

			if (types != null && types.size() > 0) {
				queryMsg = queryMsg + "(";
				for (QName type : types) {
					String queryType = "n:`" + type.toString() + "`";
					queryMsg = queryMsg + queryType + " OR ";
				}
				;
				queryMsg = queryMsg.substring(0, queryMsg.length() - " OR ".length());
				queryMsg = queryMsg + ")";
			}

			String queryProperties = "";
			if (properties != null && !properties.isEmpty()) {
				for (FindParameters prop : properties) {
					String operator = " = ";
					String rightExp = "'" + prop.getValue() + "'";
					if (prop.getOperator() == null || "equals".equals(prop.getOperator())) {
						operator = " = ";
						rightExp = "'" + prop.getValue() + "'";
					} else if (prop.getOperator() != null && "contains".equals(prop.getOperator())) {
						operator = " =~ ";
						rightExp = "'.*" + prop.getValue() + ".*'";
					}
					queryProperties = queryProperties + "n.`property_" + prop.getName() + "`" + operator + rightExp
							+ " AND ";
				}
				queryProperties = queryProperties.substring(0, queryProperties.length() - " AND ".length());
			}

			if (!queryProperties.isEmpty()) {
				if (!queryMsg.isEmpty()) {
					queryMsg = queryMsg + " AND " + "(" + queryProperties + ")";
				} else {
					queryMsg = queryMsg + "(" + queryProperties + ")";
				}
			}

			String queryNodeStatus = "";
			if (node_status != null && !node_status.isEmpty()) {
				queryNodeStatus = "n:";
				boolean first = true;
				for (String status : node_status) {
					if (first) {
						queryNodeStatus = queryNodeStatus + "`" + status + "`";
						first = false;
					} else {
						queryNodeStatus = queryNodeStatus + "|`" + status + "`";
					}
				}
			}
			if (!queryNodeStatus.isEmpty()) {
				if (!queryMsg.isEmpty()) {
					queryMsg = queryMsg + " AND " + "(" + queryNodeStatus + ")";
				} else {
					queryMsg = queryMsg + "(" + queryNodeStatus + ")";
				}
			}

			GJaxbQuery query = new GJaxbQuery();

			if (collaborationName != null && !collaborationName.isBlank() && knowlegeSpaceName != null
					&& !knowlegeSpaceName.isBlank()) {
				String match = "n" + ":`" + RegExpHelper.toRegexFriendlyName(collaborationName) + "`" + "&`"
						+ RegExpHelper.toRegexFriendlyName(knowlegeSpaceName) + "`";

				query.setQuery("match (" + match + ") where " + queryMsg + " return n; ");
			} else if (user != null) {
				String match = "n";
				query.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
				if (user.getCurrentCollaborationName() != null) {
					query.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
					match = match + ":`" + RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName()) + "`";
				}
				if (user.getCurrentKnowledgeSpaceName() != null) {
					query.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
					match = match + "&`" + RegExpHelper.toRegexFriendlyName(user.getCurrentKnowledgeSpaceName()) + "`";
				}

				query.setQuery("match (" + match + ") where " + queryMsg + " return n; ");
			} else {
				query.setQuery("match (n) where " + queryMsg + " return n; ");
			}

			GJaxbQueryResponse queryResponse = this.coreClient.query(query);
			if (queryResponse != null && queryResponse.getSingle() != null
					&& queryResponse.getSingle().getGenericModel() != null) {
				nodes.addAll(queryResponse.getSingle().getGenericModel().getNode());
			}
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}

		return nodes;
	}

	@DELETE
	@Path("/nodes/{id}")
	public GJaxbRemoveNodeResponse deleteNode(@Auth DWUser user, @PathParam("id") String id,
			@QueryParam("qid") String qid) throws Exception {
		GJaxbRemoveNodeResponse response = null;
		assert user != null;

		try {
			LOG.debug("id = " + id);
			LOG.debug("qId = " + qid);
			GJaxbRemoveNode request = new GJaxbRemoveNode();
			if (qid != null) {
				request.setId(qid);
			} else {
				request.setId(id);
			}
			request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
			request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			response = this.coreClient.removeNode(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	public GJaxbAddEdgeResponse addEdge(DWUser user, GJaxbEdge edge, String sourceId, String targetId)
			throws Exception {
		GJaxbAddEdgeResponse response = null;
		// assert user != null;
		try {

			GJaxbAddEdge request = new GJaxbAddEdge();
			request.setEdge(edge);
			request.setSourceNodeId(sourceId);
			request.setTargetNodeId(targetId);

			request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			if (user != null) {
				request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
				request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			}
			response = this.coreClient.addEdge(request);
			edge.setId(response.getId());

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/edges")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbEdge addEdge(@Auth DWUser user, GJaxbEdge edge) throws Exception {
		GJaxbAddEdgeResponse response = this.addEdge(user, edge, edge.getSource().getId(), edge.getTarget().getId());
		return edge;
	}

	@PUT
	@Path("/edges")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbEdge updateEdge(@Auth DWUser user, GJaxbUpdateEdge requestEdge) throws Exception {
		GJaxbUpdateEdgeResponse response = null;
		assert user != null;
		try {

			response = this.coreClient.updateEdge(requestEdge);
			requestEdge.getEdge().setId(response.getId());

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return requestEdge.getEdge();
	}

	@POST
	@Path("/edges/getEdge")
	public GJaxbGetEdgeResponse getEdge(@Auth DWUser user, GJaxbGetEdge request) throws Exception {
		GJaxbGetEdgeResponse response = null;
		assert user != null;

		try {
			LOG.debug("id = " + request.getId());

			if (user != null) {
				request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
				request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
				request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			}
			response = this.coreClient.getEdge(request);

			response.getEdge().setSource(null);
			response.getEdge().setTarget(null);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@DELETE
	@Path("/edges/{id}")
	public GJaxbRemoveEdgeResponse deleteEdge(@Auth DWUser user, @PathParam("id") String id,
			@QueryParam("qid") String qid) throws Exception {
		GJaxbRemoveEdgeResponse response = null;
		assert user != null;

		try {
			LOG.debug("id = " + id);
			LOG.debug("qId = " + qid);
			GJaxbRemoveEdge request = new GJaxbRemoveEdge();
			if (qid != null) {
				request.setId(qid);
			} else {
				request.setId(id);
			}
			request.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			request.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
			request.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			response = this.coreClient.removeEdge(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	public CoreGov getCoreClient() {
		return coreClient;
	}

}