/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.File;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.codec.binary.StringUtils;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataParam;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import fr.emac.gind.commons.utils.RIOConstant;
import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.event.interpretation.config.InterpretationConfigDeployHelper;
import fr.emac.gind.event.interpretation.config.InterpretationConfigsManagerCommandClient;
import fr.emac.gind.event.interpretation.predict.PredictCommandClient;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.interpretationconfigs.InterpretationCommand;
import fr.emac.gind.interpretationconfigs.PredictCommand;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.model.interpretation.config.GJaxbContext;
import fr.emac.gind.model.interpretation.config.GJaxbDeploy;
import fr.emac.gind.model.interpretation.config.GJaxbDeployResponse;
import fr.emac.gind.model.interpretation.config.GJaxbDeployResult;
import fr.emac.gind.model.interpretation.config.GJaxbGetInterpretationConfig;
import fr.emac.gind.model.interpretation.config.GJaxbGetInterpretationConfigResponse;
import fr.emac.gind.model.interpretation.config.GJaxbGetInterpretationConfigs;
import fr.emac.gind.model.interpretation.config.GJaxbGetInterpretationConfigsResponse;
import fr.emac.gind.model.interpretation.config.GJaxbInterpretationConfig;
import fr.emac.gind.model.interpretation.config.GJaxbInterpretationConfigs;
import fr.emac.gind.model.interpretation.config.GJaxbPredict;
import fr.emac.gind.model.interpretation.config.GJaxbPredictResponse;
import fr.emac.gind.model.interpretation.config.GJaxbSubscriptionRequiredType;
import fr.emac.gind.model.interpretation.config.GJaxbUndeploy;
import fr.emac.gind.model.interpretation.config.GJaxbUndeployResponse;
import fr.emac.gind.rio.dw.resources.gov.bo.InterpretationConfigByUrl;

/*
 * #%L
 * dw-users
 * %%
 * Copyright (C) 2014 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/r-ioga/interpretation_rules")
//@Api("interpretation_rules")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class InterpretationRulesResource {

	private static Logger LOG = LoggerFactory.getLogger(InterpretationRulesResource.class.getName());

	private InterpretationCommand interpretationClient = null;
	private PredictCommand predictClient = null;

	private String eventProducerAgentBrokerAddress = null;
	private boolean isAlreadyRegistering = false;
	private Configuration conf = null;

	public InterpretationRulesResource(Configuration conf) throws Exception {
		this.conf = conf;

		if (conf.getProperties().get("interpretation-engine") != null) {
			this.interpretationClient = InterpretationConfigsManagerCommandClient.createClient(conf.getProperties()
					.get("interpretation-engine").replace("/InterpretationEngine", "/LowLevelInterpretationManager"));
			this.predictClient = PredictCommandClient.createClient(conf.getProperties().get("interpretation-engine")
					.replace("/InterpretationEngine", "/LowLevelInterpretationManager"));

			this.eventProducerAgentBrokerAddress = conf.getProperties().get("event-broker");
		}

	}

	@POST
	@Path("/importRule")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Produces(MediaType.APPLICATION_JSON)
	public String importRule(@Auth DWUser user, @FormDataParam("file") InputStream is,
			@FormDataParam("file") FormDataContentDisposition fileDetail, @QueryParam("data") String data)
			throws Exception {
		JSONObject response = new JSONObject();
		try {
			Document doc = DOMUtil.getInstance().parse(is);
			GJaxbInterpretationConfig cep = XMLJAXBContext.getInstance().unmarshallDocument(doc,
					GJaxbInterpretationConfig.class);

			List<GJaxbInterpretationConfig> rules = Arrays.asList(cep);
			// TODO null should be root UC path
			this.addRules(user, rules);
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response.toString();
	}

	@POST
	@Path("/addRules")
	public String addRules(@Auth DWUser user, List<GJaxbInterpretationConfig> rules) throws Exception {// , Map<QName,
																										// List<File>>
																										// additionalFilesByRuleName)
																										// throws
																										// Exception {
		String res = null;
		try {
			// String collaborationName = user.getCurrentCollaborationName();
			// String knowledgespaceName = user.getCurrentKnowledgeSpaceName();
			String validMessage = "";
			String errorMessage = "";
			for (GJaxbInterpretationConfig rule : rules) {
				try {

					if (rule.isSetCepRule() || rule.isSetMlConfig()) {
						// analyse rule
						changeBrokerVariableByBrokerAddress(rule);
						GJaxbInterpretationConfigs configs = new GJaxbInterpretationConfigs();
						if (!rule.isSetContext()) {
							rule.setContext(new GJaxbContext());
						}
						rule.getContext().setCollaborationName(user.getCurrentCollaborationName());
						rule.getContext().setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
						configs.getInterpretationConfig().add(rule);

						GJaxbDeploy request = InterpretationConfigDeployHelper.buildRequest(configs,
								user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());
						request.setResourcesFolderBaseDir(
								new File(RIOConstant.RESOURCES_FOLDER).getCanonicalFile().toString());
						GJaxbDeployResponse response = this.interpretationClient.deploy(request);
						validMessage = validMessage + "rule '" + rule.getName() + "' loaded!!!\n";
					} else {
						LOG.warn("Rule not loaded : '" + rule.getName() + "' !!!");
					}
				} catch (Exception e) {
					e.printStackTrace();
					LOG.error("Error on rule '" + rule.getName() + "' !!!\n", e);
					GindWebApplicationException.manageError(e, this);
				}
			}
			if (!errorMessage.trim().isEmpty()) {
				throw new Exception(errorMessage);
			}
			JSONObject resp = new JSONObject();
			resp.put("message", validMessage);
			res = resp.toString();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return res;
	}

	@POST
	@Path("/addRulesFromURL")
	public String addRulesFromURL(@Auth DWUser user, List<InterpretationConfigByUrl> rules) throws Exception {
		List<GJaxbInterpretationConfig> resRules = new ArrayList<GJaxbInterpretationConfig>();
		// Map<QName, List<File>> additionalFilesByRuleName = new HashMap<QName,
		// List<File>>();
		if (rules != null) {
			for (InterpretationConfigByUrl _rule : rules) {
				GJaxbInterpretationConfig rule = XMLJAXBContext.getInstance()
						.unmarshallDocument(URI.create(_rule.getUrl()).toURL(), GJaxbInterpretationConfig.class);
				;
				// if(rule.getMlConfig() != null) {
				// List<File> additionalURLs = new ArrayList<File>();
				// for(String dataset : rule.getMlConfig().getDataSetToTrain()) {
				// additionalURLs.add(new File(URI.create(dataset)));
				// }
				// for(String model : rule.getMlConfig().getTrainedModel()) {
				// additionalURLs.add(new File(URI.create(model)));
				// }
				// additionalFilesByRuleName.put(rule.getName(), additionalURLs);
				// }
				resRules.add(rule);
			}
		}
		return this.addRules(user, resRules);// , additionalFilesByRuleName);
	}

	private void changeBrokerVariableByBrokerAddress(GJaxbInterpretationConfig rule) throws Exception {

		// Change ${eventProducerAgentBroker} by real address
		if (rule.getContext() != null && rule.getContext().getSubscriptionsRequired() != null) {
			for (GJaxbSubscriptionRequiredType subs : rule.getContext().getSubscriptionsRequired().getEntry()) {
				if (subs.getEndpointAddressToSubscribe() == null) {
					subs.setEndpointAddressToSubscribe(this.eventProducerAgentBrokerAddress);
				} else if (subs.getEndpointAddressToSubscribe() != null
						&& subs.getEndpointAddressToSubscribe().trim().equals("${eventProducerAgentBroker}")) {
					subs.setEndpointAddressToSubscribe(this.eventProducerAgentBrokerAddress);
				}
			}
		}
	}

	private void changeBrokerAddressByBrokerVariable(GJaxbInterpretationConfig rule) throws Exception {

		// Change ${eventProducerAgentBroker} by real address
		if (rule.getContext() != null && rule.getContext().getSubscriptionsRequired() != null) {
			for (GJaxbSubscriptionRequiredType subs : rule.getContext().getSubscriptionsRequired().getEntry()) {
				if (subs.getEndpointAddressToSubscribe() == null) {
					subs.setEndpointAddressToSubscribe("${eventProducerAgentBroker}");
				} else if (subs.getEndpointAddressToSubscribe() != null
						&& subs.getEndpointAddressToSubscribe().trim().equals(this.eventProducerAgentBrokerAddress)) {
					subs.setEndpointAddressToSubscribe("${eventProducerAgentBroker}");
				}
			}
		}
	}

	@GET
	@Path("/rules")
	public List<GJaxbInterpretationConfig> getRules(@Auth DWUser user) throws Exception {
		List<GJaxbInterpretationConfig> rules = new ArrayList<GJaxbInterpretationConfig>();
		String collaborationName = user.getCurrentCollaborationName();
		String knowledgespaceName = user.getCurrentKnowledgeSpaceName();
		try {

			GJaxbGetInterpretationConfigs request = new GJaxbGetInterpretationConfigs();
			request.setCollaborationName(collaborationName);
			request.setKnowledgeSpaceName(knowledgespaceName);

			GJaxbGetInterpretationConfigsResponse response = this.interpretationClient
					.getInterpretationConfigs(request);
			for (GJaxbInterpretationConfig rule : response.getInterpretationConfig()) {
				this.changeBrokerAddressByBrokerVariable(rule);
				rule.getContext().setCollaborationName(null);
				rule.getContext().setKnowledgeSpaceName(null);
				rules.add(rule);
			}

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return rules;
	}

	@GET
	@Path("/rule")
	public GJaxbGetInterpretationConfigResponse getRule(@Auth DWUser user, @QueryParam("id") String ruleId)
			throws Exception {
		GJaxbGetInterpretationConfigResponse response = null;
		GJaxbInterpretationConfig rule = null;
		String collaborationName = user.getCurrentCollaborationName();
		String knowledgespaceName = user.getCurrentKnowledgeSpaceName();
		try {

			GJaxbGetInterpretationConfig request = new GJaxbGetInterpretationConfig();
			request.setCollaborationName(collaborationName);
			request.setKnowledgeSpaceName(knowledgespaceName);
			request.setRuleId(ruleId);

			response = this.interpretationClient.getInterpretationConfig(request);
			rule = response.getInterpretationConfig();
			if (rule != null) {
				this.changeBrokerAddressByBrokerVariable(rule);
				rule.getContext().setCollaborationName(null);
				rule.getContext().setKnowledgeSpaceName(null);
			}

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@DELETE
	@Path("/deleteRules")
	public GJaxbUndeployResponse deleteRules(@Auth DWUser user, List<GJaxbInterpretationConfig> rules)
			throws Exception {
		try {
			String collaborationName = user.getCurrentCollaborationName();
			String knowledgespaceName = user.getCurrentKnowledgeSpaceName();

			GJaxbUndeploy request = new GJaxbUndeploy();
			request.setCollaborationName(collaborationName);
			request.setKnowledgeSpaceName(knowledgespaceName);
			request.setRequest(new GJaxbDeployResult());

			for (GJaxbInterpretationConfig rule : rules) {
				request.getRequest().getRuleId().add(rule.getId());
			}

			request.setResourcesFolderBaseDir(new File(RIOConstant.RESOURCES_FOLDER).getCanonicalFile().toString());
			GJaxbUndeployResponse response = this.interpretationClient.undeploy(request);
			return response;
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
			return null;
		}
	}

	@DELETE
	@Path("/deleteRule")
	public GJaxbUndeployResponse deleteRule(@Auth DWUser user, @QueryParam("ruleId") String ruleId) throws Exception {
		try {
			String collaborationName = user.getCurrentCollaborationName();
			String knowledgespaceName = user.getCurrentKnowledgeSpaceName();

			GJaxbUndeploy request = new GJaxbUndeploy();
			request.setCollaborationName(collaborationName);
			request.setKnowledgeSpaceName(knowledgespaceName);
			request.setRequest(new GJaxbDeployResult());

			request.getRequest().getRuleId().add(ruleId);

			request.setResourcesFolderBaseDir(new File(RIOConstant.RESOURCES_FOLDER).getCanonicalFile().toString());
			GJaxbUndeployResponse response = this.interpretationClient.undeploy(request);
			return response;
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
			return null;
		}
	}

	@POST
	@Path("/predict")
	public String predict(@Auth DWUser user, GJaxbPredict request) throws Exception {
		assert user != null;
		String response = null;
		try {
			// delete non ascii value
			request.getPredictInputs().getProperty().forEach(p -> {
				if (p.getValue() != null) {
					byte[] bytes = StringUtils.getBytesUtf8(p.getValue().replaceAll("[^\\p{ASCII}]", ""));
					String utf8EncodedString = StringUtils.newStringUtf8(bytes);
					p.setValue(utf8EncodedString);
				}
			});

			request.setResourcesFolderBaseDir(new File(RIOConstant.RESOURCES_FOLDER).getCanonicalFile().toString());
			GJaxbPredictResponse resp = this.predictClient.predict(request);

			response = new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(resp)).get("predictResponse")
					.toString();
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

}
