/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.ServiceLoader;

import javax.xml.namespace.QName;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import fr.emac.gind.campaign.manager.CampaignManagerItf;
import fr.emac.gind.campaign.manager.client.CampaignManagerClient;
import fr.emac.gind.campaign.manager.data.model.GJaxbInputRequest.InputData;
import fr.emac.gind.campaign.manager.data.model.GJaxbProcessToDeploy;
import fr.emac.gind.commons.utils.io.zip.InputZip;
import fr.emac.gind.commons.utils.io.zip.ZipUtil;
import fr.emac.gind.commons.utils.net.IPUtil.TRANSPORT_PROTOCOL;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.ws.SPIWebServicePrimitives;
import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.commons.utils.xml.XMLCompactPrinter;
import fr.emac.gind.commons.utils.xml.XMLPrettyPrinter;
import fr.emac.gind.event.consumer.NotificationConsumerWebService;
import fr.emac.gind.eventcommonsdata.GJaxbInstanceInfo;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.gov.deduction.Deduction;
import fr.emac.gind.gov.deduction.client.DeductionClient;
import fr.emac.gind.gov.models_gov.GJaxbCloneSyncModelFromStatus;
import fr.emac.gind.gov.models_gov.GJaxbCloneSyncModelFromStatusResponse;
import fr.emac.gind.gov.models_gov.GJaxbExtractSyncModelResponse;
import fr.emac.gind.indicators.plugin.IndicatorsRangeStrategyPluginManager;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel.GlobalProperties;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbPropertyWMM;
import fr.emac.gind.modeler.genericmodel.GJaxbStatusType;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveMetaModel;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.models.generic.modeler.plugins.EffectiveMetaModelPluginManager;
import fr.emac.gind.monitoring.client.ProcessMonitoringClient;
import fr.emac.gind.process.GJaxbDeployResult;
import fr.emac.gind.process.GJaxbGetProcess;
import fr.emac.gind.process.GJaxbGetProcessResponse;
import fr.emac.gind.process.GJaxbListProcesses;
import fr.emac.gind.process.GJaxbListProcessesResponse;
import fr.emac.gind.process.GJaxbSubscribeOnDeploymentService;
import fr.emac.gind.process.GJaxbSubscribeOnDeploymentServiceResponse;
import fr.emac.gind.process.GJaxbUndeployResponse;
import fr.emac.gind.process.instance.GJaxbExecType;
import fr.emac.gind.process.instance.GJaxbRunASync;
import fr.emac.gind.process.instance.GJaxbRunASyncResponse;
import fr.emac.gind.process.instance.GJaxbRunSync;
import fr.emac.gind.process.instance.GJaxbRunSyncResponse;
import fr.emac.gind.processgeneratorinstance.GJaxbProcessGeneratorInstance;
import fr.emac.gind.processgeneratorinstance.GJaxbProcessGeneratorInstances;
import fr.emac.gind.processmonitoring.ProcessMonitoringItf;
import fr.emac.gind.processmonitoring.data.GJaxbGetExchanges;
import fr.emac.gind.processmonitoring.data.GJaxbGetExchangesResponse;
import fr.emac.gind.processmonitoring.data.GJaxbGetProcessInstance;
import fr.emac.gind.processmonitoring.data.GJaxbGetProcessInstanceResponse;
import fr.emac.gind.processmonitoring.data.GJaxbGetProcessInstances;
import fr.emac.gind.processmonitoring.data.GJaxbGetProcessInstancesResponse;
import fr.emac.gind.processmonitoring.data.GJaxbSubscribeOnExchangesOfInstance;
import fr.emac.gind.processmonitoring.data.GJaxbSubscribeOnExchangesOfInstanceResponse;
import fr.emac.gind.processmonitoring.data.GJaxbSubscribeOnProcessInstanceEventService;
import fr.emac.gind.processmonitoring.data.GJaxbSubscribeOnProcessInstanceEventServiceResponse;
import fr.emac.gind.processmonitoring.data.GJaxbUpdateProcessInstance;
import fr.emac.gind.rio.PluginCollaborativeModel;
import fr.emac.gind.rio.dw.resources.FileResource;
import fr.emac.gind.rio.dw.resources.gov.bo.ProcessListResponse;
import fr.emac.gind.rio.dw.resources.gov.bo.RunProcessRequest;
import fr.emac.gind.rio.dw.resources.gov.bo.SubscribeExchangesByInstanceRequest;
import fr.emac.gind.transport.protocols.soap.handler.SOAPHandler;
import fr.emac.gind.transport.protocols.soap.handler.SOAPHeader;
import fr.emac.gind.transport.protocols.soap.handler.SOAPSender;
import fr.emac.gind.we.command.GJaxbCancel;
import fr.emac.gind.we.command.GJaxbCancelResponse;
import fr.emac.gind.we.command.GJaxbDelete;
import fr.emac.gind.we.command.GJaxbDeleteResponse;
import fr.emac.gind.we.command.GJaxbGetProcessModels;
import fr.emac.gind.we.command.GJaxbGetProcessModelsResponse;
import fr.emac.gind.we.command.GJaxbPause;
import fr.emac.gind.we.command.GJaxbPauseResponse;
import fr.emac.gind.we.command.GJaxbResume;
import fr.emac.gind.we.command.GJaxbResumeResponse;
import fr.emac.gind.workflow.AbstractProcessGenerator;
import fr.emac.gind.workflow.command.client.ProcessClientCommand;
import fr.emac.gind.workflow.deduction.ProcessDeductionHelper;
import fr.emac.gind.workflow.engine.ProcessClient;
import fr.emac.gind.workflow.engine.prio.deployer.dispatch.DispatcherManager;

/*
 * #%L
 * dw-r-ioga-crisis-collaborative-process
 * %%
 * Copyright (C) 2014 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.xml.soap.SOAPFault;
import jakarta.xml.ws.soap.SOAPFaultException;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/r-ioga/process")
//@Api("process")
public class ProcessResource {

	private static Logger LOG = LoggerFactory.getLogger(ProcessResource.class.getName());

	protected Configuration conf = null;

	protected SOAPSender sender = new SOAPSender();
	
	protected NotificationConsumerWebService processDeployerConsumer = null;
	protected NotificationConsumerWebService newProcessInstanceConsumer = null;
	protected NotificationConsumerWebService monitoringConsumer = null;

	private Deduction deductionClient = null;
	protected ModelsResource models = null;
	protected CoreResource core = null;
	
	protected FileResource fileResource = null;

	protected ProjectResource project = null;

	protected CampaignManagerItf campaignClient = null;
	protected ProcessMonitoringItf processMonitoringClient = null;

	protected GJaxbEffectiveMetaModel processEffectiveMetaModel = null;
	protected GJaxbEffectiveMetaModel risksEffectiveModeler = null;

	protected static DispatcherManager DISPATCHER = null;

	protected IndicatorsRangeStrategyPluginManager uncertainlyStrategyManager = null;

	private Map<String, AbstractProcessGenerator> processGenerators = new HashMap<String, AbstractProcessGenerator>();
	private GJaxbProcessGeneratorInstances processDeductionStrategies = null;
	private GJaxbEffectiveMetaModel processEffMetaModel = null;

	public ProcessResource(Configuration configuration, CoreResource core, ModelsResource models,
			ProjectResource project, GJaxbEffectiveMetaModel processEffectiveMetaModel,
			NotificationConsumerWebService processDeployerConsumer, NotificationConsumerWebService newProcessInstanceConsumer, NotificationConsumerWebService monitoringConsumer, FileResource fileResource) throws Exception {
		this.conf = configuration;
		this.core = core;
		this.models = models;
		this.project = project;
		this.processEffectiveMetaModel = processEffectiveMetaModel;
		this.campaignClient = CampaignManagerClient.createClient(conf.getProperties().get("campaign-manager"));
		this.processMonitoringClient = ProcessMonitoringClient
				.createClient(conf.getProperties().get("monitoring-server"));
		this.processDeployerConsumer = processDeployerConsumer;
		this.newProcessInstanceConsumer = newProcessInstanceConsumer;
		this.monitoringConsumer = monitoringConsumer;

		risksEffectiveModeler = EffectiveMetaModelPluginManager.getInstance()
				.getEffectiveMetaModelFromContainerByQNameMap()
				.get(new QName("http://fr.emac.gind/collaborative-model/objectives", "Objectives"));

		this.uncertainlyStrategyManager = new IndicatorsRangeStrategyPluginManager();

		this.processDeductionStrategies = this.initProcessGenerators();

		this.deductionClient = DeductionClient
				.createClient(this.conf.getProperties().get("governance").replace("/gov", "/GovDeduction"));

		this.processEffMetaModel = EffectiveMetaModelPluginManager.getInstance()
				.getEffectiveMetaModelFromContainerByQNameMap()
				.get(new QName("http://fr.emac.gind/collaborative-model/process", "Process"));
		
		this.fileResource = fileResource;

	}

	private GJaxbProcessGeneratorInstances initProcessGenerators() throws Exception {
		GJaxbProcessGeneratorInstances res = new GJaxbProcessGeneratorInstances();
		// Discover and register the available commands
		ServiceLoader<AbstractProcessGenerator> processGeneratorLoader = ServiceLoader
				.load(AbstractProcessGenerator.class);
		processGenerators.clear();
		processGeneratorLoader.reload();
		Iterator<AbstractProcessGenerator> processGeneratorsIterator = processGeneratorLoader.iterator();
		while (processGeneratorsIterator.hasNext()) {
			AbstractProcessGenerator processGenerator = processGeneratorsIterator.next();
			processGenerators.put(processGenerator.getClass().getName(), processGenerator);
			GJaxbProcessGeneratorInstance gen = new GJaxbProcessGeneratorInstance();
			if (processGenerator.getDeductionStrategy() != null) {
				gen.setName(processGenerator.getDeductionStrategy().getName());
				gen.setClazz(processGenerator.getDeductionStrategy().getClass().getName());
				gen.setDescription(processGenerator.getDeductionStrategy().getDescription());
				gen.setNgFunction(processGenerator.getDeductionStrategy().getJavascriptFunction());
				gen.setNgFunctionParams(processGenerator.getDeductionStrategy().getJavascriptFunctionParameters().toString());
				gen.setFavoriteDomains(processGenerator.getDeductionStrategy().getFavoriteDomains());
			}
			res.getProcessGeneratorInstance().add(gen);
		}
		return res;
	}

	public void setProject(ProjectResource project) {
		this.project = project;
	}

	public DispatcherManager getDispatcher() throws Exception {
		if (DISPATCHER == null) {
			ProcessResource.DISPATCHER = new DispatcherManager(this.conf.getMapObjects());
			;
		}
		return DISPATCHER;
	}

//	@POST
//	@Path("/deduceProcess")
//	@Consumes(MediaType.APPLICATION_JSON)
//	@Produces(MediaType.APPLICATION_JSON)
//	public String deduceProcess(@Auth DWUser user, DeduceInput input) throws Exception {
//		assert user != null;
//		String reportsBuffer = null;
//		try {
//
//			GJaxbProcessGeneratorInstance pgi = JSONJAXBContext.getInstance().unmarshall(
//					"{ \"processGeneratorInstance\": " + input.getProcessGeneratorInstance() + " }",
//					GJaxbProcessGeneratorInstance.class);
//			String dataJSON = input.getSpecificInputData();
//
//			JSONObject data = null;
//
//			if (dataJSON != null) {
//				data = new JSONObject(dataJSON);
//			}
//
//			GJaxbDeduce request = new GJaxbDeduce();
//			request.setProcessGeneratorInstance(pgi);
//			request.setInputDeductionData(new InputDeductionData());
//			request.setCurrentCollaborationName(user.getCurrentCollaborationName());
//			request.setCurrentKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
//
//			if (data != null) {
//				for (String key : data.keySet()) {
//					request.getInputDeductionData().getProperty()
//							.add(GenericModelHelper.createProperty(key, data.get(key).toString()));
//				}
//			}
//
//			GJaxbDeduceResponse deduceResponse = this.deductionClient.deduce(request);
//
//			GJaxbDeductionReports reports = deduceResponse.getDeductionReports();
//
//			if (reports != null && reports.getDeductionReport() != null && !reports.getDeductionReport().isEmpty()) {
//
//				GJaxbGenericModel processes = new GJaxbGenericModel();
//				for (GJaxbDeductionReport report : reports.getDeductionReport()) {
//
//					if (report != null && report.getDeductionResult() != null
//							&& report.getDeductionResult().getGenericModel() != null) {
//						GJaxbGenericModel processGraph = report.getDeductionResult().getGenericModel();
//						validProcess(processGraph);
//
//						// notify of the creation of process (re-update only 'Process' node
//						List<GJaxbNode> processNodes = GenericModelHelper
//								.findNodesByType(GenericModelHelper.collab("Process"), processGraph.getNode());
//
//						processes.getNode().addAll(processNodes);
//
//					}
//
//				}
//
//				if (processes.getNode().size() > 0) {
//					// Publish process in Neo4J
//					GJaxbSynchronizeRequest m = new GJaxbSynchronizeRequest();
//					m.setExtract(false);
//					m.getExtractionStatus().add(GJaxbStatusType.ACTIVE);
//					m.setPublish(true);
//					m.setJsonModel(JSONJAXBContext.getInstance().marshallAnyElement(processes));
//					this.models.synchronize(Optional.of(user), m);
//				}
//
//			}
//
//			reportsBuffer = JSONJAXBContext.getInstance().marshallAnyElement(reports);
//			LOG.debug("reports: \n" + reportsBuffer);
//
//		} catch (Exception e) {
//			LOG.trace(e.getMessage(), e);
//			e.printStackTrace();
//
//			GindWebApplicationException.manageError(e, this);
//		}
//
//		return new JSONObject(reportsBuffer).getJSONObject("deductionReports").toString();
//	}

	private void validProcess(GJaxbGenericModel process) throws Exception {
		for (GJaxbEdge edge : process.getEdge()) {
			if (edge.getSource() == null || edge.getTarget() == null) {
				throw new Exception("Invalid process !!!");
			}
		}
	}

	@POST
	@Path("/deploy")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbDeployResult deploy(@Auth DWUser user, GJaxbProcessToDeploy sampledProcess) throws Exception {
		GJaxbDeployResult responseDeploy = null;
		try {
			responseDeploy = this.deployProcess(user, sampledProcess);
		} catch (Exception e) {

			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return responseDeploy;
	}

	@POST
	@Path("/undeploy")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbUndeployResponse undeploy(@Auth DWUser user, GJaxbProcessToDeploy sampledProcess) throws Exception {
		GJaxbUndeployResponse responseDeploy = null;
		try {

			GJaxbGetProcessInstancesResponse instances = this.instancesByProcess(user, sampledProcess.getProcessName());
			for (GJaxbInstanceInfo info : instances.getInstanceInfo()) {
				this.deleteOrchestration(user, info.getProcessInstanceId());
			}

			responseDeploy = this.undeployProcess(user, sampledProcess);
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return responseDeploy;
	}

	@POST
	@Path("/supervise")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbCloneSyncModelFromStatusResponse supervise(@Auth DWUser user, List<String> processInstanceIds)
			throws Exception {
		GJaxbCloneSyncModelFromStatusResponse response = null;
		try {

			GJaxbCloneSyncModelFromStatus request = new GJaxbCloneSyncModelFromStatus();
			request.setStatusIn(GJaxbStatusType.ACTIVE);
			request.setStatusOut(GJaxbStatusType.EXPECTED_FREEZE);
			request.getAdditionalInLabel().add("instance");
			request.getAdditionalOutLabel().addAll(processInstanceIds);
			response = this.models.cloneModelFromStatus(user, request);
			
			for(String processInstanceId: processInstanceIds) {
				GJaxbGetProcessInstance getInst = new GJaxbGetProcessInstance();
				getInst.setProcessInstanceId(processInstanceId);
				GJaxbGetProcessInstanceResponse responseGetInst = processMonitoringClient.getProcessInstance(getInst);
				
				GJaxbInstanceInfo instanceInfo = responseGetInst.getInstanceInfo();
				instanceInfo.setSupervisedAt(response.getClonedAt());
				
				GJaxbUpdateProcessInstance upInst = new GJaxbUpdateProcessInstance();
				upInst.setInstanceInfo(instanceInfo);
				processMonitoringClient.updateProcessInstance(upInst);
			}

		} catch (Exception e) {

			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);

		}
		return response;
	}

	@POST
	@Path("/runSync")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbRunSyncResponse runSync(@Auth DWUser user, RunProcessRequest runRequest) throws Exception {
		GJaxbRunSyncResponse response = null;
		try {

			String endpointAddress = runRequest.getEndpointAddress();
			if (endpointAddress == null) {
				GJaxbDeployResult responseDeploy = this.deployProcess(user, runRequest.getProcessToDeploy());
				endpointAddress = responseDeploy.getInternalEndpointAddress();
			}
			GJaxbRunSync request = new GJaxbRunSync();
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			request.setExecutionType(GJaxbExecType.ORCHESTRATION);

			for (InputData inputData : runRequest.getProcessToDeploy().getInputRequest().getInputData()) {
				for (fr.emac.gind.campaign.manager.data.model.GJaxbInputRequest.InputData.Data data : inputData.getData()) {
					for (GJaxbPropertyWMM prop : data.getPropertyWMM()) {
						String inputDataName = prop.getProperty().getName();
						String inputDataValue = prop.getProperty().getValue();

						Document doc = DOMUtil.getInstance().newDocument();
						Element elmt = doc.createElementNS("http://www.gind.emac.fr/process/instance", inputDataName);
						elmt.setTextContent(inputDataValue);
						doc.appendChild(elmt);
						request.getInputData().getAny().add(doc.getDocumentElement());
					}
				}
			}

			SOAPHeader header = new SOAPHeader(new HashMap<QName, String>() {
				{
					put(new QName("http://fr.emac.gind/", "websocketCommandSOAPEndpoint"),
							SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP, conf.getHost(),
									Integer.parseInt(conf.getProperties().get("websocket-command-service-port")),
									Integer.parseInt(conf.getProperties().get("proxy-port")),
									"/websocketCommandService"));
					put(new QName("http://fr.emac.gind/", "currentUser"), Base64.getEncoder().encodeToString(
							JSONJAXBContext.getInstance().marshallAnyElement(user.getUser()).getBytes()));
					put(new QName("http://fr.emac.gind/", "withoutThread"), "true");
					put(new QName("http://fr.emac.gind/", "activateMonitoring"), "true");
					
				}
			});

			Document req = XMLJAXBContext.getInstance().marshallAnyElement(request);
			Document soapResp = this.sender.sendSoapRequest(req, endpointAddress,
					"http://www.gind.emac.fr/process/instance/runSync", header);
			Document resp = SOAPHandler.extractPayload(soapResp);

			response = XMLJAXBContext.getInstance().unmarshallDocument(resp, GJaxbRunSyncResponse.class);
			LOG.debug("response in dw: " + XMLCompactPrinter.print(resp));

		} catch (Exception e) {

			e.printStackTrace();
			if (e instanceof SOAPFaultException) {
				SOAPFault sf = ((SOAPFaultException) e).getFault();
				NodeList nl = sf.getDetail().getElementsByTagNameNS("http://www.gind.emac.fr", "workflowErrorDetails");
				Element error = (Element) nl.item(0);
				String message = DOMUtil.getInstance()
						.findFirstElementByNs(error, new QName("http://www.gind.emac.fr", "message")).getTextContent();
				String stacktrace = DOMUtil.getInstance()
						.findFirstElementByNs(error, new QName("http://www.gind.emac.fr", "stacktrace"))
						.getTextContent();

				GindWebApplicationException.manageError(
						new Exception("Error in workflow execution:\n\n" + "message: " + message + "\n\n" + stacktrace),
						this);
			}

			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/runASync")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbRunASyncResponse runASync(@Auth DWUser user, RunProcessRequest runRequest) throws Exception {
		GJaxbRunASyncResponse response = null;
		try {

			String endpointAddress = runRequest.getEndpointAddress();
			if (endpointAddress == null) {
				GJaxbDeployResult responseDeploy = this.deployProcess(user, runRequest.getProcessToDeploy());
				endpointAddress = responseDeploy.getInternalEndpointAddress();
			}

			GJaxbRunASync asyncRequest = new GJaxbRunASync();
			GJaxbRunSync request = new GJaxbRunSync();
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			request.setExecutionType(GJaxbExecType.ORCHESTRATION);

			for (InputData inputData : runRequest.getProcessToDeploy().getInputRequest().getInputData()) {
				for (fr.emac.gind.campaign.manager.data.model.GJaxbInputRequest.InputData.Data data : inputData.getData()) {
					for (GJaxbPropertyWMM prop : data.getPropertyWMM()) {
						String inputDataName = prop.getProperty().getName();
						String inputDataValue = prop.getProperty().getValue();

						Document doc = DOMUtil.getInstance().newDocument();
						Element elmt = doc.createElementNS("http://www.gind.emac.fr/process/instance", inputDataName);
						elmt.setTextContent(inputDataValue);
						doc.appendChild(elmt);
						request.setInputData(new GJaxbRunSync.InputData());
						request.getInputData().getAny().add(doc.getDocumentElement());
					}
				}
			}

			asyncRequest.setRunSync(request);
			asyncRequest.getCallbackAddress()
					.add("http://0.0.0.0:" + conf.getProperties().get("process-async-response-callback-port")
							+ "/ProcessAsyncResponseCallBack");
			if (runRequest.getCallbackAddress() != null && !runRequest.getCallbackAddress().isBlank()) {
				asyncRequest.getCallbackAddress().add(runRequest.getCallbackAddress());
			}
			asyncRequest.setCollaborationName(user.getCurrentCollaborationName());
			asyncRequest.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			asyncRequest.setUserId(user.getUserId());

			SOAPHeader header = new SOAPHeader(new HashMap<QName, String>() {
				{
					put(new QName("http://fr.emac.gind/", "websocketCommandSOAPEndpoint"),
							SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP, conf.getHost(),
									Integer.parseInt(conf.getProperties().get("websocket-command-service-port")),
									Integer.parseInt(conf.getProperties().get("proxy-port")),
									"/websocketCommandService"));
					put(new QName("http://fr.emac.gind/", "currentUser"),
							JSONJAXBContext.getInstance().marshallAnyElement(user.getUser()));
					put(new QName("http://fr.emac.gind/", "withoutThread"), "false");
					put(new QName("http://fr.emac.gind/", "activateMonitoring"), "true");

				}
			});

			Document req = XMLJAXBContext.getInstance().marshallAnyElement(asyncRequest);
			Document soapResp = this.sender.sendSoapRequest(req, endpointAddress,
					"http://www.gind.emac.fr/process/instance/runASync", header);
			Document resp = SOAPHandler.extractPayload(soapResp);

			response = XMLJAXBContext.getInstance().unmarshallDocument(resp, GJaxbRunASyncResponse.class);
			LOG.debug("response in dw: " + XMLCompactPrinter.print(resp));

		} catch (Exception e) {

			e.printStackTrace();
			if (e instanceof SOAPFaultException) {
				SOAPFault sf = ((SOAPFaultException) e).getFault();
				NodeList nl = sf.getDetail().getElementsByTagNameNS("http://www.gind.emac.fr", "workflowErrorDetails");
				Element error = (Element) nl.item(0);
				String message = DOMUtil.getInstance()
						.findFirstElementByNs(error, new QName("http://www.gind.emac.fr", "message")).getTextContent();
				String stacktrace = DOMUtil.getInstance()
						.findFirstElementByNs(error, new QName("http://www.gind.emac.fr", "stacktrace"))
						.getTextContent();

				Exception newE = new Exception(
						"Error in workflow execution:\n\n" + "message: " + message + "\n\n" + stacktrace);
				GindWebApplicationException.manageError(newE, this);
			}

			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	public GJaxbDeployResult deployProcess(DWUser user, GJaxbProcessToDeploy sampledProcess) throws Exception {

		GJaxbGenericModel model = extractSingleProcess(user, sampledProcess.getProcessId(), sampledProcess.getProcessName());

		List<InputZip> zips = new ArrayList<InputZip>();
		zips.add(new InputZip(model.getName() + "/" + model.getPackage().trim() + ".prio", new ByteArrayInputStream(
				XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(model)).getBytes())));

		Map<String, GJaxbGenericModel> rscs = this.findWorldModels(user);

		Document docProjects = XMLJAXBContext.getInstance().marshallAnyElement(rscs.get("project"));
		zips.add(new InputZip(model.getName() + "/" + "project.xml", new ByteArrayInputStream(XMLPrettyPrinter.print(docProjects).getBytes())));

		if (rscs.get(RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.OBJECTIVES_PACKAGE_NAME)) != null) {
			Document riskProjects = XMLJAXBContext.getInstance().marshallAnyElement(
					rscs.get(RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.OBJECTIVES_PACKAGE_NAME)));
			zips.add(new InputZip(
					model.getName() + "/" + RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.OBJECTIVES_PACKAGE_NAME) + ".xml",
					new ByteArrayInputStream(XMLPrettyPrinter.print(riskProjects).getBytes())));
		}

		if (rscs.get(
				RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.CONTEXT_PARTNERS_PACKAGE_NAME)) != null) {
			Document functionsProjects = XMLJAXBContext.getInstance().marshallAnyElement(
					rscs.get(RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.CONTEXT_PARTNERS_PACKAGE_NAME)));
			zips.add(new InputZip(
					model.getName() + "/" + RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.CONTEXT_PARTNERS_PACKAGE_NAME) + ".xml",
					new ByteArrayInputStream(XMLPrettyPrinter.print(functionsProjects).getBytes())));
			
			// Get all wsdls of goods
			for(GJaxbNode node: rscs.get(RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.CONTEXT_PARTNERS_PACKAGE_NAME)).getNode()) {
				if(node.getType().equals(GenericModelHelper.collab("Good"))) {
					if(GenericModelHelper.findProperty("wsdl", node.getProperty()) != null) {
						String wsdlPath = GenericModelHelper.findProperty("wsdl", node.getProperty()).getValue();
						String wsdlName = wsdlPath.substring(wsdlPath.lastIndexOf("/") + "/".length());
						String wsdlBuffer = this.fileResource.download(user, wsdlPath);
						
						if(wsdlBuffer != null && !wsdlBuffer.isEmpty()) {
							zips.add(new InputZip(
									model.getName() + "/" + "wsdls/goods/" + node.getId() + "/" + wsdlName,
									new ByteArrayInputStream(wsdlBuffer.getBytes())));
						}
					}
				}
			}
		}
		
		
		
		

		//File parentDir = new File("./target/");
		//parentDir.mkdirs();
		//File zip = new File(parentDir, model.getName() + ".zip");
		
		//ByteArrayOutputStream output = new ByteArrayOutputStream();
		//ZipUtil.zip(output, zips.toArray(new InputZip[zips.size()]));
		
	    ByteArrayOutputStream output = new ByteArrayOutputStream();
		ZipUtil.zip(output, zips.toArray(new InputZip[zips.size()]));
		
		


		GJaxbDeployResult responseDeploy = this.getDispatcher().deploy(output, null, user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());

		return responseDeploy;
	}

	public GJaxbUndeployResponse undeployProcess(DWUser user, GJaxbProcessToDeploy sampledProcess) throws Exception {
		GJaxbUndeployResponse responseDeploy = this.getDispatcher()
				.undeploy(new QName("http://www.gind.emac.fr/process/instance", sampledProcess.getProcessName()), user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());
		return responseDeploy;
	}



	private GJaxbGenericModel extractSingleProcess(DWUser user, String processId, String processName) throws Exception {

		GJaxbGenericModel model = ProcessDeductionHelper.getInstance().extractProcess(models.getModelsClient(),
				processId,
				new QName("http://fr.emac.gind/collaborative-model/process",
						PluginCollaborativeModel.PROCESS_PACKAGE_NAME),
				user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());

		String collaboration = user.getCurrentCollaborationName();
		String knowledge = user.getCurrentKnowledgeSpaceName();
		String procesName = RegExpHelper.toRegexFriendlyName(processName);
		if (collaboration != null) {
			procesName = procesName + "__c_" + RegExpHelper.toRegexFriendlyName(collaboration);
		}
		if (knowledge != null) {
			procesName = procesName + "__k_" + RegExpHelper.toRegexFriendlyName(knowledge);
		}
		model.setName(processName);
		model.setPackage(PluginCollaborativeModel.PROCESS_PACKAGE_NAME.toLowerCase());

		if (!model.isSetGlobalProperties()) {
			model.setGlobalProperties(new GlobalProperties());
		}
		model.getGlobalProperties().getProperty()
				.add(GenericModelHelper.createProperty("userId", user.getUserId()));
		if (collaboration != null) {
			model.getGlobalProperties().getProperty().add(GenericModelHelper.createProperty("collaborationName",
					RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName())));
		}
		if (knowledge != null) {
			model.getGlobalProperties().getProperty().add(GenericModelHelper.createProperty("knowledgeSpaceName",
					RegExpHelper.toRegexFriendlyName(user.getCurrentKnowledgeSpaceName())));
		}
		return model;
	}

	@GET
	@Path("/list")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public List<ProcessListResponse> processDeployedList(@Auth DWUser user) throws Exception {
		List<ProcessListResponse> res = new ArrayList<ProcessListResponse>();
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {
				GJaxbListProcesses request = new GJaxbListProcesses();
				request.setCollaborationName(user.getCurrentCollaborationName());
				request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				GJaxbListProcessesResponse result = wcd.getValue().listProcesses(request);
				ProcessListResponse plr = new ProcessListResponse();
				plr.setWorkflowEngineName(wcd.getKey());
				plr.setDeployedProcesses(result);
				res.add(plr);
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();

			GindWebApplicationException.manageError(e, this);
		}
		return res;
	}

	@GET
	@Path("/instancesByProcess/{processName}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetProcessInstancesResponse instancesByProcess(@Auth DWUser user,
			@PathParam("processName") String processName) throws Exception {

		GJaxbGetProcessInstances request = new GJaxbGetProcessInstances();
		GJaxbGetProcessInstancesResponse response = null;
		try {
			request.setProcessQName(new QName("http://www.gind.emac.fr/process/instance", processName));
			response = this.processMonitoringClient.getProcessInstances(request);
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();

			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/processInstance/{instanceId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetProcessInstanceResponse processInstance(@Auth DWUser user,
			@PathParam("instanceId") String instanceId) throws Exception {
		GJaxbGetProcessInstanceResponse response = null;
		try {
			GJaxbGetProcessInstance request = new GJaxbGetProcessInstance();
			request.setProcessInstanceId(instanceId);
			response = this.processMonitoringClient.getProcessInstance(request);
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();

			GindWebApplicationException.manageError(e, this);
		}
		return response;

	}

	@GET
	@Path("/exchangesByInstance/{instanceId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetExchangesResponse exchangesByInstance(@Auth DWUser user, @PathParam("instanceId") String instanceId)
			throws Exception {
		GJaxbGetExchangesResponse response = null;
		try {
			GJaxbGetExchanges request = new GJaxbGetExchanges();
			request.setProcessInstanceId(instanceId);
			response = this.processMonitoringClient.getExchanges(request);
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}
	
	@POST
	@Path("/subscribeOnDeploymentService/")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbSubscribeOnDeploymentServiceResponse subscribeOnDeploymentService(@Auth DWUser user,
			GJaxbSubscribeOnDeploymentService parameters) throws Exception {
		GJaxbSubscribeOnDeploymentServiceResponse response = null;
		try {
			if (parameters.getConsumerAddress() == null) {
				parameters.setConsumerAddress(this.processDeployerConsumer.getAllAddresses().get(0));
			} 
			response = this.getDispatcher().subscribeOnDeploymentService(parameters, user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}
	
	@POST
	@Path("/subscribeOnProcessInstanceEventService/")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbSubscribeOnProcessInstanceEventServiceResponse subscribeOnProcessInstanceEventService(@Auth DWUser user,
			GJaxbSubscribeOnProcessInstanceEventService parameters) throws Exception {
		GJaxbSubscribeOnProcessInstanceEventServiceResponse response = null;
		try {
			if (parameters.getConsumerAddress() == null) {
				parameters.setConsumerAddress(this.newProcessInstanceConsumer.getAllAddresses().get(0));
			} 
			response = this.processMonitoringClient.subscribeOnProcessInstanceEventService(parameters);
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/subscribeExchangesByInstance/")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbSubscribeOnExchangesOfInstanceResponse subscribeExchangesByInstance(@Auth DWUser user,
			SubscribeExchangesByInstanceRequest req) throws Exception {
		GJaxbSubscribeOnExchangesOfInstanceResponse response = null;
		try {
			GJaxbSubscribeOnExchangesOfInstance request = new GJaxbSubscribeOnExchangesOfInstance();
			request.setProcessInstanceId(req.getProcessInstanceId());
			if (req.getCallbackAddress() == null) {
				request.setConsumerAddress(this.monitoringConsumer.getAllAddresses().get(0));
			} else {
				request.setConsumerAddress(req.getCallbackAddress());
			}
			response = this.processMonitoringClient.subscribeOnExchangesOfInstance(request);
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/getProcessModels/{processName}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String getProcessModels(@Auth DWUser user, @PathParam("processName") String processName) throws Exception {
		GJaxbGetProcessModelsResponse res = new GJaxbGetProcessModelsResponse();
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {

				ProcessClientCommand pcc = new ProcessClientCommand(
						wcd.getValue().getServerAddress().replace("_deployerService", "_commandService"));
				GJaxbGetProcessModels request = new GJaxbGetProcessModels();
				request.setProcessQName(new QName("http://www.gind.emac.fr/process/instance", processName));

				GJaxbGetProcessModelsResponse result = pcc.getProcessModels(request);
				if (result.isSetProcess()) {
					res = result;
					break;
				}
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(res))
				.getJSONObject("getProcessModelsResponse").toString();
	}

	@GET
	@Path("/getProcessDeployResult/{processName}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetProcessResponse getProcessDeployResult(@Auth DWUser user,
			@PathParam("processName") String processName) throws Exception {
		GJaxbGetProcessResponse res = new GJaxbGetProcessResponse();
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {
				ProcessClient client = wcd.getValue();
				GJaxbGetProcess request = new GJaxbGetProcess();
				request.setProcessName(new QName("http://www.gind.emac.fr/process/instance", processName));
				GJaxbGetProcessResponse result = client.getProcess(request);
				if (result.getRequest() != null) {
					res = result;
					break;
				}
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return res;
	}



	private Map<String, GJaxbGenericModel> findWorldModels(DWUser user) throws Exception {
		Map<String, GJaxbGenericModel> rscs = new HashMap<String, GJaxbGenericModel>();
		try {
			GJaxbNode project = this.project.getProjectByNameInCollaboration(user, user.getCurrentKnowledgeSpaceName(),
					user.getCurrentCollaborationName());
			List<QName> functionsConceptModel = Arrays.asList(
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Function"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Community"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Person"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Good"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Role"));
			List<QName> functionsRelationModel = Arrays.asList();
			List<QName> risksConceptModel = Arrays.asList(
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Potentiality"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Potential"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Objective"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Actuality"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Procedure"));
			List<QName> risksRelationModel = Arrays.asList(
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Concerns"),
					new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Treated_By"));

			GJaxbGenericModel projectModel = new GJaxbGenericModel();
			projectModel.getNode().add(project);

			GJaxbExtractSyncModelResponse extractActiveFunctionsResp = this.models.extraction(user, functionsConceptModel,
					functionsRelationModel, Arrays.asList(GJaxbStatusType.ACTIVE), null, null);
			
			GJaxbExtractSyncModelResponse extractFreezeFunctionsResp = this.models.extraction(user, functionsConceptModel,
					functionsRelationModel, Arrays.asList(GJaxbStatusType.FREEZE), null, null);
			
			GJaxbGenericModel allFunctionsModel = new GJaxbGenericModel();
			allFunctionsModel.getNode().addAll(extractActiveFunctionsResp.getGenericModel().getNode());
			allFunctionsModel.getNode().addAll(extractFreezeFunctionsResp.getGenericModel().getNode());
			allFunctionsModel.getEdge().addAll(extractActiveFunctionsResp.getGenericModel().getEdge());
			allFunctionsModel.getEdge().addAll(extractFreezeFunctionsResp.getGenericModel().getEdge());
			GenericModelHelper.deleteDuplicateNodes(allFunctionsModel.getNode());
			GenericModelHelper.deleteDuplicateEdges(allFunctionsModel.getEdge());
			
			GJaxbExtractSyncModelResponse extractRisksResp = this.models.extraction(user, risksConceptModel,
					risksRelationModel, Arrays.asList(GJaxbStatusType.ACTIVE), null, null);

			rscs.put("project", projectModel);
			rscs.put(RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.CONTEXT_PARTNERS_PACKAGE_NAME),
					allFunctionsModel);
			rscs.put(RegExpHelper.toRegexFriendlyName(PluginCollaborativeModel.OBJECTIVES_PACKAGE_NAME),
					extractRisksResp.getGenericModel());

		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return rscs;
	}

	

	@POST
	@Path("/findAssociatedStrategiesToRisk")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGenericModel findAssociatedProceduresToRisk(@Auth DWUser user, Map<String, Object> jsonObject)
			throws Exception {
		GJaxbGenericModel resp = null;
		try {
			String labels = "";
			String collaborationName = (String) jsonObject.get("collaborationName");
			if (collaborationName != null) {
				labels = labels + ":" + "`" + collaborationName + "`";
			}
			String knowledgeSpaceName = (String) jsonObject.get("knowledgeSpaceName");
			if (knowledgeSpaceName != null) {
				labels = labels + ":" + "`" + knowledgeSpaceName + "`";
			}
			String riskId = (String) jsonObject.get("riskId");
			if (collaborationName != null) {
				riskId = riskId + "_c__" + collaborationName;
			}
			if (knowledgeSpaceName != null) {
				riskId = riskId + "_k__" + knowledgeSpaceName;
			}
			resp = this.models.query(user, "Match (n1:`{http://fr.emac.gind/collaborative-model}Potentiality`" + labels
					+ "{ modelNodeId:'" + riskId
					+ "' })<-[:`{http://fr.emac.gind/collaborative-model}Concerns`]-(o:`{http://fr.emac.gind/collaborative-model}Objective`)-[:`{http://fr.emac.gind/collaborative-model}Treated_By`]->(n2:`{http://fr.emac.gind/collaborative-model}Procedure`"
					+ labels + " ) return n2;", "process");
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return resp;
	}

	public CoreResource getCore() {
		return core;
	}

	@GET
	@Path("/pauseOrchestration/{processInstanceId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbPauseResponse pauseOrchestration(@Auth DWUser user,
			@PathParam("processInstanceId") String processInstanceId) throws Exception {
		GJaxbPauseResponse result = null;
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {
				ProcessClientCommand pcc = new ProcessClientCommand(
						wcd.getValue().getServerAddress().replace("_deployerService", "_commandService"));
				GJaxbPause pauseParameters = new GJaxbPause();
				pauseParameters.setProcessInstanceId(processInstanceId);
				result = pcc.pause(pauseParameters);
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return result;
	}

	@GET
	@Path("/resumeOrchestration/{processInstanceId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbResumeResponse resumeOrchestration(@Auth DWUser user,
			@PathParam("processInstanceId") String processInstanceId) throws Exception {
		GJaxbResumeResponse result = null;
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {
				ProcessClientCommand pcc = new ProcessClientCommand(
						wcd.getValue().getServerAddress().replace("_deployerService", "_commandService"));
				GJaxbResume resumeParameters = new GJaxbResume();
				resumeParameters.setProcessInstanceId(processInstanceId);
				result = pcc.resume(resumeParameters);
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return result;
	}

	@DELETE
	@Path("/deleteOrchestration/{processInstanceId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbDeleteResponse deleteOrchestration(@Auth DWUser user,
			@PathParam("processInstanceId") String processInstanceId) throws Exception {
		GJaxbDeleteResponse result = null;
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {
				// wcd.getValue().
				ProcessClientCommand pcc = new ProcessClientCommand(
						wcd.getValue().getServerAddress().replace("_deployerService", "_commandService"));
				GJaxbDelete deleteParameters = new GJaxbDelete();
				deleteParameters.setProcessInstanceId(processInstanceId);
				result = pcc.delete(deleteParameters);
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return result;
	}

	@POST
	@Path("/cancelOrchestration/")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbCancelResponse cancelOrchestration(@Auth DWUser user,
			GJaxbCancel cancelReq) throws Exception {
		GJaxbCancelResponse result = new GJaxbCancelResponse();
		try {
			for (Entry<String, ProcessClient> wcd : this.getDispatcher().getWorkflowEngines().entrySet()) {
				ProcessClientCommand pcc = new ProcessClientCommand(
						wcd.getValue().getServerAddress().replace("_deployerService", "_commandService"));
				
				result = pcc.cancel(cancelReq);
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return result;
	}

	public GJaxbProcessGeneratorInstances getProcessDeductionStrategies() {
		return this.processDeductionStrategies;
	}

}
