/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URL;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.xml.namespace.QName;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.eclipse.persistence.internal.oxm.ByteArrayDataSource;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataParam;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import fr.emac.gind.campaign.manager.data.model.GJaxbProcessToDeploy;
import fr.emac.gind.commons.gis.DistanceTimeMatrixManager;
import fr.emac.gind.commons.gis.GlobalDistanceTimeMatrices;
import fr.emac.gind.commons.utils.RIOConstant;
import fr.emac.gind.commons.utils.excel.ExcelModelReader;
import fr.emac.gind.commons.utils.excel.ExcelModelWriter;
import fr.emac.gind.commons.utils.io.FileUtil;
import fr.emac.gind.commons.utils.io.zip.GZipUtil;
import fr.emac.gind.commons.utils.io.zip.ZipUtil;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.xml.XMLNSCleaner;
import fr.emac.gind.commons.utils.xml.XMLPrettyPrinter;
import fr.emac.gind.distance.data.GJaxbDistanceTimeMatrix;
import fr.emac.gind.dsl.kpil.IndicatorManager;
import fr.emac.gind.dsl.kpil.parser.definition.IndicatorsAnalyzer;
import fr.emac.gind.game_master.GameMasterApi;
import fr.emac.gind.game_master.GameMasterManagerClient;
import fr.emac.gind.game_master.data.GJaxbCreateGameScenario;
import fr.emac.gind.game_master.data.GJaxbDeleteGameScenarios;
import fr.emac.gind.game_master.data.GJaxbExportGameScenarios;
import fr.emac.gind.game_master.data.GJaxbExportGameScenariosResponse;
import fr.emac.gind.game_master.data.GJaxbGameScenario;
import fr.emac.gind.generic.application.DWApplicationService;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.generic.application.utils.LoadedUsecase;
import fr.emac.gind.gov.ai.chatbot.service.AIChatbotImpl;
import fr.emac.gind.gov.ai_chatbot.GJaxbContext;
import fr.emac.gind.gov.ai_chatbot.GJaxbRestoreRAGFromDirectory;
import fr.emac.gind.gov.ai_chatbot.GJaxbRestoreRAGFromDirectoryResponse;
import fr.emac.gind.gov.core.client.util.Neo4JId;
import fr.emac.gind.gov.core.client.util.Neo4JReqConstant;
import fr.emac.gind.gov.core_gov.GJaxbGetNode;
import fr.emac.gind.gov.core_gov.GJaxbGetNodeResponse;
import fr.emac.gind.gov.meta_models_gov.GJaxbExtractSyncEffectiveDomain;
import fr.emac.gind.gov.meta_models_gov.GJaxbExtractSyncEffectiveDomainResponse;
import fr.emac.gind.gov.meta_models_gov.GJaxbPublishSyncEffectiveDomain;
import fr.emac.gind.gov.models_gov.GJaxbSynchronizeRequest;
import fr.emac.gind.gov.models_gov.GJaxbSynchronizeResponse;
import fr.emac.gind.gov.system_gov.GJaxbAddKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbAddKnowledgeSpaceResponse;
import fr.emac.gind.gov.system_gov.GJaxbConnectUserToCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbFindKnowledgeSpacesByUserAndCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbGetKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbGetKnowledgeSpaceByNameInCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbSelectedKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbUpdateKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbUpdateUser;
import fr.emac.gind.indicators.Indicator;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.ll.LLManager;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.model.interpretation.config.GJaxbEventAction;
import fr.emac.gind.model.interpretation.config.GJaxbInterpretationConfig;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.modeler.genericmodel.GJaxbStatusType;
import fr.emac.gind.modeler.metamodel.GJaxbConnectionRule;
import fr.emac.gind.modeler.metamodel.GJaxbConnectionRules;
import fr.emac.gind.modeler.metamodel.GJaxbContainmentRule;
import fr.emac.gind.modeler.metamodel.GJaxbDomain;
import fr.emac.gind.modeler.metamodel.GJaxbDomainDefinition;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveConceptType;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveDomain;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveMetaModel;
import fr.emac.gind.modeler.metamodel.GJaxbEffectivePackage;
import fr.emac.gind.modeler.metamodel.GJaxbEffectivePackage.EffectiveMetaModelOfPackage;
import fr.emac.gind.modeler.metamodel.GJaxbEffectivePackage.Rules;
import fr.emac.gind.modeler.metamodel.GJaxbFormtypeType;
import fr.emac.gind.modeler.metamodel.GJaxbMetaModel;
import fr.emac.gind.modeler.metamodel.GJaxbPackage;
import fr.emac.gind.modeler.metamodel.GJaxbRelation;
import fr.emac.gind.modeler.metamodel.GJaxbRules;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.models.generic.modeler.plugins.EffectiveMetaModelPluginManager;
import fr.emac.gind.models.generic.modeler.utils.svg.SVGGenerator;
import fr.emac.gind.process.GJaxbDeployResult;
import fr.emac.gind.rio.PluginCollaborativeModel;
import fr.emac.gind.rio.dw.resources.gov.bo.ExportProjectRequest;
import fr.emac.gind.rio.dw.resources.gov.bo.FindParameters;
import fr.emac.gind.rio.dw.resources.gov.bo.ProcessListResponse;
import fr.emac.gind.sensors.controler.data.GJaxbSensorControlerBackup;
import fr.emac.gind.sensors.manager.SensorManager;
import fr.emac.gind.sensors.manager.SensorManagerClient;
import fr.emac.gind.sensors.manager.data.GJaxbBackupTimeSeriesDB;
import fr.emac.gind.sensors.manager.data.GJaxbBackupTimeSeriesDBResponse;
import fr.emac.gind.sensors.manager.data.GJaxbCreateSensor;
import fr.emac.gind.sensors.manager.data.GJaxbRestoreTimeSeriesDB;
import fr.emac.gind.sensors.manager.data.GJaxbRestoreTimeSeriesDBResponse;
import fr.emac.gind.usecase.GJaxbCompleteUsecaseDefinition;
import fr.emac.gind.usecase.GJaxbCompleteUsecaseDefinition.ResourcesByType;
import fr.emac.gind.usecase.GJaxbLocationType;
import fr.emac.gind.usecase.GJaxbModelURLType;
import fr.emac.gind.usecase.GJaxbResourceType;
import fr.emac.gind.usecase.GJaxbResourceURLType;
import fr.emac.gind.usecase.GJaxbShortUsecaseDefinitionType;
import fr.emac.gind.usecases.RIOAbstractProject;
import fr.emac.gind.usecases.RIOUsecasesManager;

/*
 * #%L
 * dw-prorisk-project-editor
 * %%
 * Copyright (C) 2014 - 2015 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import io.dropwizard.auth.Auth;
import jakarta.activation.DataHandler;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.StreamingOutput;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/r-ioga/projects")
//@Api("projects")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class ProjectResource {

	private static Logger LOG = LoggerFactory.getLogger(ProjectResource.class);

	private Configuration conf = null;
	private String applicationName = null;
	private MetaModelsResource metaModel = null;
	private CoreResource core = null;
	private ModelsResource models = null;
	private SystemResource system = null;
	private ProcessResource process = null;
	private SensorManagerResource sensorManagerResource = null;
	private InterpretationRulesResource interpretationRules = null;
	private AIChatbotResource aichatBotResource = null;

	private GJaxbEffectiveMetaModel projectEffectiveModeler = null;
	private GameMasterApi gmClient = null;
	private SensorManager sensorManagerClient = null;

	private static final String CACHE_MATRIX_PATH = "distance_time_cache_matrix/matrix.xml";
	private static final String INTERPRETATION_RULES_FOLDER = "interpretation_rules";

	public static Map<String, LoadedUsecase> USECASES_ALREADY_LOADED = new HashMap<String, LoadedUsecase>();

	public static String REALWORLD_UC_NAME = "RealWorld";
	public static String WORLD_COLLABORATION_NAME = "World";

	public static GJaxbNode REALWORLD_UC_NODE = null;
	public static GJaxbNode WORLD_COLLABORATION_NODE = null;

	public ProjectResource(Configuration conf, String applicationName, SystemResource system,
			MetaModelsResource metaModel, CoreResource coreResource, ModelsResource models, ProcessResource process,
			SensorManagerResource eventProducerAgentResource, InterpretationRulesResource interpretationRules,
			AIChatbotResource aichatBotResource, GJaxbEffectiveMetaModel projectEffectiveModeler) throws Exception {
		this.conf = conf;
		this.system = system;
		this.metaModel = metaModel;
		this.core = coreResource;
		this.models = models;
		this.process = process;
		this.aichatBotResource = aichatBotResource;
		this.interpretationRules = interpretationRules;
		this.sensorManagerResource = eventProducerAgentResource;
		this.projectEffectiveModeler = projectEffectiveModeler;
		this.applicationName = applicationName;
		this.gmClient = GameMasterManagerClient.createClient(conf.getProperties().get("game-master-server"));
		this.sensorManagerClient = SensorManagerClient.createClient(conf.getProperties().get("sensor-manager-server"));

		SVGGenerator svgGen = new SVGGenerator();
		for (GJaxbEffectivePackage effPack : EffectiveMetaModelPluginManager.getInstance().getUmlEffectiveDomain()
				.getEffectivePackage()) {
			for (GJaxbRelation relation : effPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
					.getRelation()) {
				DWApplicationService.generateIcon(svgGen, relation, "uml", null);
			}
		}
		for (GJaxbRelation relation : EffectiveMetaModelPluginManager.getInstance().getUmlEffectiveDomain()
				.getGlobalEffectiveMetaModel().getEffectiveMetaModel().getRelation()) {
			DWApplicationService.generateIcon(svgGen, relation, "uml", null);
		}

		// load predefined usecase
		String loadPredefinedUsecases = conf.getProperties().get("load-predefined-usecases");
		if (loadPredefinedUsecases != null) {
			JSONArray predefineducs = new JSONArray(loadPredefinedUsecases);

			for (int i = 0; i < predefineducs.length(); i++) {
				JSONObject preduc = predefineducs.getJSONObject(i);

				String collaborationName = preduc.getString("collaborationName");
				if (collaborationName == null || collaborationName.isBlank()) {
					throw new Exception("CollaborationName not defined !!!");
				}

				String knownledgeSpaceName = preduc.getString("knowledgeSpaceName");
				if (knownledgeSpaceName == null || knownledgeSpaceName.isBlank()) {
					throw new Exception("knownledgeSpaceName not defined !!!");
				}

				String idUsecase = "_c__" + collaborationName + "_k__" + knownledgeSpaceName;

				if (USECASES_ALREADY_LOADED.get(idUsecase) == null) {
					GJaxbNode uc_collab = SystemResource.createCollaboration(collaborationName,
							system.getSystemClient(), system.getCollaborationMetaModel());

					DWUser defaultAdmin = new DWUser(SystemResource.getDEFAULT_ADMIN(), true,
							this.system.getSystemClient());
					GJaxbConnectUserToCollaboration reqConn = new GJaxbConnectUserToCollaboration();
					reqConn.setCollaborationId(uc_collab.getId());
					reqConn.setUserId(SystemResource.getDEFAULT_ADMIN().getId());
					this.system.connectUserToCollaboration(defaultAdmin, reqConn);

					GJaxbNode fakeUserNode = new GJaxbNode();
					if (SystemResource.getDEFAULT_ADMIN() != null) {
						fakeUserNode = SystemResource.getDEFAULT_ADMIN();
					}

					DWUser fakeUser = new DWUser(fakeUserNode, true, this.system.getSystemClient());
					fakeUser.setCurrentCollaborationName(collaborationName);
					fakeUser.setCurrentKnowledgeSpaceName(knownledgeSpaceName);

					RIOAbstractProject founduc = null;
					Optional<RIOAbstractProject> opfounduc = RIOUsecasesManager.getInstance(this.conf.getVersion())
							.getEmbeddedUsecases().stream()
							.filter(uc -> uc.getUsecaseDef().getName().equals(knownledgeSpaceName)).findFirst();
					if (opfounduc != null && opfounduc.isPresent()) {
						founduc = opfounduc.get();
					}

					if (founduc == null) {
						LOG.info("Not embedded usecase... try to download...");
						Optional<GJaxbShortUsecaseDefinitionType> downloaducS = RIOUsecasesManager
								.getInstance(this.conf.getVersion()).getDownloadUsecases().stream()
								.filter(uc -> uc.getName().equals(knownledgeSpaceName)).findFirst();
						if (downloaducS != null && downloaducS.isPresent()) {
							founduc = RIOUsecasesManager.getInstance(this.conf.getVersion())
									.download(downloaducS.get());
							LOG.info("predefined usecase downloaded");
						}
						if (founduc == null) {
							throw new Exception("Impossible to found uc '" + knownledgeSpaceName + "' in list: "
									+ RIOUsecasesManager.getInstance(this.conf.getVersion()).getEmbeddedUsecases()
											.stream().map(uc -> uc.getUsecaseDef().getName()));
						}
					}
					GJaxbNode uc_kn = this.loadProject(founduc, fakeUser.getCurrentCollaborationName(),
							fakeUser.getCurrentKnowledgeSpaceName(), fakeUser, false, false, new ArrayList<String>(),
							false);

					USECASES_ALREADY_LOADED.put(idUsecase, new LoadedUsecase(uc_collab, uc_kn));

					if (REALWORLD_UC_NAME.equals(knownledgeSpaceName)) {
						REALWORLD_UC_NODE = uc_kn;
					}
					if (WORLD_COLLABORATION_NAME.equals(collaborationName)) {
						WORLD_COLLABORATION_NODE = uc_collab;
					}

					GJaxbAddKnowledgeSpace req = new GJaxbAddKnowledgeSpace();
					req.setNode(uc_kn);
					req.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
					req.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
					req.getSelectedKnowledgeSpace().setKnowledgeName(knownledgeSpaceName);

					GJaxbAddKnowledgeSpaceResponse resp = this.system.getSystemClient().addKnowledgeSpace(req);
					uc_kn.setId(resp.getId());

					this.system.attachKnowledgeSpaceToCollaboration(fakeUser, uc_kn.getId(),
							GenericModelHelper.getName(uc_kn), collaborationName);

					if (REALWORLD_UC_NAME.equals(knownledgeSpaceName)) {
						Boolean activateTweetsListener = true;
						if (preduc.has("activateTweetsListener")) {
							activateTweetsListener = preduc.getBoolean("activateTweetsListener");
						}

						// get twitter sensor
						GJaxbGetNode getNodeReq = new GJaxbGetNode();
						getNodeReq.setId("node_06785c2e-fdae-67d3-dc6c-7b083072ef36");
						getNodeReq.setSelectedKnowledgeSpace(
								new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
						getNodeReq.getSelectedKnowledgeSpace()
								.setCollaborationName(req.getSelectedKnowledgeSpace().getCollaborationName());
						getNodeReq.getSelectedKnowledgeSpace()
								.setKnowledgeName(req.getSelectedKnowledgeSpace().getKnowledgeName());
						GJaxbGetNodeResponse getNodeResp = this.core.getNode(fakeUser, getNodeReq);
						GJaxbNode twitterNode = getNodeResp.getNode();

						GenericModelHelper.findProperty("type", twitterNode.getProperty(), true).setValue("SOCIAL");
						GenericModelHelper.findProperty("nature", twitterNode.getProperty(), true).setValue("TWITTER");
						GenericModelHelper.findProperty("activateTweetsListener", twitterNode.getProperty(), true)
								.setValue(activateTweetsListener.toString());

						// start twitter sensor
						GJaxbCreateSensor paReq = new GJaxbCreateSensor();
						paReq.setCollaborationName(req.getSelectedKnowledgeSpace().getCollaborationName());
						paReq.setKnowledgeSpaceName(req.getSelectedKnowledgeSpace().getKnowledgeName());
						paReq.setStartSensor(true);
						paReq.setNode(twitterNode);

						String twitterKey = SensorManagerResource.createTwitterKey(fakeUser);

						paReq.setContext(new GJaxbCreateSensor.Context());
						paReq.getContext().getProperty()
								.add(GenericModelHelper.createProperty("twitterKey", twitterKey));

						this.sensorManagerResource.createSensor(fakeUser, paReq);
					}

				}

			}
		}

	}
	
	

	public SystemResource getSystem() {
		return system;
	}



	@GET
	@Path("/findProjectsByUserAndCollaboration/{collaborationName}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public List<GJaxbNode> findProjectsByUserAndCollaboration(@Auth DWUser user,
			@PathParam("collaborationName") String collaborationName,
			@QueryParam("qCollaborationName") String qCollaborationName) throws Exception {
		assert user != null;
		List<GJaxbNode> knowledges = null;
		try {
			GJaxbFindKnowledgeSpacesByUserAndCollaboration req = new GJaxbFindKnowledgeSpacesByUserAndCollaboration();
			req.setUserId(user.getUser().getId());
			req.setCollaborationName(qCollaborationName);
			knowledges = this.system.getSystemClient().findKnowledgeSpacesByUserAndCollaboration(req).getNode();

		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return knowledges;
	}

	@POST
	@Path("/projects")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbNode addProject(@Auth DWUser user, GJaxbNode projectNode) throws Exception {
		assert user != null;
		_addProject(user, projectNode, user.getCurrentCollaborationName(), null, new ArrayList<String>(), false);
		return projectNode;
	}

	public void _addProject(DWUser user, GJaxbNode projectNode, String collaborationName, String knowledgeSpaceName,
			List<String> projectsAlreadyLoaded, boolean metaModelAlreadyLoaded) throws Exception {
		try {
			// verify if a project with a same name does not exist
			String name = GenericModelHelper.getName(projectNode);
			if (!user.isFake()) {
				GJaxbGenericModel model = this.models.query(user, "match (n:" + Neo4JReqConstant.system("Project")
						+ ":`" + collaborationName + "`) where n.property_name = '" + name + "' return n;", null);
				if (model != null && model.getNode().size() > 0) {
					throw new Exception("A project with the same name '" + name + "' already exist !!!");
				}
			}

			// verify if indicators script is valid
			IndicatorManager indMan = new IndicatorManager(projectNode);

			JSONObject indicatorsJson = new JSONObject();
			for (Entry<String, Indicator> entry : indMan.getIndicators().entrySet()) {
				indicatorsJson.put(entry.getKey(), entry.getValue().toJSON());
			}

			GenericModelHelper.findProperty("indicators", projectNode.getProperty(), true)
					.setValue(indicatorsJson.toString());

			if (!user.isFake() && knowledgeSpaceName == null) {
				user.setCurrentKnowledgeSpaceName(name);
				knowledgeSpaceName = name;
				this.system.updateUser(user, user.getUser());
			}

			if (projectNode.getId() == null) {
				projectNode.setId("project_" + UUID.randomUUID());
			}
			String friendlyProjectId = RegExpHelper.toRegexFriendlyName(projectNode.getId());
			projectNode.setId(friendlyProjectId);

			if (!user.isFake() && !projectNode.getStatus().contains(GJaxbStatusType.REFERENCE)
					&& !projectNode.getStatus().contains(GJaxbStatusType.SHARE)) {
				GJaxbAddKnowledgeSpace req = new GJaxbAddKnowledgeSpace();
				req.setNode(projectNode);
				req.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
				req.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
				req.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);

				GJaxbAddKnowledgeSpaceResponse resp = this.system.getSystemClient().addKnowledgeSpace(req);
				projectNode.setId(resp.getId());
				this.system.attachKnowledgeSpaceToCollaboration(user, projectNode.getId(), name, collaborationName);

				GlobalDistanceTimeMatrices.create(user.getUser());

			}

			if (!metaModelAlreadyLoaded && !projectNode.getStatus().contains(GJaxbStatusType.REFERENCE)
					&& !projectNode.getStatus().contains(GJaxbStatusType.SHARE)) {
				metaModelAlreadyLoaded = true;
				createEffectiveDomain(user, projectNode, (RIOAbstractProject) projectNode.getUserData("project"),
						collaborationName, knowledgeSpaceName);
			}

			loadProjectsAsReferences(user, projectNode, collaborationName, knowledgeSpaceName, projectsAlreadyLoaded,
					metaModelAlreadyLoaded);
			loadProjectsAsSharedResources(user, projectNode, collaborationName, knowledgeSpaceName,
					projectsAlreadyLoaded, metaModelAlreadyLoaded);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
	}

	public void loadProjectsAsReferences(DWUser user, GJaxbNode projectNode, String collaborationName,
			String knowledgeSpaceName, List<String> projectsAlreadyLoaded, boolean metaModelAlreadyLoaded)
			throws Exception {
		// Load references projects
		GJaxbProperty referencesProjectsProp = GenericModelHelper.findProperty("references projects",
				projectNode.getProperty());
		if (referencesProjectsProp != null && referencesProjectsProp.getValue() != null
				&& !referencesProjectsProp.getValue().isBlank()) {
			JSONArray referencesProjectsArray = new JSONArray(referencesProjectsProp.getValue());
			for (int i = 0; i < referencesProjectsArray.length(); i++) {
				JSONArray refProjJson = referencesProjectsArray.getJSONArray(i);
				List<GJaxbProperty> projRefProps = GenericModelHelper.convertJSONArrayToPropertyList(refProjJson);
				String projectName = GenericModelHelper.findProperty("project name", projRefProps).getValue();
				String projectId = GenericModelHelper.findProperty("project id", projRefProps).getValue();

				String projectKey = projectName + "##" + "reference";
				if (!projectsAlreadyLoaded.contains(projectKey)) {
					projectsAlreadyLoaded.add(projectKey);

					RIOAbstractProject internalProject = RIOUsecasesManager.getInstance(this.conf.getVersion())
							.findAvailableUsecaseByFriendlyName(RegExpHelper.toRegexFriendlyName(projectName), true,
									null);
					if (internalProject == null) {
						throw new Exception(projectName + " cannot be found !!!");
					}

					selectUsedResources(internalProject, null);
					GJaxbNode projectRef = loadProject(internalProject, collaborationName, knowledgeSpaceName, user,
							true, false, projectsAlreadyLoaded, metaModelAlreadyLoaded);
				} else {
					LOG.debug("Project already loaded as reference: " + projectName);
				}
			}
		}
	}

	public void loadProjectsAsSharedResources(DWUser user, GJaxbNode projectNode, String collaborationName,
			String knowledgeSpaceName, List<String> projectsAlreadyLoaded, boolean metaModelAlreadyLoaded)
			throws Exception {
		// Load shared resources projects
		GJaxbProperty sharedResourcesProjectsProp = GenericModelHelper.findProperty("shared resources projects",
				projectNode.getProperty());
		if (sharedResourcesProjectsProp != null && sharedResourcesProjectsProp.getValue() != null
				&& !sharedResourcesProjectsProp.getValue().isBlank()) {
			JSONArray sharedResourcesProjectsArray = new JSONArray(sharedResourcesProjectsProp.getValue());
			for (int i = 0; i < sharedResourcesProjectsArray.length(); i++) {
				JSONArray sharedRscProjJson = sharedResourcesProjectsArray.getJSONArray(i);
				List<GJaxbProperty> projSharedRscProps = GenericModelHelper
						.convertJSONArrayToPropertyList(sharedRscProjJson);
				String projectName = GenericModelHelper.findProperty("project name", projSharedRscProps).getValue();
				String projectId = GenericModelHelper.findProperty("project id", projSharedRscProps).getValue();

				String projectKey = projectName + "##" + "shared_resource";
				if (!projectsAlreadyLoaded.contains(projectKey)) {
					projectsAlreadyLoaded.add(projectKey);

					RIOAbstractProject internalProject = RIOUsecasesManager.getInstance(this.conf.getVersion())
							.findAvailableUsecaseByFriendlyName(RegExpHelper.toRegexFriendlyName(projectName), true,
									null);
					if (internalProject == null) {
						throw new Exception(projectName + " cannot be found !!!");
					}

					selectUsedResources(internalProject, null);
					GJaxbNode projectSharedRsc = loadProject(internalProject, collaborationName, knowledgeSpaceName,
							user, false, true, projectsAlreadyLoaded, metaModelAlreadyLoaded);
				} else {
					LOG.debug("Project already loaded as share resource: " + projectName);
				}
			}
		}
	}

	@PUT
	@Path("/projects")
	@Consumes(MediaType.APPLICATION_JSON)
	public GJaxbNode updateProject(@Auth DWUser user, GJaxbNode node) throws Exception {
		assert user != null;
		try {

			// verify if indicators script is valid
			IndicatorManager indMan = new IndicatorManager(node);

			JSONObject indicatorsJson = new JSONObject();
			for (Entry<String, Indicator> entry : indMan.getIndicators().entrySet()) {
				indicatorsJson.put(entry.getKey(), entry.getValue().toJSON());
			}
			;
			GenericModelHelper.findProperty("indicators", node.getProperty(), true).setValue(indicatorsJson.toString());

			GJaxbUpdateKnowledgeSpace req = new GJaxbUpdateKnowledgeSpace();
			req.setNode(node);
			req.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			req.getSelectedKnowledgeSpace().setKnowledgeName(user.getCurrentKnowledgeSpaceName());
			req.getSelectedKnowledgeSpace().setCollaborationName(user.getCurrentCollaborationName());
			this.system.getSystemClient().updateKnowledgeSpace(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return node;
	}

	private GJaxbEffectiveDomain createEffectiveDomain(DWUser user, GJaxbNode projectNode, RIOAbstractProject project,
			String collaborationName, String knowledgeSpaceName) throws Exception {

		List<String> selectedDomains = this.getDomains(projectNode);
		String currentDomain = selectedDomains.get(0);

		if (project != null) {
			GJaxbEffectiveDomain effDomain = null;

			// metamodel known but not in project
			GJaxbDomainDefinition knownDomain = null;
			for (GJaxbDomainDefinition dd : EffectiveMetaModelPluginManager.getInstance().getDomainsDefinition()) {
				if (dd.getName().toLowerCase().equals(currentDomain)) {
					knownDomain = dd;
					break;
				}
			}

			if (knownDomain != null) {
				effDomain = EffectiveMetaModelPluginManager.getInstance()
						.createEffectiveDomainFromDefinition(knownDomain);

				if (project.getMap().get(GJaxbResourceType.DOMAIN).size() > 0) {
					// extended metamodel in projets
					GJaxbResourceURLType rscDomain = project.getMap().get(GJaxbResourceType.DOMAIN).get(0);
					GJaxbDomain extendedDomain = XMLJAXBContext.getInstance()
							.unmarshallDocument(URI.create(rscDomain.getUrl()).toURL(), GJaxbDomain.class);

					Map<String, List<GJaxbPackage>> mapPackagesByDomain = EffectiveMetaModelPluginManager.getInstance()
							.getExistingPackagesByDomainMap();
					List<GJaxbPackage> extendedPackages = new ArrayList<GJaxbPackage>();
					if (mapPackagesByDomain.get(extendedDomain.getName()) != null) {
						extendedPackages.addAll(mapPackagesByDomain.get(extendedDomain.getName()));
					}
					// List<GJaxbMetaModel> extendedMetaModels = new ArrayList<GJaxbMetaModel>();

					/**
					 * for (GJaxbResourceURLType rscP :
					 * project.getMap().get(GJaxbResourceType.META_MODEL)) {
					 * extendedMetaModels.add(XMLJAXBContext.getInstance()
					 * .unmarshallDocument(URI.create(rscP.getUrl()).toURL(),
					 * GJaxbMetaModel.class)); }
					 */

					for (GJaxbResourceURLType rscP : project.getMap().get(GJaxbResourceType.PACKAGE)) {
						GJaxbPackage extendedPackage = XMLJAXBContext.getInstance()
								.unmarshallDocument(URI.create(rscP.getUrl()).toURL(), GJaxbPackage.class);
						extendedPackages.add(extendedPackage);
						EffectiveMetaModelPluginManager.getInstance().getPackages().put(
								new QName(extendedPackage.findTargetNamespace(), extendedPackage.getName()),
								extendedPackage);

						/**
						 * GJaxbEffectiveMetaModel effMetaModel =
						 * EffectiveMetaModelHelper.createMergedEffectiveMetaModelFromPackage( new
						 * QName(extendedPackage.getTargetNamespace(), extendedPackage.getName()), new
						 * ArrayList<GJaxbImportedNode>(extendedPackage.getImportedNode()),
						 * extendedMetaModels);
						 */
					}

					GJaxbEffectiveDomain extendedEffDomain = EffectiveMetaModelPluginManager.getInstance()
							.createEffectiveDomainFromResources(extendedDomain, extendedPackages);

					effDomain = this.mergeExtendedEffectiveDomain(effDomain, extendedEffDomain,
							user.getCurrentCollaborationName());

					EffectiveMetaModelPluginManager.getInstance().setCacheEffectiveDomainByCollaborationAndKnowledge(
							user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName(), effDomain);
				}

			} else {
				// new metamodel
				if (project.getMap().get(GJaxbResourceType.DOMAIN).size() > 0) {
					// metamodel in projets
					GJaxbResourceURLType rscDomain = project.getMap().get(GJaxbResourceType.DOMAIN).get(0);
					GJaxbDomain domain = XMLJAXBContext.getInstance()
							.unmarshallDocument(URI.create(rscDomain.getUrl()).toURL(), GJaxbDomain.class);

					List<GJaxbPackage> _packages = new ArrayList<GJaxbPackage>();
					List<GJaxbResourceURLType> rscPackages = project.getMap().get(GJaxbResourceType.PACKAGE);
					for (GJaxbResourceURLType rscP : rscPackages) {
						GJaxbPackage _package = XMLJAXBContext.getInstance()
								.unmarshallDocument(URI.create(rscP.getUrl()).toURL(), GJaxbPackage.class);
						_packages.add(_package);
						EffectiveMetaModelPluginManager.getInstance().getPackages()
								.put(new QName(_package.findTargetNamespace(), _package.getName()), _package);
					}
					effDomain = EffectiveMetaModelPluginManager.getInstance().createEffectiveDomainFromResources(domain,
							_packages);

				}
			}

			// generate svg icon
			SVGGenerator svgGen = new SVGGenerator();
			for (GJaxbEffectivePackage effPack : effDomain.getEffectivePackage()) {
				for (GJaxbRelation relation : effPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
						.getRelation()) {
					DWApplicationService.generateIcon(svgGen, relation,
							RegExpHelper.toRegexFriendlyName(collaborationName),
							RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
				}
			}
			for (GJaxbRelation relation : effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel()
					.getRelation()) {
				DWApplicationService.generateIcon(svgGen, relation, RegExpHelper.toRegexFriendlyName(collaborationName),
						RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
			}

			// push effective domain in collab/kn corresponding
			GJaxbPublishSyncEffectiveDomain publishReq = new GJaxbPublishSyncEffectiveDomain();
			publishReq.setEffectiveDomain(effDomain);
			publishReq.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
			publishReq.getSelectedKnowledgeSpace()
					.setCollaborationName(RegExpHelper.toRegexFriendlyName(collaborationName));
			publishReq.getSelectedKnowledgeSpace()
					.setKnowledgeName(RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
			this.metaModel.publishSyncEffectiveDomain(user, publishReq);

			EffectiveMetaModelPluginManager.getInstance().setCacheEffectiveDomainByCollaborationAndKnowledge(
					collaborationName, knowledgeSpaceName, effDomain);

		} else {
			// new project
			GJaxbDomainDefinition knownDomain = null;
			for (GJaxbDomainDefinition dd : EffectiveMetaModelPluginManager.getInstance().getDomainsDefinition()) {
				if (dd.getName().toLowerCase().equals(currentDomain)) {
					knownDomain = dd;
					break;
				}
			}

			GJaxbEffectiveDomain effDomain = EffectiveMetaModelPluginManager.getInstance()
					.createEffectiveDomainFromDefinition(knownDomain);

			// set all readonly
			effDomain.getEffectivePackage().stream().forEach(ep -> {
				ep.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getEffectiveConcept().stream()
						.forEach(ec -> ec.setReadonly(true));
				ep.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getRelation().stream()
						.forEach(r -> r.setReadonly(true));
			});
			effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel().getEffectiveConcept().stream()
					.forEach(ec -> ec.setReadonly(true));
			effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel().getRelation().stream()
					.forEach(r -> r.setReadonly(true));

			// generate svg icon
			SVGGenerator svgGen = new SVGGenerator();
			for (GJaxbEffectivePackage effPack : effDomain.getEffectivePackage()) {
				for (GJaxbRelation relation : effPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
						.getRelation()) {
					DWApplicationService.generateIcon(svgGen, relation,
							RegExpHelper.toRegexFriendlyName(collaborationName),
							RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
				}
			}
			for (GJaxbRelation relation : effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel()
					.getRelation()) {
				DWApplicationService.generateIcon(svgGen, relation, RegExpHelper.toRegexFriendlyName(collaborationName),
						RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
			}

			// push effective domain in collab/kn corresponding
			GJaxbPublishSyncEffectiveDomain publishReq = new GJaxbPublishSyncEffectiveDomain();
			publishReq.setEffectiveDomain(effDomain);
			publishReq.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
			publishReq.getSelectedKnowledgeSpace()
					.setCollaborationName(RegExpHelper.toRegexFriendlyName(collaborationName));
			publishReq.getSelectedKnowledgeSpace()
					.setKnowledgeName(RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
			this.metaModel.publishSyncEffectiveDomain(user, publishReq);

			EffectiveMetaModelPluginManager.getInstance().setCacheEffectiveDomainByCollaborationAndKnowledge(
					collaborationName, knowledgeSpaceName, effDomain);

		}

		GJaxbExtractSyncEffectiveDomain extractReq = new GJaxbExtractSyncEffectiveDomain();
		extractReq.setDomain(currentDomain);
		extractReq.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
		extractReq.getSelectedKnowledgeSpace()
				.setCollaborationName(RegExpHelper.toRegexFriendlyName(collaborationName));
		extractReq.getSelectedKnowledgeSpace().setKnowledgeName(RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
		GJaxbExtractSyncEffectiveDomainResponse extractResp = this.metaModel.extractSyncEffectiveDomain(user,
				extractReq);

		return extractResp.getEffectiveDomain();
	}

	private GJaxbEffectiveDomain mergeExtendedEffectiveDomain(GJaxbEffectiveDomain effDomain,
			GJaxbEffectiveDomain extendedEffDomain, String collaborationName) {
		for (GJaxbEffectivePackage extendedEffPack : extendedEffDomain.getEffectivePackage()) {
			boolean found = false;
			for (GJaxbEffectivePackage effPack : effDomain.getEffectivePackage()) {
				if (RegExpHelper.toRegexFriendlyName(extendedEffPack.getName())
						.equals(RegExpHelper.toRegexFriendlyName(effPack.getName()))) {

					for (GJaxbEffectiveConceptType concept : extendedEffPack.getEffectiveMetaModelOfPackage()
							.getEffectiveMetaModel().getEffectiveConcept()) {
						if (concept.getType().getNamespaceURI().contains("${collab}")) {
							QName newType = new QName(
									concept.getType().getNamespaceURI().replace("${collab}",
											RegExpHelper.toRegexFriendlyName(collaborationName)),
									concept.getType().getLocalPart());
							concept.setType(newType);
						}
					}

					for (GJaxbContainmentRule.Contain _contain : extendedEffPack.getRules().getContainmentRule()
							.getContain()) {
						if (_contain.getType().getNamespaceURI().contains("${collab}")) {
							QName newType = new QName(
									_contain.getType().getNamespaceURI().replace("${collab}",
											RegExpHelper.toRegexFriendlyName(collaborationName)),
									_contain.getType().getLocalPart());
							_contain.setType(newType);
						}
					}

					for (GJaxbConnectionRule rule : extendedEffPack.getEffectiveMetaModelOfPackage()
							.getEffectiveMetaModel().getRules().getConnectionRules().getConnectionRule()) {
						if (rule.getRelationType().getNamespaceURI().contains("${collab}")) {
							QName newType = new QName(
									rule.getRelationType().getNamespaceURI().replace("${collab}",
											RegExpHelper.toRegexFriendlyName(collaborationName)),
									rule.getRelationType().getLocalPart());
							rule.setRelationType(newType);
						}

						if (rule.getFrom().getNamespaceURI().contains("${collab}")) {
							QName newType = new QName(
									rule.getFrom().getNamespaceURI().replace("${collab}",
											RegExpHelper.toRegexFriendlyName(collaborationName)),
									rule.getFrom().getLocalPart());
							rule.setFrom(newType);
						}
						if (rule.getTo().getNamespaceURI().contains("${collab}")) {
							QName newType = new QName(
									rule.getTo().getNamespaceURI().replace("${collab}",
											RegExpHelper.toRegexFriendlyName(collaborationName)),
									rule.getTo().getLocalPart());
							rule.setTo(newType);
						}
					}

					effPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getEffectiveConcept()
							.addAll(extendedEffPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
									.getEffectiveConcept());
					effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel().getEffectiveConcept()
							.addAll(extendedEffPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
									.getEffectiveConcept());
					effPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getRules().getConnectionRules()
							.getConnectionRule().addAll(extendedEffPack.getEffectiveMetaModelOfPackage()
									.getEffectiveMetaModel().getRules().getConnectionRules().getConnectionRule());
					effPack.getRules().getContainmentRule().getContain()
							.addAll(extendedEffPack.getRules().getContainmentRule().getContain());

					effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel().getRules().getConnectionRules()
							.getConnectionRule().addAll(extendedEffPack.getEffectiveMetaModelOfPackage()
									.getEffectiveMetaModel().getRules().getConnectionRules().getConnectionRule());

					found = true;
				}
			}

			if (!found) {
				effDomain.getEffectivePackage().add(extendedEffPack);
			}
		}

		return effDomain;
	}

	private GJaxbEffectiveDomain getAndRefreshEffectiveDomainIfNotExist(DWUser user, GJaxbNode projectNode,
			RIOAbstractProject project, String collaborationName, String knowledgeSpaceName, boolean forceToRefresh)
			throws Exception {

		GJaxbEffectiveDomain effDomain = EffectiveMetaModelPluginManager.getInstance()
				.getCacheEffectiveDomainByCollaborationAndKnowledge(collaborationName, knowledgeSpaceName);

		if ((effDomain == null || forceToRefresh) && projectNode != null) {
			GJaxbExtractSyncEffectiveDomain extractReq = new GJaxbExtractSyncEffectiveDomain();
			extractReq.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
			extractReq.getSelectedKnowledgeSpace()
					.setCollaborationName(RegExpHelper.toRegexFriendlyName(collaborationName));
			extractReq.getSelectedKnowledgeSpace()
					.setKnowledgeName(RegExpHelper.toRegexFriendlyName(knowledgeSpaceName));
			extractReq.setReload(true);
			GJaxbExtractSyncEffectiveDomainResponse extractResp = this.metaModel.extractSyncEffectiveDomain(user,
					extractReq);

			EffectiveMetaModelPluginManager.getInstance().setCacheEffectiveDomainByCollaborationAndKnowledge(
					collaborationName, knowledgeSpaceName, extractResp.getEffectiveDomain());
			effDomain = extractResp.getEffectiveDomain();
		}

		return effDomain;
	}

	private List<String> getDomains(GJaxbNode projectNode) {
		List<String> selectedDomains = new ArrayList<String>();
		JSONObject domains = new JSONObject(
				GenericModelHelper.findProperty("domains", projectNode.getProperty()).getValue());
		for (String domain : domains.keySet()) {
			if (domains.getJSONObject(domain).getBoolean("checked") == true) {
				if (!selectedDomains.contains(domain)) {
					selectedDomains.add(domain);
				}
			}
		}

		assert selectedDomains.size() == 1;
		return selectedDomains;
	}

	@GET
	@Path("/projects/{id}")
	public GJaxbNode getProject(@Auth DWUser user, @PathParam("id") String id, @QueryParam("qid") String qid)
			throws Exception {
		assert user != null;
		GJaxbNode project = null;
		try {
			String collaborationName = user.getCurrentCollaborationName();
			String knowledgeSpaceName = user.getCurrentKnowledgeSpaceName();

			if (qid.indexOf("_k__") > 0) {
				knowledgeSpaceName = qid.substring(qid.indexOf("_k__") + "_k__".length());
			}

			GJaxbGetKnowledgeSpace req = new GJaxbGetKnowledgeSpace();
			req.setId(Neo4JId.cleanId(qid));
			req.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			req.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
			req.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);
			project = this.system.getSystemClient().getKnowledgeSpace(req).getNode();

			user.setCurrentKnowledgeSpaceName(GenericModelHelper.getName(project));

			this.system.updateUser(null, user.getUser());
			// createEffectiveDomainIfNotExist(user, project, null,
			// user.getCurrentCollaborationName());
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return project;
	}

	@GET
	@Path("/getProjectByNameInCollaboration")
	public GJaxbNode getProjectByNameInCollaboration(@Auth DWUser user, @QueryParam("projectName") String projectName,
			@QueryParam("collaborationName") String collaborationName) throws Exception {
		assert user != null;
		GJaxbNode project = null;
		try {
			GJaxbGetKnowledgeSpaceByNameInCollaboration req = new GJaxbGetKnowledgeSpaceByNameInCollaboration();
			req.setKnowledgeSpaceName(projectName);
			req.setCollaborationName(collaborationName);
			project = this.system.getSystemClient().getKnowledgeSpaceByNameInCollaboration(req).getNode();

			if (project != null) {
				user.setCurrentKnowledgeSpaceName(GenericModelHelper.getName(project));
				this.system.updateUser(user, user.getUser());
				// createEffectiveDomainIfNotExist(user, project, null, collaborationName);
			}

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return project;
	}

	@GET
	@Path("/findEffectiveDomainByProjectId")
	public GJaxbEffectiveDomain findEffectiveDomainByProjectId(@Auth DWUser user,
			@QueryParam("projectId") String projectId, @QueryParam("forceToRefresh") Boolean forceToRefresh)
			throws Exception {
		assert user != null;
		GJaxbEffectiveDomain effDomain = null;
		try {

			String projIdFriendly = projectId;
			String collaborationName = user.getCurrentCollaborationName();
			String knowledgeSpaceName = user.getCurrentKnowledgeSpaceName();
			if (!projIdFriendly.contains("_c__")) {
				projIdFriendly = projectId + "_c__" + RegExpHelper.toRegexFriendlyName(collaborationName) + "_k__"
						+ RegExpHelper.toRegexFriendlyName(knowledgeSpaceName);
			} else {
				collaborationName = projIdFriendly.split("_c__")[1].split("_k__")[0];
				knowledgeSpaceName = projIdFriendly.split("_c__")[1].split("_k__")[1];
			}

			GJaxbNode projectNode = this.system.getKnowledgeSpace(user, projIdFriendly, projIdFriendly);
			if (forceToRefresh == null) {
				forceToRefresh = false;
			}
			// forceToRefresh = true;
			effDomain = this.getAndRefreshEffectiveDomainIfNotExist(user, projectNode, null, collaborationName,
					knowledgeSpaceName, forceToRefresh);

		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return effDomain;
	}

	@GET
	@Path("/findEffectivePackageByNameByProjectId")
	public GJaxbEffectivePackage findEffectivePackageByNameByProjectId(@Auth DWUser user,
			@QueryParam("projectId") String projectId, @QueryParam("category") String category) throws Exception {
		assert user != null;
		GJaxbEffectivePackage mcv = null;
		try {
			mcv = findEffectivePackage(user, projectId, category);
			// fix bug - still useful? - commented to avoid marshalling / connection lost
			// we need to find how it is possible to delete setter when generate json
			if (mcv == null) {
				throw new Exception("Impossible to find effectiveMetaModel");
			}
		} catch (Exception e) {
			LOG.trace(e.getMessage(), e);
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return mcv;
	}

	public synchronized GJaxbEffectivePackage findEffectivePackage(DWUser user, String projectId, String category)
			throws Exception {
		String projId = projectId;
		if (!projId.contains("_c__")) {
			projId = projectId + "_c__" + user.getCurrentCollaborationName() + "_k__"
					+ user.getCurrentKnowledgeSpaceName();
		}
		String projIdFriendly = projectId;
		if (!projIdFriendly.contains("_c__")) {
			projIdFriendly = projectId + "_c__" + RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName())
					+ "_k__" + RegExpHelper.toRegexFriendlyName(user.getCurrentKnowledgeSpaceName());
		}
		// GJaxbNode projectNode = this.system.getKnowledgeSpace(user, projId, projId);
		GJaxbEffectiveDomain effDomain = EffectiveMetaModelPluginManager.getInstance()
				.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
						user.getCurrentKnowledgeSpaceName());
		/**
		 * GJaxbEffectiveDomain effDomain = this.createEffectiveDomainIfNotExist(user,
		 * projectNode, null, user.getCurrentCollaborationName(),
		 * user.getCurrentKnowledgeSpaceName());
		 */

		GJaxbEffectivePackage mcv = effDomain.getEffectivePackage().stream().filter(m -> {
			return RegExpHelper.toRegexFriendlyName(m.getName().toLowerCase())
					.equals(RegExpHelper.toRegexFriendlyName(category.toLowerCase()));
		}).findFirst().get();

		return mcv;
	}

	@POST
	@Path("/downloadRemoteUsecase")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String downloadRemoteUsecaseString(@Auth DWUser user, GJaxbShortUsecaseDefinitionType ucDef)
			throws Exception {
		GJaxbCompleteUsecaseDefinition projectDef = this.downloadRemoteUsecase(user, ucDef);
		return new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(projectDef))
				.getJSONObject("completeUsecaseDefinition").toString();
	}

	public GJaxbCompleteUsecaseDefinition downloadRemoteUsecase(@Auth DWUser user,
			GJaxbShortUsecaseDefinitionType ucDef) throws Exception {
		RIOAbstractProject project = RIOUsecasesManager.getInstance(this.conf.getVersion()).download(ucDef);
		return project.createCompleteUsecaseDefinition();
	}

	@POST
	@Path("/loadProjectFromUsecase")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String loadProjectFromUsecase(@Auth DWUser user, GJaxbCompleteUsecaseDefinition usecase,
			@QueryParam("collaborationName") String collaborationName) throws Exception {
		JSONObject response = new JSONObject();
		assert user != null;
		assert usecase != null;

		LOG.debug("[user]:" + user.getName() + " - [loadProjectFromUsecase] with usecase: " + usecase.getName());
		if (collaborationName != null && !user.getCurrentCollaborationName().equals(collaborationName)) {
			GJaxbNode collab = null;
			List<GJaxbNode> collabs = this.system.findCollaborationsByName(user, collaborationName);
			if (collabs.isEmpty()) {
				collab = SystemResource.createCollaboration(collaborationName, this.system.getSystemClient(),
						this.system.getCollaborationMetaModel());
			} else {
				collab = collabs.get(0);
			}
			this.system.selectCollaboration(user, collab.getId(), collab.getId());
		}

		if (collaborationName == null) {
			collaborationName = user.getCurrentCollaborationName();
		}

		RIOAbstractProject internalProject = RIOUsecasesManager.getInstance(this.conf.getVersion())
				.findAvailableUsecaseByFriendlyName(usecase.getFriendlyName(), true, usecase);

		selectUsedResources(internalProject, usecase);
		GJaxbNode projectNode = loadProject(internalProject, collaborationName, usecase.getName(), user, false, false,
				new ArrayList<String>(), false);

		response.put("projectId", projectNode.getId());
		response.put("projectName", GenericModelHelper.getName(projectNode));
		return response.toString();
	}

	public GJaxbNode loadProject(RIOAbstractProject project, String collaborationName, String knowledgeSpaceName,
			DWUser user, boolean isReference, boolean isSharedResource, List<String> projectsAlreadyLoaded,
			boolean metaModelAlreadyLoaded) throws Exception {
		GJaxbNode projectModel = null;
		try {
			if (!user.isFake()) {
				if (user.getCurrentCollaborationName() == null
						|| !user.getCurrentCollaborationName().equals(collaborationName)) {
					user.setCurrentCollaborationName(collaborationName);
				}
				if (user.getCurrentKnowledgeSpaceName() == null
						|| !user.getCurrentKnowledgeSpaceName().equals(knowledgeSpaceName)) {
					user.setCurrentKnowledgeSpaceName(knowledgeSpaceName);
				}
				this.system.updateUser(user, user.getUser());
			}

			if (project.getProject() != null) {
				LOG.debug("load project: " + project.getUsecaseDef().getName());

				// create project
				projectModel = XMLJAXBContext.getInstance()
						.unmarshallDocument(URI.create(project.getProject().getUrl()).toURL(), GJaxbGenericModel.class)
						.getNode().get(0);

				projectModel.getStatus().remove(GJaxbStatusType.ACTIVE);
				projectModel.getStatus().remove(GJaxbStatusType.FREEZE);
				projectModel.getStatus().add(GJaxbStatusType.ACTIVE);

				if (isReference) {
					projectModel.getStatus().remove(GJaxbStatusType.ACTIVE);
					projectModel.getStatus().add(GJaxbStatusType.REFERENCE);
				}
				if (isSharedResource) {
					projectModel.getStatus().remove(GJaxbStatusType.ACTIVE);
					projectModel.getStatus().add(GJaxbStatusType.SHARE);
				}

				projectModel.setUserData("project", project);
				this._addProject(user, projectModel, collaborationName, knowledgeSpaceName, projectsAlreadyLoaded,
						metaModelAlreadyLoaded);

			} else {
				LOG.debug(
						"load project as reference or share resources knowledge: " + project.getUsecaseDef().getName());
			}

			// copy resources
			this.copyResourcesInRioResourcesFolder(project, user, collaborationName, knowledgeSpaceName);

			// load FREEZE selected models
			List<GJaxbModelURLType> modelsByUrlSelected = new ArrayList<GJaxbModelURLType>();
			project.getMap().get(GJaxbResourceType.MODEL).forEach(r -> {
				if (r.isSelected() && !r.getName().equals("project")) {
					GJaxbModelURLType m = (GJaxbModelURLType) r;
					if (m.getStatus().equals(GJaxbStatusType.FREEZE)) {
						GJaxbModelURLType model = new GJaxbModelURLType();
						model.setName(m.getName());
						model.setUrl(m.getUrl().toString());
						model.setStatus(m.getStatus());
						model.setIsReference(isReference);
						model.setIsSharedResource(isSharedResource);
						model.setCategory(m.getCategory());
						modelsByUrlSelected.add(model);
						LOG.debug("project freeze model to load: " + model.getUrl());
					}
				}
			});
			this.models.addModelsFromURL(Optional.of(user), modelsByUrlSelected);
			LOG.debug("project freeze models loaded: " + modelsByUrlSelected.size());

			modelsByUrlSelected.clear();
			project.getMap().get(GJaxbResourceType.MODEL).forEach(r -> {
				if (r.isSelected() && !r.getName().equals("project")) {
					GJaxbModelURLType m = (GJaxbModelURLType) r;
					if (m.getStatus().equals(GJaxbStatusType.ACTIVE)) {
						GJaxbModelURLType model = new GJaxbModelURLType();
						model.setName(m.getName());
						model.setUrl(m.getUrl().toString());
						model.setStatus(m.getStatus());
						model.setIsReference(isReference);
						model.setIsSharedResource(isSharedResource);
						model.setCategory(m.getCategory());
						modelsByUrlSelected.add(model);
						LOG.debug("project active model to load: " + model.getUrl());
					}
				}
			});
			this.models.addModelsFromURL(Optional.of(user), modelsByUrlSelected);
			LOG.debug("project active models loaded: " + modelsByUrlSelected.size());

			// Re-Interpretre partial 2L DSL
			LLManager.getInstance().analyzePartialInterpretors(collaborationName, knowledgeSpaceName);

			// load selected rules
			// We set URLs of potential additional files HERE because we have the usecase
			// root folder in usecase.getProject().getModelUrl()
			List<GJaxbInterpretationConfig> configs = new ArrayList<GJaxbInterpretationConfig>();
			for (GJaxbResourceURLType r : project.getMap().get(GJaxbResourceType.CEP_RULE)) {
				if (r.isSelected()) {
					GJaxbInterpretationConfig rule = XMLJAXBContext.getInstance()
							.unmarshallDocument(URI.create(r.getUrl()).toURL(), GJaxbInterpretationConfig.class);
					configs.add(rule);
				}
			}

			for (GJaxbResourceURLType r : project.getMap().get(GJaxbResourceType.ML_RULE)) {
				if (r.isSelected()) {
					GJaxbInterpretationConfig rule = XMLJAXBContext.getInstance()
							.unmarshallDocument(URI.create(r.getUrl()).toURL(), GJaxbInterpretationConfig.class);
					configs.add(rule);

				}
			}

			String rulesLoadedResponse = this.interpretationRules.addRules(user, configs);

			// retrieve cache matrix
			for (GJaxbResourceURLType r : project.getMap().get(GJaxbResourceType.DISTANCE_TIME_MATRIX)) {
				GJaxbDistanceTimeMatrix cachedMatrixModel = XMLJAXBContext.getInstance()
						.unmarshallDocument(URI.create(r.getUrl()).toURL(), GJaxbDistanceTimeMatrix.class);
				GlobalDistanceTimeMatrices.set(user.getUser(), cachedMatrixModel);
			}

			// import game scenario
			for (GJaxbResourceURLType r : project.getMap().get(GJaxbResourceType.GAME_SCENARIO)) {
				GJaxbGameScenario gs = XMLJAXBContext.getInstance().unmarshallDocument(URI.create(r.getUrl()).toURL(),
						GJaxbGameScenario.class);
				for (GJaxbSensorControlerBackup rds : gs.getSensorControlerBackup()) {
					GJaxbResourceURLType dr = project.findResourceByName(
							rds.getDatasetInfo().getName().strip().stripIndent(), GJaxbResourceType.DATASET);
					if (dr == null) {
						throw new Exception("Impossible to find: " + rds.getDatasetInfo().getName());
					}
					String buffer = XMLNSCleaner
							.clean(FileUtil.getContents(URI.create(dr.getUrl()).toURL().openStream()));
					byte[] compressedBuffer = GZipUtil.zip(buffer);
					rds.getDatasetInfo()
							.setZip(new DataHandler(new ByteArrayDataSource(compressedBuffer, "application/gzip")));
					rds.getDatasetInfo().setUrl(null);
				}

				GJaxbCreateGameScenario createReq = new GJaxbCreateGameScenario();
				createReq.setCollaborationName(user.getCurrentCollaborationName());
				createReq.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				createReq.setGameScenario(gs);

				this.gmClient.createGameScenario(createReq);
			}

			// restore database
			GJaxbRestoreTimeSeriesDB restoreReq = new GJaxbRestoreTimeSeriesDB();
			for (GJaxbResourceURLType r : project.getMap().get(GJaxbResourceType.BACKUP_DATABASE)) {
				if (r.isSelected() && r.getName().contains("timeseries")) {
					restoreReq.setCollaborationName(user.getCurrentCollaborationName());
					restoreReq.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
					DataHandler dh = new DataHandler(URI.create(r.getUrl()).toURL());
					restoreReq.getZip().add(dh);
				}
			}
			if (!restoreReq.getZip().isEmpty()) {
				GJaxbRestoreTimeSeriesDBResponse restoreResp = this.sensorManagerClient.restoreTimeSeriesDB(restoreReq);
			}

			// restore RAG
			if (project.getMap().get(GJaxbResourceType.RAG).size() > 0) {

				GJaxbRestoreRAGFromDirectory restoreRag = new GJaxbRestoreRAGFromDirectory();
				restoreRag.setContext(new GJaxbContext());
				restoreRag.getContext().setCurrentCollaborationName(user.getCurrentCollaborationName());
				restoreRag.getContext().setCurrentKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());

				for (GJaxbResourceURLType r : project.getMap().get(GJaxbResourceType.RAG)) {
					if (r.isSelected()) {
						URL url = URI.create(r.getUrl()).toURL();
						String fileName = url.getFile().substring(url.getFile().lastIndexOf("/") + "/".length());
						File ragFileInFolder = new File(AIChatbotImpl.getOrCreateRAGDirectory(restoreRag.getContext()),
								fileName);

						FileOutputStream out = new FileOutputStream(ragFileInFolder);
						FileUtil.copy(url.openStream(), out);
						out.close();
					}
				}

				GJaxbRestoreRAGFromDirectoryResponse restoreRagResp = this.aichatBotResource
						.restoreRAGFromDirectory(restoreRag);

				LOG.debug("RAG Initialized: " + JSONJAXBContext.getInstance().marshallAnyElement(restoreRagResp));
			}

			// create sensors if exists
			this.createAllSensors(user);

		} catch (Throwable e) {
			e.printStackTrace();
			if (projectModel != null) {
				this.deleteProject(user, projectModel.getId(), projectModel.getId());
			}
			throw e;
		}
		return projectModel;
	}

	private void createAllSensors(DWUser user) throws Exception {
		List<GJaxbNode> dataSources = this.core.findNodesByTypesAndOrProperties(user,
				Arrays.asList(new QName("http://fr.emac.gind/core-model", "Data_Source")), (List<FindParameters>) null,
				Arrays.asList("ACTIVE"), user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName());

		for (GJaxbNode dataSource : dataSources) {
			GJaxbCreateSensor req = new GJaxbCreateSensor();
			req.setNode(dataSource);
			req.setCollaborationName(user.getCurrentCollaborationName());
			req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			this.sensorManagerResource.createSensor(user, req);
		}
	}

	private void copyResourcesInRioResourcesFolder(RIOAbstractProject project, DWUser user, String collaborationName,
			String knowledgeSpaceName) throws Exception {
		String rootDir = project.getRootDirectory();

		URL root = null;
		LOG.debug("project mode (location): " + project.getUsecaseDef().getLocation());
		if (project.getUsecaseDef().getLocation().equals(GJaxbLocationType.EMBEDDED)) {
			root = Thread.currentThread().getContextClassLoader().getResource(rootDir);
		} else {
			root = new File(rootDir).toURI().toURL();
		}

		if (root != null) {
			LOG.debug("project root: " + root);

			String friendlyCollaborationName = RegExpHelper.toRegexFriendlyName(WORLD_COLLABORATION_NAME);
			if (collaborationName != null) {
				friendlyCollaborationName = RegExpHelper.toRegexFriendlyName(collaborationName).toLowerCase();
			}
			LOG.debug("project friendlyCollaborationName: " + friendlyCollaborationName);
			String friendlyKnowledgeSpaceName = RegExpHelper
					.toRegexFriendlyName(project.getUsecaseDef().getFriendlyName()).toLowerCase();
			if (knowledgeSpaceName != null) {
				friendlyKnowledgeSpaceName = RegExpHelper.toRegexFriendlyName(knowledgeSpaceName).toLowerCase();
			}
			LOG.debug("project friendlyKnowledgeSpaceName: " + friendlyKnowledgeSpaceName);

			java.nio.file.Path myPath;
			if (root != null && root.toURI().getScheme().equals("jar")) {
				FileSystem fileSystem = FileSystems.getFileSystem(root.toURI());
				myPath = fileSystem.getPath(rootDir);
			} else {
				myPath = Paths.get(root.toURI());
			}
			Stream<java.nio.file.Path> walk = Files.walk(myPath);
			for (Iterator<java.nio.file.Path> it = walk.iterator(); it.hasNext();) {
				java.nio.file.Path p = it.next();
				URL url = p.toUri().toURL();
				String shortPath = url.getFile().substring(
						url.getFile().indexOf("/META-INF/resources/") + "/META-INF/resources/".length(),
						url.getFile().length());
				if (shortPath.indexOf(".", shortPath.lastIndexOf("/")) > 0) {
					if (shortPath.contains("/resources/")) {
						String pathAndFileName = shortPath
								.substring(
										shortPath.indexOf(project.getUsecaseDef().getFriendlyName())
												+ project.getUsecaseDef().getFriendlyName().length(),
										shortPath.length());
						String path = pathAndFileName.substring(0, pathAndFileName.lastIndexOf("/") + 1);
						String beginIndex = "-uc-resources/gind/usecases/" + project.getUsecaseDef().getFriendlyName()
								+ "/";
						if (path.indexOf(beginIndex) >= 0) {
							path = path.substring(path.indexOf(beginIndex) + beginIndex.length());
						}
						String fileName = pathAndFileName.substring(pathAndFileName.lastIndexOf("/") + 1,
								pathAndFileName.length());
						File rscDir = new File(RIOConstant.RESOURCES_FOLDER + "/" + friendlyCollaborationName + "/"
								+ friendlyKnowledgeSpaceName + "/" + path);
						rscDir.mkdirs();
						File rsc = new File(rscDir, fileName);
						rsc.createNewFile();
						FileUtils.copyURLToFile(url, rsc);
						/**
						 * LOG.debug("rsc = " + rsc); LOG.debug("--------Affiche copy: ");
						 * LOG.debug(url); LOG.debug("to"); LOG.debug(rsc.getCanonicalFile());
						 */
					} else if (url.toString().contains("/users/user_node_") && url.toString().contains(".")) {
						// load users in contained in jars
						String queryPath = url.toString()
								.substring(url.toString().indexOf("/users/") + "/users/".length());
						File destination = new File("./target/listener", queryPath);
						FileUtils.copyURLToFile(url, destination);

						// load user
						if (url.toString().endsWith(".xml")) {
							GJaxbNode predefinedUser = XMLJAXBContext.getInstance().unmarshallDocument(url,
									GJaxbNode.class);

							GenericModelHelper.findProperty("Presence", predefinedUser.getProperty(), true)
									.setValue("Available");

							GenericModelHelper
									.findProperty("currentCollaborationName", predefinedUser.getProperty(), true)
									.setValue(collaborationName);
							GenericModelHelper
									.findProperty("currentKnowledgeSpaceName", predefinedUser.getProperty(), true)
									.setValue(knowledgeSpaceName);

							GJaxbUpdateUser upReq = new GJaxbUpdateUser();
							upReq.setNode(predefinedUser);
							upReq.setCreateIfNotExist(true);
							this.system.updateUser(user, predefinedUser);

							List<GJaxbNode> collabNodes = this.system.findCollaborationsByName(user, collaborationName);
							GJaxbNode collabNode = collabNodes.get(0);

							GJaxbConnectUserToCollaboration reqConnColl = new GJaxbConnectUserToCollaboration();
							reqConnColl.setCollaborationId(collabNode.getId());
							reqConnColl.setUserId(predefinedUser.getId());
							this.system.connectUserToCollaboration(user, reqConnColl);
						}

					}

				}
			}
			walk.close();
		}
	}

	private void selectUsedResources(RIOAbstractProject internalProject, GJaxbCompleteUsecaseDefinition usecase) {

		if (usecase != null) {
			if (usecase.getResourcesByType() != null && !usecase.getResourcesByType().isEmpty()) {
				internalProject.getMap().values().forEach(rscs -> {
					rscs.forEach(rsc -> {
						rsc.setSelected(false);
					});
				});

				for (ResourcesByType rscsByType : usecase.getResourcesByType()) {
					for (GJaxbResourceURLType rscBrowser : rscsByType.getResources()) {
						GJaxbResourceURLType rsc = internalProject.findResourceByName(rscBrowser.getName(),
								rscsByType.getType());
						if (rscBrowser.isSelected()) {
							rsc.setSelected(true);
						}
					}
				}
			} else {
				internalProject.getMap().values().forEach(rscs -> {
					rscs.forEach(rsc -> {
						rsc.setSelected(true);
					});
				});
			}
		}

	}

	@DELETE
	@Path("/projects/{id}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String deleteProject(@Auth DWUser user, @PathParam("id") String id, @QueryParam("qid") String qid)
			throws Exception {
		JSONObject resp = new JSONObject();
		try {
			assert user != null;

			GJaxbNode project = this.system.getKnowledgeSpace(user, id, qid);
			if (project != null) {
				String projectName = GenericModelHelper.getName(project);

				// undeploy all rules
				this.interpretationRules.deleteRules(user, this.interpretationRules.getRules(user));

				// delete game scenario
				GJaxbDeleteGameScenarios delReq = new GJaxbDeleteGameScenarios();
				delReq.setCollaborationName(user.getCurrentCollaborationName());
				delReq.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				this.gmClient.deleteGameScenarios(delReq);

				// delete process deployed
				List<ProcessListResponse> list = this.process.processDeployedList(user);
				for (ProcessListResponse respP : list) {
					for (GJaxbDeployResult deploy : respP.getDeployedProcesses().getResult()) {
						GJaxbProcessToDeploy sampledProcess = new GJaxbProcessToDeploy();
						sampledProcess.setProcessName(deploy.getServiceQName().getLocalPart());
						this.process.undeploy(user, sampledProcess);
					}
				}

				// delete all nodes in knowledge
				String currentKnowledge = RegExpHelper.toRegexFriendlyName(projectName);
				String currentCollaboration = RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName());
				this.models.query(null,
						"match (n:`" + currentCollaboration + "`:`" + currentKnowledge + "`), (n1:`"
								+ currentCollaboration + "`:`" + currentKnowledge + "`)-[r]-(n2:`"
								+ currentCollaboration + "`:`" + currentKnowledge + "`) detach delete n1, n2, n, r",
						null);

				// add project node to be delete
				this.system.deleteKnowledgeSpace(user, id, qid);
				// models.getNode().add(project);

				// GJaxbSynchronizeRequest m = new GJaxbSynchronizeRequest();
				// m.setJsonModel(JSONJAXBContext.getInstance().marshallAnyElement(models));
				// this.models.deleteModels(user, Arrays.asList(m));

				if (projectName.equals(user.getCurrentKnowledgeSpaceName())) {
					user.setCurrentKnowledgeSpaceName(null);
					this.system.updateUser(user, user.getUser());
				}

				// delete resources folder
				FileUtils.deleteQuietly(new File(RIOConstant.RESOURCES_FOLDER + "/"
						+ RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName()) + "/"
						+ RegExpHelper.toRegexFriendlyName(projectName)));

				resp.put("projectName", projectName);

			}

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return resp.toString();
	}

	@DELETE
	@Path("/resetAll")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String resetAll(@Auth DWUser user) throws Exception {
		JSONObject resp = new JSONObject();
		try {
			assert user != null;

			// undeploy all rules
			this.interpretationRules.deleteRules(user, this.interpretationRules.getRules(user));

			// delete process deployed
			List<ProcessListResponse> list = this.process.processDeployedList(user);
			for (ProcessListResponse respP : list) {
				for (GJaxbDeployResult deploy : respP.getDeployedProcesses().getResult()) {
					GJaxbProcessToDeploy sampledProcess = new GJaxbProcessToDeploy();
					sampledProcess.setProcessName(deploy.getServiceQName().getLocalPart());
					this.process.undeploy(user, sampledProcess);
				}
			}

			// delete all nodes (except users and collaboration)
			GJaxbGenericModel models = this.models.query(null,
					"match (n), (n1)-[r]-(n2) where not n:" + Neo4JReqConstant.system("User") + " and not n1:"
							+ Neo4JReqConstant.system("User") + " and not n2:" + Neo4JReqConstant.system("User")
							+ " and not n:" + Neo4JReqConstant.system("Collaboration") + " and not n1:"
							+ Neo4JReqConstant.system("Collaboration") + " and not n2:"
							+ Neo4JReqConstant.system("Collaboration") + " return n1, n2, n, r",
					null);
			GJaxbSynchronizeRequest m = new GJaxbSynchronizeRequest();
			m.setJsonModel(JSONJAXBContext.getInstance().marshallAnyElement(models));
			this.models.deleteModels(user, Arrays.asList(m));

			// set current knowledge of user to null
			user.setCurrentKnowledgeSpaceName(null);
			this.system.updateUser(null, user.getUser());

			// delete resources folder
			FileUtils.deleteQuietly(new File(RIOConstant.RESOURCES_FOLDER + "/"
					+ RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName())));

			resp.put("result", true);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return resp.toString();
	}

	@POST
	@Path("/exportProject")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public Response exportProject(@Auth DWUser user, ExportProjectRequest request) throws Exception {
		Response response = null;
		try {
			GJaxbNode projModel = this.system.getKnowledgeSpace(user, request.getProjectId(), request.getProjectId());
			if (projModel == null) {
				throw new Exception("Impossible to find project with this id: " + request.getProjectId());
			}

			GJaxbGenericModel projectModel = new GJaxbGenericModel();
			projectModel.setName(request.getProjectName());
			projectModel.getNode().add(projModel);

			String shortProjectId = request.getProjectId();
			if (request.getProjectId().indexOf("_c__") > 0) {
				shortProjectId = shortProjectId.substring(0, request.getProjectId().indexOf("_c__"));
			}
			projectModel.getNode().get(0).setId(shortProjectId);
			Document docProject = XMLJAXBContext.getInstance().marshallAnyElement(projectModel);

			JSONArray array = new JSONArray();

			// extract meta-model if necessary
			GJaxbProperty useDynamicMMProp = GenericModelHelper.findProperty("use dynamic MetaModel",
					projModel.getProperty());
			if (useDynamicMMProp != null && useDynamicMMProp.getValue() != null
					&& useDynamicMMProp.getValue().toLowerCase().trim().equals("yes")) {
				String targetNameSpace = "http://fr.emac.gind/${collab}/"
						+ RegExpHelper.toRegexFriendlyName(user.getCurrentKnowledgeSpaceName());

				GJaxbEffectiveDomain dom = (GJaxbEffectiveDomain) getAndRefreshEffectiveDomainIfNotExist(user,
						projModel, null, user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName(),
						false); // EffectiveMetaModelPluginManager.getInstance().getCacheEffectiveDomainByCollaboration(RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName())).get(RegExpHelper.toRegexFriendlyName(request.getProjectId()));

				GJaxbEffectiveDomain cloned_domain = GenericModelHelper.clone(dom, dom.getClass());

				List<GJaxbEffectivePackage> validEffPacks = new ArrayList<GJaxbEffectivePackage>();
				for (GJaxbEffectivePackage effP : cloned_domain.getEffectivePackage()) {
					GJaxbEffectivePackage effPack = this.keepOnlyAddedClassInPackage(effP,
							user.getCurrentCollaborationName());
					if (!effPack.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getEffectiveConcept()
							.isEmpty()) {
						validEffPacks.add(effPack);
					}

				}

				for (GJaxbEffectivePackage validEffPack : validEffPacks) {
					GJaxbPackage _package = EffectiveMetaModelPluginManager.getInstance()
							.createPackageFromEffectivePackage(targetNameSpace, validEffPack);

					JSONObject fileEffPackage = new JSONObject();
					fileEffPackage.put("fileName",
							"metamodel/" + "Extended_" + _package.getName() + "_Package" + ".xml");
					fileEffPackage.put("fileContent", XMLPrettyPrinter
							.print(XMLJAXBContext.getInstance().marshallAnyElement(_package)).getBytes());
					array.put(fileEffPackage);
				}

				GJaxbMetaModel extendedMetaModel = EffectiveMetaModelPluginManager.getInstance()
						.createMetaModelFromEffectivePackages(targetNameSpace, validEffPacks);
				JSONObject fileEffPackage = new JSONObject();
				fileEffPackage.put("fileName", "metamodel/" + "Extended_" + "Collaborative" + "_MetaModel" + ".xml");
				fileEffPackage.put("fileContent", XMLPrettyPrinter
						.print(XMLJAXBContext.getInstance().marshallAnyElement(extendedMetaModel)).getBytes());
				array.put(fileEffPackage);

				cloned_domain.setName("Collaborative");
				cloned_domain.setGlobalEffectiveMetaModel(null);
				cloned_domain.getEffectivePackage().clear();
				JSONObject fileEffDomain = new JSONObject();
				fileEffDomain.put("fileName", "metamodel/" + "domain.xml");
				fileEffDomain.put("fileContent", XMLPrettyPrinter
						.print(XMLJAXBContext.getInstance().marshallAnyElement(cloned_domain)).getBytes());
				array.put(fileEffDomain);
			}

			// extract active models by category
			Map<String, Document> otherModelDocuments = new HashMap<String, Document>();
			for (GJaxbEffectivePackage mcv : EffectiveMetaModelPluginManager.getInstance()
					.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
							user.getCurrentKnowledgeSpaceName())
					.getEffectivePackage()) {
				String category = mcv.getName();
				GJaxbSynchronizeResponse extractResourcesResp = this.models.extraction(
						user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName(),
						new QName(mcv.getTargetNamespace(), mcv.getName()), null, Arrays.asList(GJaxbStatusType.ACTIVE),
						Arrays.asList(GJaxbStatusType.SHARE), null, null);
				GJaxbGenericModel model = extractResourcesResp.getSyncResponse().getExtractSyncModelResponse()
						.getGenericModel();
				extractSpecificExportDocuments(model, otherModelDocuments, user.getCurrentCollaborationName());

				if (extractResourcesResp.getSyncResponse().getExtractSyncModelResponse().getGenericModel() != null) {
					Document docResources = XMLJAXBContext.getInstance().marshallAnyElement(model);
					otherModelDocuments.put(category, docResources);
				}
			}

			// extract freezed models by category
			for (GJaxbEffectivePackage mcv : EffectiveMetaModelPluginManager.getInstance()
					.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
							user.getCurrentKnowledgeSpaceName())
					.getEffectivePackage()) {
				String category = mcv.getName();
				GJaxbSynchronizeResponse extractResourcesResp = this.models.extraction(
						user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName(),
						new QName(mcv.getTargetNamespace(), mcv.getName()), null, Arrays.asList(GJaxbStatusType.FREEZE),
						Arrays.asList(GJaxbStatusType.SHARE), null, null);
				GJaxbGenericModel model = extractResourcesResp.getSyncResponse().getExtractSyncModelResponse()
						.getGenericModel();
				extractSpecificExportDocuments(model, otherModelDocuments, user.getCurrentCollaborationName());

				if (extractResourcesResp.getSyncResponse().getExtractSyncModelResponse().getGenericModel() != null) {
					Document docResources = XMLJAXBContext.getInstance().marshallAnyElement(model);
					otherModelDocuments.put("freezed_" + category, docResources);
				}
			}

			JSONObject fileProject = new JSONObject();
			fileProject.put("fileName", "models/project.xml");
			fileProject.put("fileContent", XMLPrettyPrinter.print(docProject).getBytes());
			array.put(fileProject);

			for (Entry<String, Document> entry : otherModelDocuments.entrySet()) {
				JSONObject fileResources = new JSONObject();
				fileResources.put("fileName", "models/" + entry.getKey().toLowerCase() + ".xml");
				fileResources.put("fileContent", XMLPrettyPrinter.print(entry.getValue()).getBytes());
				array.put(fileResources);
			}

			// get specific users from person
			GJaxbGenericModel users = this.models.query(user,
					"match (personOrCommunity:ACTIVE)-[:`" + GenericModelHelper.collab("Assigned_To") + "`]->(u:`"
							+ GenericModelHelper.system("User") + "`) return distinct u",
					null);
			for (GJaxbNode userN : users.getNode()) {
				System.out.println("user name: " + GenericModelHelper.getName(userN));

				Document docUser = XMLJAXBContext.getInstance().marshallAnyElement(userN);

				JSONObject userResources = new JSONObject();
				userResources.put("fileName", "users/" + userN.getId() + "/" + userN.getId() + ".xml");
				userResources.put("fileContent", XMLPrettyPrinter.print(docUser).getBytes());
				array.put(userResources);

				GJaxbProperty pictureProp = GenericModelHelper.findProperty("picture", userN.getProperty());
				if (pictureProp != null && pictureProp.getValue() != null
						&& !pictureProp.getValue().contains("/share/assets")) {

					String pictureFile = pictureProp.getValue()
							.substring(
									pictureProp.getValue().indexOf(RIOConstant.WEBJARS_RESOURCES_FOLDER)
											+ RIOConstant.WEBJARS_RESOURCES_FOLDER.length(),
									pictureProp.getValue().length());
					String fileName = pictureFile.replace("\\", "/");

					String path = RIOConstant.RESOURCES_FOLDER + "/" + pictureFile;

					JSONObject fileResources = new JSONObject();

					fileResources.put("fileName", fileName);
					fileResources.put("fileContent", FileUtils.readFileToByteArray(new File(path)));
					array.put(fileResources);
				}
			}

			// get interpretation_rules
			List<GJaxbInterpretationConfig> interpretaionConfigs = this.interpretationRules.getRules(user);
			for (GJaxbInterpretationConfig rule : interpretaionConfigs) {
				LOG.debug("exporting rule " + rule.getName());
				
				List<GJaxbEventAction> eventActionsCloned = new ArrayList<GJaxbEventAction>();
				for(GJaxbEventAction ea: rule.getEventAction()) {
					GJaxbEventAction eaCloned = GenericModelHelper.clone(ea, GJaxbEventAction.class);
					eventActionsCloned.add(eaCloned);
				}
				
				rule.getEventAction().forEach(ea -> {
					ea.getPublishDestination().removeIf(dest -> {
						return dest.contains("HighLevelInterpretationNotifier");
					});
				});

				String pathOfRules = "interpretation_rules/";
				if (rule.getMlConfig() != null) {
					pathOfRules += "ml_rules/" + RegExpHelper.toRegexFriendlyName(rule.getName()) + "/resources/";
					JSONObject fileResources = new JSONObject();
					fileResources.put("fileName", pathOfRules + "mlConfig.xml");
					fileResources.put("fileContent",
							XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(rule)).getBytes());
					array.put(fileResources);

				} else {
					pathOfRules += "cep_rules/";
					JSONObject fileResources = new JSONObject();
					fileResources.put("fileName",
							pathOfRules + RegExpHelper.toRegexFriendlyName(rule.getId()) + ".xml");
					fileResources.put("fileContent",
							XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(rule)).getBytes());
					array.put(fileResources);
				}

				rule.getEventAction().clear();
				rule.getEventAction().addAll(eventActionsCloned);
			}

			// get distance matrix cached paths
			DistanceTimeMatrixManager matrixCache = GlobalDistanceTimeMatrices.get(user.getCurrentCollaborationName(),
					user.getCurrentKnowledgeSpaceName());
			if (matrixCache != null) {
				GJaxbDistanceTimeMatrix matrixCacheModel = matrixCache.getCachedMatrix();
				JSONObject fileResources = new JSONObject();
				fileResources.put("fileName", CACHE_MATRIX_PATH);
				fileResources.put("fileContent", XMLPrettyPrinter
						.print(XMLJAXBContext.getInstance().marshallAnyElement(matrixCacheModel)).getBytes());
				array.put(fileResources);
			}

			// export game scenario
			String pathOfGameScenarios = "game_scenarios/";
			GJaxbExportGameScenarios exportReq = new GJaxbExportGameScenarios();
			exportReq.setCollaborationName(user.getCurrentCollaborationName());
			exportReq.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			GJaxbExportGameScenariosResponse exportResp = this.gmClient.exportGameScenarios(exportReq);

			for (GJaxbGameScenario gs : exportResp.getGameScenario()) {
				gs.setMongoDBId(null);
				for (GJaxbSensorControlerBackup rds : gs.getSensorControlerBackup()) {
					if (rds.getDatasetInfo() != null) {
						DataHandler dh = rds.getDatasetInfo().getZip();

						String buffer = GZipUtil.unzip(dh.getDataSource().getInputStream().readAllBytes());
						String datasetName = rds.getDatasetInfo().getName();
						rds.getDatasetInfo().setZip(null);
						rds.getDatasetInfo().setUrl("./datasets/" + gs.getScenarioId() + "/" + datasetName + ".xml");

						JSONObject fileResources = new JSONObject();
						String datasetFileName = pathOfGameScenarios + "datasets/" + gs.getScenarioId() + "/"
								+ datasetName + ".xml";
						if (!this.containsFile(array, datasetFileName)) {
							fileResources.put("fileName", datasetFileName);
							fileResources.put("fileContent", buffer.getBytes());
							array.put(fileResources);
						}
					}
				}

				String gsFileName = pathOfGameScenarios + gs.getName() + ".xml";
				if (!this.containsFile(array, gsFileName)) {
					JSONObject fileResources = new JSONObject();
					fileResources.put("fileName", gsFileName);
					fileResources.put("fileContent",
							XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(gs)).getBytes());
					array.put(fileResources);
				}
			}

			// backup database if admin
			if (user.isAdmin() && request.getOptions().getSaveTimeSeriesDb() == true) {

				// backup timeseries database
				GJaxbBackupTimeSeriesDB backupReq = new GJaxbBackupTimeSeriesDB();
				backupReq.setCollaborationName(user.getCurrentCollaborationName());
				backupReq.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				GJaxbBackupTimeSeriesDBResponse backupResp = this.sensorManagerClient.backupTimeSeriesDB(backupReq);
				for (DataHandler dh : backupResp.getZip()) {
					String pathOfDatabase = "database/";
					JSONObject fileResources = new JSONObject();
					String tsSimpleName = dh.getName().substring(dh.getName().lastIndexOf("/") + "/".length(),
							dh.getName().length());
					String tsDbFileName = pathOfDatabase + "backup/timeseries/" + tsSimpleName;
					fileResources.put("fileName", tsDbFileName);
					fileResources.put("fileContent", IOUtils.toByteArray(dh.getInputStream()));
					array.put(fileResources);
				}

			}

			// find resources in resourceFolder
			File rootDirectory = new File(RIOConstant.RESOURCES_FOLDER);
			rootDirectory = new File(rootDirectory,
					RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName()));
			rootDirectory = new File(rootDirectory, RegExpHelper.toRegexFriendlyName(request.getProjectName()));
			IOFileFilter filter = new IOFileFilter() {

				@Override
				public boolean accept(File dir, String name) {
					if (name.startsWith(".")) {
						return false;
					}
					return true;
				}

				@Override
				public boolean accept(File file) {
					if (file.getName().startsWith(".")) {
						return false;
					}
					return true;
				}
			};

			if (rootDirectory.exists() && rootDirectory.isDirectory()) {
				Collection<File> files = FileUtils.listFilesAndDirs(rootDirectory, filter, filter);
				if (files != null && files.size() > 0) {
					for (File file : files) {
						JSONObject fileResources = new JSONObject();
						if (file.isFile()) {
							String path = file.getPath().substring(
									file.getPath().indexOf(RegExpHelper.toRegexFriendlyName(request.getProjectName()))
											+ RegExpHelper.toRegexFriendlyName(request.getProjectName()).length(),
									file.getPath().indexOf(file.getName()));
							path = path.replace("\\", "/");
							fileResources.put("fileName", path + file.getName());
							fileResources.put("fileContent", FileUtils.readFileToByteArray(file));
							array.put(fileResources);
						}
					}
				}
			}

			// finally create the zip
			StreamingOutput stream = new StreamingOutput() {
				@Override
				public void write(OutputStream output) throws IOException, WebApplicationException {
					ZipOutputStream zos = new ZipOutputStream(output);

					for (int i = 0; i < array.length(); i++) {
						ZipEntry ze = new ZipEntry(array.getJSONObject(i).getString("fileName"));
						zos.putNextEntry(ze);
						zos.write((byte[]) array.getJSONObject(i).get("fileContent"));
						zos.closeEntry();
					}

					zos.flush();
					zos.close();
				}
			};

			response = Response.ok(stream)
					.header("Content-Disposition",
							"attachment; filename="
									+ RegExpHelper.toRegexFriendlyName(request.getProjectName()).toLowerCase() + ".zip")
					.header("Content-Transfer-Encoding", "binary").build();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	private GJaxbEffectivePackage keepOnlyAddedClassInPackage(GJaxbEffectivePackage effP, String collaborationName)
			throws Exception {
		GJaxbEffectivePackage neffP = new GJaxbEffectivePackage();
		neffP.setName(effP.getName());
		neffP.setEffectiveMetaModelOfPackage(new EffectiveMetaModelOfPackage());
		neffP.getEffectiveMetaModelOfPackage().setEffectiveMetaModel(new GJaxbEffectiveMetaModel());
		neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
				.setName(effP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getName());
		neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().setRules(new GJaxbRules());
		neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getRules()
				.setConnectionRules(new GJaxbConnectionRules());
		neffP.setRules(new Rules());
		neffP.getRules().setContainmentRule(new GJaxbContainmentRule());

		for (GJaxbEffectiveConceptType concept : effP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
				.getEffectiveConcept()) {
			if (concept.getType().getNamespaceURI().contains(RegExpHelper.toRegexFriendlyName(collaborationName))) {
				neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getEffectiveConcept().add(concept);
			}
		}

		for (GJaxbConnectionRule rule : effP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getRules()
				.getConnectionRules().getConnectionRule()) {
			GJaxbEffectiveConceptType from = EffectiveMetaModelPluginManager.getInstance()
					.getConceptByType(rule.getFrom());
			GJaxbEffectiveConceptType to = EffectiveMetaModelPluginManager.getInstance().getConceptByType(rule.getTo());
			if (from != null || to != null) {
				for (GJaxbEffectiveConceptType concept : neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
						.getEffectiveConcept()) {
					if (concept.getType().equals(rule.getFrom()) || concept.getType().equals(rule.getTo())) {
						neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getRules().getConnectionRules()
								.getConnectionRule().add(rule);
					}
				}
			}
		}

		// remove collaboration in qname
		for (GJaxbEffectiveConceptType concept : neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
				.getEffectiveConcept()) {
			if (concept.getType().getNamespaceURI().contains(RegExpHelper.toRegexFriendlyName(collaborationName))) {
				QName newType = new QName(concept.getType().getNamespaceURI()
						.replace(RegExpHelper.toRegexFriendlyName(collaborationName), "${collab}"),
						concept.getType().getLocalPart());
				concept.setType(newType);
			}
		}

		for (GJaxbConnectionRule rule : neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel().getRules()
				.getConnectionRules().getConnectionRule()) {
			if (rule.getRelationType().getNamespaceURI()
					.contains(RegExpHelper.toRegexFriendlyName(collaborationName))) {
				QName newType = new QName(
						rule.getRelationType().getNamespaceURI()
								.replace(RegExpHelper.toRegexFriendlyName(collaborationName), "${collab}"),
						rule.getRelationType().getLocalPart());
				rule.setRelationType(newType);
			}

			if (rule.getFrom().getNamespaceURI().contains(RegExpHelper.toRegexFriendlyName(collaborationName))) {
				QName newType = new QName(rule.getFrom().getNamespaceURI()
						.replace(RegExpHelper.toRegexFriendlyName(collaborationName), "${collab}"),
						rule.getFrom().getLocalPart());
				rule.setFrom(newType);
			}
			if (rule.getTo().getNamespaceURI().contains(RegExpHelper.toRegexFriendlyName(collaborationName))) {
				QName newType = new QName(rule.getTo().getNamespaceURI().replace(
						RegExpHelper.toRegexFriendlyName(collaborationName), "${collab}"), rule.getTo().getLocalPart());
				rule.setTo(newType);
			}
		}

		for (GJaxbEffectiveConceptType concept : neffP.getEffectiveMetaModelOfPackage().getEffectiveMetaModel()
				.getEffectiveConcept()) {
			if (!concept.isAbstract()) {
				GJaxbContainmentRule.Contain _contain = new GJaxbContainmentRule.Contain();
				_contain.setType(concept.getType());
				neffP.getRules().getContainmentRule().getContain().add(_contain);
			}
		}

		return neffP;
	}

	private void extractSpecificExportDocuments(GJaxbGenericModel model, Map<String, Document> otherDocuments,
			String collaborationName) throws Exception {
		Map<String, GJaxbGenericModel> newExportedModels = new HashMap<String, GJaxbGenericModel>();

		for (GJaxbNode n : model.getNode()) {

			// replace collab by generic key $collab
			if (n.getType().getNamespaceURI().contains(RegExpHelper.toRegexFriendlyName(collaborationName))) {
				QName newType = new QName(n.getType().getNamespaceURI().replace(
						RegExpHelper.toRegexFriendlyName(collaborationName), "${collab}"), n.getType().getLocalPart());
				n.setType(newType);
			}

			if (n.isSetSpecificExportPackage() && !n.getSpecificExportPackage().isBlank()) {
				GJaxbGenericModel newModel = newExportedModels.get(n.getSpecificExportPackage());
				if (newModel == null) {
					newModel = new GJaxbGenericModel();
					newExportedModels.put(n.getSpecificExportPackage(), newModel);
				}
				newModel.getNode().add(n);
			}
		}

		for (GJaxbEdge e : model.getEdge()) {
			if (e.isSetSpecificExportPackage() && !e.getSpecificExportPackage().isBlank()) {
				GJaxbGenericModel newModel = newExportedModels.get(e.getSpecificExportPackage());
				if (newModel == null) {
					newModel = new GJaxbGenericModel();
					newExportedModels.put(e.getSpecificExportPackage(), newModel);
				}
				newModel.getEdge().add(e);
			}
		}

		for (GJaxbGenericModel newModel : newExportedModels.values()) {
			model.getNode().removeIf(n -> GenericModelHelper.contains(n, newModel.getNode()));
			model.getEdge().removeIf(e -> GenericModelHelper.contains(e, newModel.getEdge()));
		}

		for (Entry<String, GJaxbGenericModel> entry : newExportedModels.entrySet()) {
			otherDocuments.put(entry.getKey(), XMLJAXBContext.getInstance().marshallAnyElement(entry.getValue()));
		}

	}

	private boolean containsFile(JSONArray array, String fileName) {
		for (int i = 0; i < array.length(); i++) {
			JSONObject rsc = array.getJSONObject(i);
			if (rsc.getString("fileName").equals(fileName)) {
				return true;
			}
		}
		return false;
	}

	@POST
	@Path("/importProject")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Produces(MediaType.APPLICATION_JSON)
	public String importProject(@Auth DWUser user, @FormDataParam("file") InputStream is,
			@FormDataParam("file") FormDataContentDisposition fileDetail, @QueryParam("data") String data)
			throws Exception {
		JSONObject response = new JSONObject();
		String projectName = fileDetail.getFileName().substring(0, fileDetail.getFileName().lastIndexOf("."));
		projectName = projectName.replaceAll("(\\s\\(\\d\\))+", "");
		String newKnowledgeSpaceName = RegExpHelper.toRegexFriendlyName(projectName);

		File userProjectDir = new File(
				RIOConstant.RESOURCES_FOLDER + "/" + "users" + "/" + user.getUserId() + "/" + newKnowledgeSpaceName);
		ZipUtil.unZip(is, userProjectDir);

		RIOAbstractProject internalProject = new RIOAbstractProject(userProjectDir.toString()) {

			@Override
			public void beforeInit() throws Exception {
			}

			@Override
			public void afterInit() throws Exception {

			}

		};
		internalProject.getUsecaseDef().setLocation(GJaxbLocationType.DOWNLOADED);

		// load all
		internalProject.getMap().forEach((k, v) -> {
			v.forEach(r -> {
				r.setSelected(true);
			});
		});

		GJaxbNode projectNode = loadProject(internalProject, user.getCurrentCollaborationName(),
				internalProject.getUsecaseDef().getName(), user, false, false, new ArrayList<String>(), false);
		response.put("projectId", projectNode.getId());
		response.put("projectName", GenericModelHelper.getName(projectNode));

		// delete temp project user directory
		FileUtils.deleteQuietly(userProjectDir);

		return response.toString();
	}

	@GET
	@Path("/exportExcelData/{projectId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public Response exportExcelData(@Auth DWUser user, @PathParam("projectId") String projectId,
			@QueryParam("projectName") String projectName) throws Exception {

		GJaxbGenericModel projectModel = this.models.query(user,
				"Match (n:`" + user.getCurrentCollaborationName() + "`:`" + user.getCurrentKnowledgeSpaceName()
						+ "` { type : '{" + this.projectEffectiveModeler.getTargetNamespace()
						+ "}Project'}) where n.modelNodeId =~ '.*" + projectId + ".*' return n;",
				null);
		projectModel.setName(GenericModelHelper.getName(projectModel.getNode().get(0)));
		// projectModel.setFromMetaModel(this.projectEffectiveModeler.getName());

		String shortProjectId = RegExpHelper.toRegexFriendlyName(projectId);
		/**
		 * if(shortProjectId.indexOf("_c__") > 0) { shortProjectId =
		 * shortProjectId.substring(0, shortProjectId.indexOf("_c__")); }
		 */

		// publish others models
		Map<String, GJaxbGenericModel> otherDocuments = new HashMap<String, GJaxbGenericModel>();
		for (GJaxbEffectivePackage mcv : EffectiveMetaModelPluginManager.getInstance()
				.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
						user.getCurrentKnowledgeSpaceName())
				.getEffectivePackage()) {

			String category = mcv.getName();
			GJaxbSynchronizeResponse extractResourcesResp = this.models.extraction(user.getCurrentCollaborationName(),
					user.getCurrentKnowledgeSpaceName(), new QName(mcv.getTargetNamespace(), mcv.getName()), null,
					Arrays.asList(GJaxbStatusType.ACTIVE), Arrays.asList(GJaxbStatusType.SHARE), null, null);
			otherDocuments.put(category,
					extractResourcesResp.getSyncResponse().getExtractSyncModelResponse().getGenericModel());
		}

		List<GJaxbEffectiveMetaModel> effmodels = new ArrayList<GJaxbEffectiveMetaModel>();
		for (GJaxbEffectivePackage mcv : EffectiveMetaModelPluginManager.getInstance()
				.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
						user.getCurrentKnowledgeSpaceName())
				.getEffectivePackage()) {
			if (!mcv.getName().toLowerCase().equals(PluginCollaborativeModel.PROCESS_PACKAGE_NAME.toLowerCase())) {
				effmodels.add(mcv.getEffectiveMetaModelOfPackage().getEffectiveMetaModel());
			}
		}

		ExcelModelWriter writer = new ExcelModelWriter(
				effmodels.toArray(new GJaxbEffectiveMetaModel[effmodels.size()]));
		ByteArrayOutputStream os = writer.writeInByteArray(
				otherDocuments.values().toArray(new GJaxbGenericModel[otherDocuments.values().size()]));

		return Response.ok(os.toByteArray(), MediaType.APPLICATION_OCTET_STREAM).build();
	}

	@POST
	@Path("/importExcelData")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Produces(MediaType.APPLICATION_JSON)
	public String importExcelData(@Auth DWUser user, @FormDataParam("file") InputStream is,
			@FormDataParam("file") FormDataContentDisposition fileDetail, @QueryParam("data") String data)
			throws Exception {
		JSONObject response = new JSONObject();

		JSONObject dataJson = new JSONObject(data);
		String projectId = dataJson.get("projectId").toString();

		String shortProjectId = projectId;
		if (projectId.indexOf("_c__") > 0) {
			shortProjectId = shortProjectId.substring(0, projectId.indexOf("_c__"));
		}

		List<GJaxbEffectiveMetaModel> effmodels = new ArrayList<GJaxbEffectiveMetaModel>();
		for (GJaxbEffectivePackage mcv : EffectiveMetaModelPluginManager.getInstance()
				.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
						user.getCurrentKnowledgeSpaceName())
				.getEffectivePackage()) {
			if (!mcv.getName().toLowerCase().equals(PluginCollaborativeModel.PROCESS_PACKAGE_NAME.toLowerCase())) {
				effmodels.add(mcv.getEffectiveMetaModelOfPackage().getEffectiveMetaModel());
			}
		}

		ExcelModelReader excelReader = new ExcelModelReader(
				effmodels.toArray(new GJaxbEffectiveMetaModel[effmodels.size()]));
		Map<String, GJaxbGenericModel> dataModels = excelReader.read(is);

		for (Entry<String, GJaxbGenericModel> dataModel : dataModels.entrySet()) {
			// Publish models
			this.models.publish(user.getCurrentCollaborationName(), user.getCurrentKnowledgeSpaceName(),
					dataModel.getValue(), null, null);
		}

		return response.toString();
	}

	@GET
	@Path("/containsRangeOnMetricsOrIndicators/{id}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String containsRangeOnMetricsOrIndicators(@Auth DWUser user, @PathParam("id") String id,
			@QueryParam("qid") String qid) throws Exception {
		JSONObject res = new JSONObject();

		String collaborationName = user.getCurrentCollaborationName();
		String knowledgeSpaceName = user.getCurrentKnowledgeSpaceName();

		GJaxbNode project = null;
		try {
			project = this.system.getKnowledgeSpace(user, id, qid);
			IndicatorManager indManager = new IndicatorManager(project);

			final List<String> metricsOrIndicatorsProperties = new ArrayList<String>();
			GJaxbEffectiveDomain effDomain = EffectiveMetaModelPluginManager.getInstance()
					.getCacheEffectiveDomainByCollaborationAndKnowledge(user.getCurrentCollaborationName(),
							user.getCurrentKnowledgeSpaceName());
			GJaxbEffectiveMetaModel effMeta = effDomain.getGlobalEffectiveMetaModel().getEffectiveMetaModel();
			effMeta.getEffectiveConcept().forEach(n -> {
				n.getMetaProperty().forEach(p -> {
					if (p.getType() == GJaxbFormtypeType.METRIC) {
						if (!metricsOrIndicatorsProperties.contains(p.getName())) {
							metricsOrIndicatorsProperties.add(p.getName());
						}
					}
				});
			});

			indManager.getIndicators().keySet().forEach(indicatorName -> {
				metricsOrIndicatorsProperties.add(indicatorName);
			});

			final StringBuffer condition = new StringBuffer();
			metricsOrIndicatorsProperties.forEach(p -> {
				condition.append("(" + "exists(c.`property_" + p + "`) and " + "c.`property_" + p
						+ "` contains 'ranges'" + ") or ");
			});

			// create new association
			GJaxbGenericModel rangesModel = this.models.query(user,
					"match (c:ACTIVE:`" + RegExpHelper.toRegexFriendlyName(collaborationName) + "`:`"
							+ RegExpHelper.toRegexFriendlyName(knowledgeSpaceName) + "` ) where "
							+ condition.toString().substring(0, condition.toString().length() - " or ".length())
							+ " return c;",
					null);
			if (!rangesModel.getNode().isEmpty()) {
				res.put("hasRanges", true);
			} else {
				res.put("hasRanges", false);
			}

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}

		return res.toString();
	}

}
