/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.xml.namespace.QName;

import org.glassfish.jersey.media.multipart.FormDataBodyPart;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import fr.emac.gind.commons.utils.color.ColorHelper;
import fr.emac.gind.commons.utils.net.IPUtil;
import fr.emac.gind.commons.utils.net.IPUtil.TRANSPORT_PROTOCOL;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.commons.utils.xml.XMLGregorianCalendarHelper;
import fr.emac.gind.commons.utils.xml.XMLPrettyPrinter;
import fr.emac.gind.event.consumer.AbstractNotifierClient;
import fr.emac.gind.event.consumer.NotificationConsumerWebService;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.RestWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.json_connector.GJaxbData;
import fr.emac.gind.json_connector.GJaxbPoint;
import fr.emac.gind.json_connector.GJaxbPosition;
import fr.emac.gind.json_connector.GJaxbSensorEvent;
import fr.emac.gind.json_connector.GJaxbSensorNatureSocialType;
import fr.emac.gind.json_connector.GJaxbSensorType;
import fr.emac.gind.json_connector.GJaxbData.Social;
import fr.emac.gind.json_connector.GJaxbData.Social.Value;
import fr.emac.gind.json_connector.GJaxbSensorEvent.SensorNature;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.modeler.genericmodel.GJaxbStatusType;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.rio.dw.resources.FileResource;
import fr.emac.gind.sensors.manager.SensorManager;
import fr.emac.gind.sensors.manager.SensorManagerClient;
import fr.emac.gind.sensors.manager.SensorUIConfigurationHelper;
import fr.emac.gind.sensors.manager.data.GJaxbContinuousSignalInput;
import fr.emac.gind.sensors.manager.data.GJaxbContinuousSignalIsStartedOnJSONConnector;
import fr.emac.gind.sensors.manager.data.GJaxbContinuousSignalIsStartedOnJSONConnectorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbContinuousSignalOutput;
import fr.emac.gind.sensors.manager.data.GJaxbCreateSensor;
import fr.emac.gind.sensors.manager.data.GJaxbCreateSensorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbDeleteSensor;
import fr.emac.gind.sensors.manager.data.GJaxbDeleteSensorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbFindSensorByInternalUrl;
import fr.emac.gind.sensors.manager.data.GJaxbFindSensorByInternalUrlResponse;
import fr.emac.gind.sensors.manager.data.GJaxbFindSensorByProtocol;
import fr.emac.gind.sensors.manager.data.GJaxbFindSensorByProtocolResponse;
import fr.emac.gind.sensors.manager.data.GJaxbFindSensorByTopic;
import fr.emac.gind.sensors.manager.data.GJaxbFindSensorByTopicResponse;
import fr.emac.gind.sensors.manager.data.GJaxbGetSensor;
import fr.emac.gind.sensors.manager.data.GJaxbGetSensorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbGetSensors;
import fr.emac.gind.sensors.manager.data.GJaxbGetSensorsResponse;
import fr.emac.gind.sensors.manager.data.GJaxbListen;
import fr.emac.gind.sensors.manager.data.GJaxbListen.Payload;
import fr.emac.gind.sensors.manager.data.GJaxbListenResponse;
import fr.emac.gind.sensors.manager.data.GJaxbPauseContinuousSignalOnJSONConnector;
import fr.emac.gind.sensors.manager.data.GJaxbPauseContinuousSignalOnJSONConnectorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbSetLocked;
import fr.emac.gind.sensors.manager.data.GJaxbSetLockedResponse;
import fr.emac.gind.sensors.manager.data.GJaxbStartContinuousSignalOnJSONConnector;
import fr.emac.gind.sensors.manager.data.GJaxbStartContinuousSignalOnJSONConnectorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbStartSensor;
import fr.emac.gind.sensors.manager.data.GJaxbStartSensorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbStopSensor;
import fr.emac.gind.sensors.manager.data.GJaxbStopSensorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbUpdateConfigOfSensor;
import fr.emac.gind.sensors.manager.data.GJaxbUpdateConfigOfSensorResponse;
import fr.emac.gind.sensors.manager.data.GJaxbUpdateContinuousSignalOnJSONConnector;
import fr.emac.gind.sensors.manager.data.GJaxbUpdateContinuousSignalOnJSONConnectorResponse;
import fr.emac.gind.sensors.model.GJaxbSensorModel;
import fr.emac.gind.sensors.model.GJaxbSensorUIConfiguration;
import fr.emac.gind.tweet.GJaxbMediaObject;
import fr.emac.gind.tweet.GJaxbTweetObject;
import fr.emac.gind.tweet.ObjectFactory;
import fr.emac.gind.websocket.command.WebsocketCommand;
import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/sensorManager")
//@Api("eventSensor")
@Produces(MediaType.APPLICATION_JSON)
public class SensorManagerResource {

	private static Logger LOG = LoggerFactory.getLogger(SensorManagerResource.class);

	private SensorManager sensorManagerClient = null;
	private FileResource fileResource = null;
	private NotificationConsumerWebService logConsumer = null;
	private Map<QName, String> subscriptionIds = Collections.synchronizedMap(new HashMap<QName, String>());
	private AbstractNotifierClient notifier = null;
	private WebsocketCommand webSocketCommand = null;
	private Configuration conf = null;

	public SensorManagerResource(Configuration conf, FileResource fileResource, NotificationConsumerWebService consumer,
			AbstractNotifierClient notifier, WebsocketCommand command) throws Exception {
		this.sensorManagerClient = SensorManagerClient.createClient(conf.getProperties().get("sensor-manager-server"));
		this.fileResource = fileResource;
		this.logConsumer = consumer;
		this.notifier = notifier;
		this.webSocketCommand = command;
		this.conf = conf;

		if (this.logConsumer == null) {
			throw new Exception("LogConsumer cannot be null !!!");
		}
	}

	@POST
	@Path("/createSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbCreateSensorResponse createSensor(@Auth DWUser user, GJaxbCreateSensor req) throws Exception {
		GJaxbCreateSensorResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			String nature = GenericModelHelper.findProperty("nature", req.getNode().getProperty()).getValue();
			if (nature.toLowerCase().equals("twitter")) {

				String twitterKey = createTwitterKey(user);

				req.setContext(new GJaxbCreateSensor.Context());
				req.getContext().getProperty().add(GenericModelHelper.createProperty("twitterKey", twitterKey));
			}
			response = createEventSensor(this.sensorManagerClient, this.fileResource, user, req);

			if (req.isSetStartSensor() && req.isStartSensor()) {
				for (QName topicProvided : response.getSensorModel().getTopicsProvided()) {
					String subscriptionId = logConsumer.getImplementation(this.notifier.getClass())
							.subscribeOn(response.getSensorModel().getInternalUrl(), topicProvided);
					subscriptionIds.put(topicProvided, subscriptionId);
				}

				JSONObject method = new JSONObject();
				method.put("method_name", "startSensor");
				method.put("method_status", "SUCCESSFUL");
//				sendEventOnRioseAndOrRiosemit(user, response.getSensorModel(), method, req.getCollaborationName(),
//						req.getKnowledgeSpaceName());
			}

		} catch (Exception e) {
			if (req.isSetStartSensor() && req.isStartSensor()) {
				JSONObject method = new JSONObject();
				method.put("method_name", "startSensor");
				method.put("method_status", "ERROR");

				GJaxbSensorModel agent = new GJaxbSensorModel();
				agent.setName(GenericModelHelper.findProperty("name", req.getNode().getProperty()).getValue());
				agent.setStatus(null);
//				sendEventOnRioseAndOrRiosemit(user, agent, method, req.getCollaborationName(),
//						req.getKnowledgeSpaceName());
			}
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	public static String createTwitterKey(DWUser user) throws Exception {
		String twitterKey = null;
		JSONArray twitterAPI = new JSONArray(user.getPropertyValue("Twitter API"));
		if (twitterAPI.length() > 0) {
			List<GJaxbProperty> tweetProps = GenericModelHelper
					.convertJSONArrayToPropertyList(twitterAPI.getJSONArray(0));

			String consumer_key = GenericModelHelper.findProperty("consumer key", tweetProps).getValue();
			String consumer_secret = GenericModelHelper.findProperty("consumer secret", tweetProps).getValue();
			String token = GenericModelHelper.findProperty("token", tweetProps).getValue();
			String token_secret = GenericModelHelper.findProperty("token secret", tweetProps).getValue();

			if (consumer_key != null && consumer_secret != null && token != null && token_secret != null) {
				twitterKey = Base64.getEncoder().encodeToString(
						new String(consumer_key + "@@@" + consumer_secret + "@@@" + token + "@@@" + token_secret)
								.getBytes());
			}

		} else if (System.getenv("TWITTER_CONSUMER_KEY") != null && System.getenv("TWITTER_CONSUMER_SECRET") != null
				&& System.getenv("TWITTER_TOKEN") != null && System.getenv("TWITTER_TOKEN_SECRET") != null) {
			twitterKey = Base64.getEncoder()
					.encodeToString(new String(System.getenv("TWITTER_CONSUMER_KEY") + "@@@"
							+ System.getenv("TWITTER_CONSUMER_SECRET") + "@@@" + System.getenv("TWITTER_TOKEN") + "@@@"
							+ System.getenv("TWITTER_TOKEN_SECRET")).getBytes());
		}

		if (twitterKey == null) {
			throw new Exception(
					"twitterKey cannot be null. Add twitter informations in your r-io account or in environment variable");
		}
		return twitterKey;
	}

	public static GJaxbCreateSensorResponse createEventSensor(SensorManager epsClient, FileResource fileResource,
			DWUser user, GJaxbCreateSensor req) throws Exception {

		String uiConfigurationURL = GenericModelHelper.findProperty("ui configuration", req.getNode().getProperty(), true)
				.getValue();
		
		if (uiConfigurationURL != null && !uiConfigurationURL.isBlank()) {
			uiConfigurationURL = uiConfigurationURL.replace("resourcesFolder/",
					"resourcesFolder/" + RegExpHelper.toRegexFriendlyName(user.getCurrentCollaborationName()) + "/");
			GJaxbSensorUIConfiguration uiConfiguration = getUIConfiguration(fileResource, user, uiConfigurationURL);
			req.setSensorUIConfiguration(uiConfiguration);
		} else {
			GJaxbSensorUIConfiguration uiConfiguration = createUIConfiguration(fileResource, user, req.getNode());
			req.setSensorUIConfiguration(uiConfiguration);
		}

		GJaxbCreateSensorResponse response = epsClient.createSensor(req);
		response.setUserData("input_request", GenericModelHelper.getName(req.getNode()));

		// PATCH: fix serialization error
		response.getSensorModel().getNode().setBrokenEdges(null);

		return response;
	}

	@POST
	@Path("/startSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbStartSensorResponse startSensor(@Auth DWUser user, GJaxbStartSensor req) throws Exception {
		GJaxbStartSensorResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorManagerClient.startSensor(req);

			for (QName topicProvided : response.getSensorModel().getTopicsProvided()) {
				String subscriptionId = logConsumer.getImplementation(this.notifier.getClass())
						.subscribeOn(req.getUrl(), topicProvided);
				subscriptionIds.put(topicProvided, subscriptionId);
			}

			JSONObject method = new JSONObject();
			method.put("method_name", "startSensor");
			method.put("method_status", "SUCCESSFUL");
//			sendEventOnRioseAndOrRiosemit(user, response.getSensorModel(), method, req.getCollaborationName(),
//					req.getKnowledgeSpaceName());
		} catch (Exception e) {

			GJaxbFindSensorByInternalUrl find = new GJaxbFindSensorByInternalUrl();
			find.setUrl(req.getUrl());
			find.setCollaborationName(req.getCollaborationName());
			find.setKnowledgeSpaceName(req.getKnowledgeSpaceName());
			GJaxbFindSensorByInternalUrlResponse findResp = this.sensorManagerClient.findSensorByInternalUrl(find);

			JSONObject method = new JSONObject();
			method.put("method_name", "startSensor");
			method.put("method_status", "ERROR");
//			sendEventOnRioseAndOrRiosemit(user, findResp.getSensorModel(), method, req.getCollaborationName(),
//					req.getKnowledgeSpaceName());

			GindWebApplicationException.manageError(e, this);
		}
		JSONObject jsonMsg = new JSONObject(
				"{ \"user\" : \"" + user.getName() + "\", " + "\"req.getUrl\" : \"" + req.getUrl() + "\" }");
		LOG.debug(jsonMsg.toString());
		LOG.debug("Start event producer at url: " + req.getUrl());
		return response;
	}

//	private void sendEventOnRioseAndOrRiosemit(DWUser user, GJaxbSensorModel sensorModel, JSONObject additionalInfos,
//			String collaborationName, String knowlegeSpaceName) throws Exception, GetResultFault {
//		GJaxbGetResult request = new GJaxbGetResult();
//		JSONObject json = new JSONObject();
//		JSONObject userJson = new JSONObject();
//		userJson.put("id", user.getUserId());
//		userJson.put("name", user.getName());
//
//		JSONObject eventsFromRiose = new JSONObject();
//		if (sensorModel != null) {
//			eventsFromRiose.put("sensorModel",
//					new JSONObject(JSONJAXBContext.getInstance().marshallAnyElement(sensorModel))
//							.getJSONObject("sensorModel"));
//		}
//		eventsFromRiose.put("additionalInfos", additionalInfos);
//		eventsFromRiose.put("user", userJson);
//		json.put("eventsFromRiose", eventsFromRiose);
//		request.setJsonResult(json.toString());
//
//		request.setWebsocketId("riose" + "/" + RegExpHelper.toRegexFriendlyName(collaborationName).hashCode() + "/"
//				+ RegExpHelper.toRegexFriendlyName(knowlegeSpaceName).hashCode());
//		webSocketCommand.getResult(request);
//
//		request.setWebsocketId("riosemit" + "/" + RegExpHelper.toRegexFriendlyName(collaborationName).hashCode() + "/"
//				+ RegExpHelper.toRegexFriendlyName(knowlegeSpaceName).hashCode());
//		webSocketCommand.getResult(request);
//
//	}

	@POST
	@Path("/stopSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbStopSensorResponse stopSensor(@Auth DWUser user, GJaxbStopSensor req) throws Exception {

		GJaxbStopSensorResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			for (String subscriptionId : subscriptionIds.values()) {
				logConsumer.getImplementation(this.notifier.getClass()).unsubscribeOn(req.getUrl(), subscriptionId);
			}

			response = this.sensorManagerClient.stopSensor(req);
			JSONObject method = new JSONObject();
			method.put("method_name", "stopSensor");
			method.put("method_status", "SUCCESSFUL");
//			sendEventOnRioseAndOrRiosemit(user, response.getSensorModel(), method, req.getCollaborationName(),
//					req.getKnowledgeSpaceName());
		} catch (Exception e) {
			e.printStackTrace();

			GJaxbFindSensorByInternalUrl find = new GJaxbFindSensorByInternalUrl();
			find.setUrl(req.getUrl());
			find.setCollaborationName(req.getCollaborationName());
			find.setKnowledgeSpaceName(req.getKnowledgeSpaceName());
			GJaxbFindSensorByInternalUrlResponse findResp = this.sensorManagerClient.findSensorByInternalUrl(find);

			JSONObject method = new JSONObject();
			method.put("method_name", "stopSensor");
			method.put("method_status", "ERROR");
//			sendEventOnRioseAndOrRiosemit(user, findResp.getSensorModel(), method, req.getCollaborationName(),
//					req.getKnowledgeSpaceName());

			GindWebApplicationException.manageError(e, this);
		}
		// resp.put("status", "STOPPED");
		JSONObject jsonMsg = new JSONObject(
				"{ \"user\" : \"" + user.getName() + "\", " + "\"req.getUrl\" : \"" + req.getUrl() + "\" }");
		LOG.debug(jsonMsg.toString());
		return response;
	}

	@POST
	@Path("/deleteSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbDeleteSensorResponse deleteSensor(@Auth DWUser user, GJaxbDeleteSensor req) throws Exception {
		GJaxbDeleteSensorResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorManagerClient.deleteSensor(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		LOG.debug("Start event producer at url: " + req.getUrl());
		return response;
	}

	@GET
	@Path("/findSensorByProtocol/{type}/{nature}/{collaborationName}/{knowledgeSpaceName}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbFindSensorByProtocolResponse findSensorByProtocol(@Auth DWUser user, @PathParam("type") String type,
			@PathParam("nature") String nature, @PathParam("collaborationName") String collaborationName,
			@PathParam("knowledgeSpaceName") String knowledgeSpaceName) throws Exception {
		JSONObject jsonMsg = new JSONObject("{ \"user\" : " + user.getName() + ", " + "\"collaborationName\" :"
				+ collaborationName + ", " + "\"knowledgeSpaceName\" :" + knowledgeSpaceName + ", " + "\"type\" : "
				+ type + ", " + "\"nature\" : " + nature + " }");
		LOG.debug(jsonMsg.toString());
		GJaxbFindSensorByProtocolResponse response = null;
		try {
			GJaxbFindSensorByProtocol request = new GJaxbFindSensorByProtocol();
			request.setCollaborationName(collaborationName);
			request.setKnowledgeSpaceName(knowledgeSpaceName);
			if (type != null) {
				request.getType().add(type);
			}
			if (nature != null) {
				request.getNature().add(nature);
			}
			response = this.sensorManagerClient.findSensorByProtocol(request);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/getSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetSensorResponse getSensor(@Auth DWUser user, GJaxbGetSensor req) throws Exception {

		GJaxbGetSensorResponse response = null;
		try {
			GJaxbGetSensor request = req;
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			response = this.sensorManagerClient.getSensor(request);

			// PATCH: fix serialization error
			if (response.getSensorModel() != null && response.getSensorModel().getNode() != null) {
				response.getSensorModel().getNode().setBrokenEdges(null);
			}

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/getSensors")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetSensorsResponse getSensors(@Auth DWUser user) throws Exception {

		GJaxbGetSensorsResponse response = null;
		try {
			GJaxbGetSensors request = new GJaxbGetSensors();
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			response = this.sensorManagerClient.getSensors(request);

			// PATCH: fix serialization error
			response.getSensorModel().forEach(sm -> {
				sm.getNode().setBrokenEdges(null);
			});

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/setLocked")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbSetLockedResponse setLocked(@Auth DWUser user, GJaxbSetLocked request) {
		GJaxbSetLockedResponse response = null;
		try {
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			response = this.sensorManagerClient.setLocked(request);
		} catch (Exception e) {
			throw new RestWebApplicationException(e);
		}
		return response;
	}

	public static GJaxbSensorUIConfiguration getUIConfiguration(FileResource fileResource, DWUser user,
			String uiConfigurationURL) throws Exception {
		GJaxbSensorUIConfiguration uiConfiguration = null;
		if (uiConfigurationURL != null && !uiConfigurationURL.isBlank()) {
			String buffer = fileResource.download(user, uiConfigurationURL);
			if (buffer == null) {
				throw new Exception("Cannot Open URL PATH: " + uiConfigurationURL);
			}
			Document doc = DOMUtil.getInstance().parse(new ByteArrayInputStream(buffer.getBytes()));
			// System.out.println("doc: \n" + XMLPrettyPrinter.print(doc));
			uiConfiguration = XMLJAXBContext.getInstance().unmarshallDocument(doc, GJaxbSensorUIConfiguration.class);
		}
		return uiConfiguration;
	}

	public static GJaxbSensorUIConfiguration createUIConfiguration(FileResource fileResource, DWUser user,
			GJaxbNode sensor) throws Exception {
		String type = GenericModelHelper.findProperty("type", sensor.getProperty()).getValue().toLowerCase();
		String nature = GenericModelHelper.findProperty("nature", sensor.getProperty()).getValue().toLowerCase();

		GJaxbSensorUIConfiguration uiConfiguration = SensorUIConfigurationHelper.createUIConfiguration(type, nature,
				sensor);
		String dataXML = XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(uiConfiguration));

		String fileName = type + "_" + nature + "UIConfig.xml";
		JSONObject data = new JSONObject();
		data.put("nodeId", sensor.getId());
		data.put("category", "node");
		data.put("collaborationName", user.getCurrentCollaborationName());
		data.put("knowledgeSpaceName", user.getCurrentKnowledgeSpaceName());
		data.put("fileName", fileName);

		String url = fileResource.upload(user, new ByteArrayInputStream(dataXML.getBytes()),
				FormDataContentDisposition.name(fileName).build(), data.toString());

		GenericModelHelper.findProperty("ui configuration", sensor.getProperty(), true).setValue(url);

		return uiConfiguration;
	}

	@POST
	@Path("/findSensorByTopic")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbFindSensorByTopicResponse findSensorByTopic(@Auth DWUser user, GJaxbFindSensorByTopic req)
			throws Exception {
		GJaxbFindSensorByTopicResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorManagerClient.findSensorByTopic(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/startContinuousSignalOnJSONConnector")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbStartContinuousSignalOnJSONConnectorResponse startContinuousSignalOnJSONConnector(@Auth DWUser user,
			GJaxbStartContinuousSignalOnJSONConnector req) throws Exception {
		GJaxbStartContinuousSignalOnJSONConnectorResponse response = null;
		try {
			for (GJaxbContinuousSignalInput r : req.getContinuousSignalInput()) {
				if (r.getCollaborationName() == null) {
					r.setCollaborationName(user.getCurrentCollaborationName());
				}
				if (r.getKnowledgeSpaceName() == null) {
					r.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				}
			}
			response = sensorManagerClient.startContinuousSignalOnJSONConnector(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/pauseContinuousSignalOnJSONConnector")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbPauseContinuousSignalOnJSONConnectorResponse pauseContinuousSignalOnJSONConnector(@Auth DWUser user,
			GJaxbPauseContinuousSignalOnJSONConnector req) throws Exception {
		GJaxbPauseContinuousSignalOnJSONConnectorResponse response = null;
		try {
			for (GJaxbContinuousSignalOutput r : req.getContinuousSignalOutput()) {
				if (r.getCollaborationName() == null) {
					r.setCollaborationName(user.getCurrentCollaborationName());
				}
				if (r.getKnowledgeSpaceName() == null) {
					r.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				}
			}

			response = sensorManagerClient.pauseContinuousSignalOnJSONConnector(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/updateContinuousSignalOnJSONConnector")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbUpdateContinuousSignalOnJSONConnectorResponse updateContinuousSignalOnJSONConnector(@Auth DWUser user,
			GJaxbUpdateContinuousSignalOnJSONConnector req) throws Exception {
		GJaxbUpdateContinuousSignalOnJSONConnectorResponse response = null;
		try {
			for (GJaxbContinuousSignalInput r : req.getContinuousSignalInput()) {
				if (r.getCollaborationName() == null) {
					r.setCollaborationName(user.getCurrentCollaborationName());
				}
				if (r.getKnowledgeSpaceName() == null) {
					r.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				}
			}

			response = sensorManagerClient.updateContinuousSignalOnJSONConnector(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/continuousSignalIsStartedOnJSONConnector")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbContinuousSignalIsStartedOnJSONConnectorResponse continuousSignalIsStartedOnJSONConnector(
			@Auth DWUser user, GJaxbContinuousSignalIsStartedOnJSONConnector req) throws Exception {
		GJaxbContinuousSignalIsStartedOnJSONConnectorResponse response = null;
		try {

			for (GJaxbContinuousSignalOutput r : req.getContinuousSignalOutput()) {
				if (r.getCollaborationName() == null) {
					r.setCollaborationName(user.getCurrentCollaborationName());
				}
				if (r.getKnowledgeSpaceName() == null) {
					r.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
				}
			}
			response = sensorManagerClient.continuousSignalIsStartedOnJSONConnector(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/updateConfigOfSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbUpdateConfigOfSensorResponse updateConfigOfSensor(@Auth DWUser user, GJaxbUpdateConfigOfSensor req)
			throws Exception {
		GJaxbUpdateConfigOfSensorResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}
			response = sensorManagerClient.updateConfigOfSensor(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/sendTweetToAllSensors")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbListenResponse sendTweetToAllSensors(@Auth DWUser user, FormDataMultiPart multiPart) {
		GJaxbListenResponse res = null;
		try {
			FormDataBodyPart jsonTweet = multiPart.getField("tweet");
			String collaborationName = multiPart.getField("collaborationName").getValue();
			String knowledgeSpaceName = multiPart.getField("knowledgeSpaceName").getValue();
			GJaxbTweetObject tweet = JSONJAXBContext.getInstance()
					.unmarshall("{ \"tweet\" : " + jsonTweet.getValue() + " }", GJaxbTweetObject.class);

			List<FormDataBodyPart> fileParts = multiPart.getFields("file");
			if (fileParts != null) {
				for (FormDataBodyPart part : fileParts) {
					InputStream is = part.getValueAs(InputStream.class);
					FormDataContentDisposition fileDetail = part.getFormDataContentDisposition();

					GJaxbMediaObject media = tweet.getExtendedEntities().getMedia().stream()
							.filter(f -> f.getMediaUrl().endsWith(fileDetail.getFileName())).findFirst().get();

					JSONObject data = new JSONObject();
					data.put("category", "person");
					data.put("collaborationName", collaborationName);
					data.put("knowledgeSpaceName", knowledgeSpaceName);
					if (tweet.getUser() != null && tweet.getUser().getIdStr() != null
							&& !tweet.getUser().getIdStr().isBlank()) {
						data.put("nodeId", tweet.getUser().getIdStr());
					} else {
						data.put("nodeId", "shareNode");
					}
					String newUrl = this.fileResource.upload(user, is, fileDetail, data.toString());
					newUrl = newUrl.replace(RegExpHelper.toRegexFriendlyName(knowledgeSpaceName) + "/",
							RegExpHelper.toRegexFriendlyName(collaborationName) + "/"
									+ RegExpHelper.toRegexFriendlyName(knowledgeSpaceName) + "/");

					String pretty_address = IPUtil.createPrettyAddressFomContext(this.conf.getMapObjects());
					newUrl = pretty_address + newUrl;
					media.setMediaUrl(newUrl);

				}
			}

			GJaxbFindSensorByProtocolResponse resp = this.findSensorByProtocol(user, "Social", "Twitter",
					collaborationName, knowledgeSpaceName);
			for (GJaxbSensorModel sensorModel : resp.getSensorModel()) {

				GJaxbSensorEvent sensorEvent = new GJaxbSensorEvent();
				sensorEvent.setEventId("event_" + UUID.randomUUID());
				sensorEvent.setSensorId(sensorModel.getNode().getId());
				sensorEvent.setSensorType(GJaxbSensorType.SOCIAL);
				sensorEvent.setSensorNature(new SensorNature());
				sensorEvent.getSensorNature().setSocialNature(GJaxbSensorNatureSocialType.TWITTER);
				sensorEvent.setDate(XMLGregorianCalendarHelper.getInstance().getNewCalendar());
				tweet.setCreatedAt(sensorEvent.getDate().toString());
				tweet.setIdStr("tweet_" + sensorEvent.getEventId());
				sensorEvent.setSensorEventIndex(-1);
				sensorEvent.setSensorEventStackIndex(0);
				sensorEvent.setSensorColor(ColorHelper.formatColorToHexaString(ColorHelper.randomBrighterColor()));

				sensorEvent.setData(new GJaxbData());
				sensorEvent.getData().setSocial(new Social());
				sensorEvent.getData().getSocial().setValue(new Value());
				sensorEvent.getData().getSocial().getValue().setTweet(tweet);

				sensorEvent.setPosition(new GJaxbPosition());
				sensorEvent.getPosition().setPoint(new GJaxbPoint());
				sensorEvent.getPosition().getPoint()
						.setLatitude(Float.parseFloat(tweet.getCoordinates().getCoordinates().get(0)));
				sensorEvent.getPosition().getPoint()
						.setLongitude(Float.parseFloat(tweet.getCoordinates().getCoordinates().get(1)));

				Document payload = XMLJAXBContext.getInstance().marshallAnyElement(sensorEvent);

				GJaxbListen request = new GJaxbListen();
				request.setPayload(new Payload());
				request.getPayload().setAny(payload.getDocumentElement());

				request.setCollaborationName(collaborationName);
				request.setKnowledgeSpaceName(knowledgeSpaceName);
				request.setDataSourceId(sensorModel.getNode().getId());

				res = sensorManagerClient.listen(request);
			}

		} catch (Exception e) {
			throw new RestWebApplicationException(e);
		}
		return res;
	}


}