/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources.gov;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.xml.namespace.QName;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import fr.emac.gind.commons.utils.RIOConstant;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.security.GindCrypto;
import fr.emac.gind.commons.utils.xml.XMLPrettyPrinter;
import fr.emac.gind.generic.application.ApplicationContext;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.generic.application.utils.LoadedUsecase;
import fr.emac.gind.gov.ai.chatbot.client.AIChatbotClient;
import fr.emac.gind.gov.ai_chatbot.AiChatbot;
import fr.emac.gind.gov.ai_chatbot.GJaxbAiChatBotType;
import fr.emac.gind.gov.ai_chatbot.GJaxbContext;
import fr.emac.gind.gov.ai_chatbot.GJaxbContext.ChatGPTContext;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPersons;
import fr.emac.gind.gov.ai_chatbot.GJaxbInventPersonsResponse;
import fr.emac.gind.gov.ai_chatbot.GJaxbPersonInput;
import fr.emac.gind.gov.core.client.CoreGovClient;
import fr.emac.gind.gov.core.client.SystemGovClient;
import fr.emac.gind.gov.core.client.util.Neo4JId;
import fr.emac.gind.gov.core.client.util.Neo4JReqConstant;
import fr.emac.gind.gov.core_gov.CoreGov;
import fr.emac.gind.gov.core_gov.FaultMessage;
import fr.emac.gind.gov.core_gov.GJaxbAddNodeResponse;
import fr.emac.gind.gov.core_gov.GJaxbQuery;
import fr.emac.gind.gov.core_gov.GJaxbQueryResponse;
import fr.emac.gind.gov.core_gov.GJaxbUpdateNode;
import fr.emac.gind.gov.core_gov.GJaxbUpdateNodeResponse;
import fr.emac.gind.gov.models.client.ModelsGovClient;
import fr.emac.gind.gov.models_gov.GJaxbPublishSyncModel;
import fr.emac.gind.gov.models_gov.ModelsGov;
import fr.emac.gind.gov.system_gov.GJaxbAddCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbAddCollaborationResponse;
import fr.emac.gind.gov.system_gov.GJaxbAddUser;
import fr.emac.gind.gov.system_gov.GJaxbAttachKnowledgeSpaceToCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbConnectUserToCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbConnectUserToCollaborationResponse;
import fr.emac.gind.gov.system_gov.GJaxbExportCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbFindCollaborationsByName;
import fr.emac.gind.gov.system_gov.GJaxbFindCollaborationsByNameResponse;
import fr.emac.gind.gov.system_gov.GJaxbFindCollaborationsOfUser;
import fr.emac.gind.gov.system_gov.GJaxbFindUserByEmail;
import fr.emac.gind.gov.system_gov.GJaxbFindUserByLoginAndPassword;
import fr.emac.gind.gov.system_gov.GJaxbGetAdministrators;
import fr.emac.gind.gov.system_gov.GJaxbGetAllCollaborations;
import fr.emac.gind.gov.system_gov.GJaxbGetAllUsers;
import fr.emac.gind.gov.system_gov.GJaxbGetCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbGetKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbGetKnowledgeSpaceResponse;
import fr.emac.gind.gov.system_gov.GJaxbGetKnowledgeSpacesInCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbGetUser;
import fr.emac.gind.gov.system_gov.GJaxbGetUserResponse;
import fr.emac.gind.gov.system_gov.GJaxbGetUsersInCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbRemoveCollaboration;
import fr.emac.gind.gov.system_gov.GJaxbRemoveCollaborationResponse;
import fr.emac.gind.gov.system_gov.GJaxbRemoveKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbRemoveUser;
import fr.emac.gind.gov.system_gov.GJaxbRemoveUserResponse;
import fr.emac.gind.gov.system_gov.GJaxbSelectedKnowledgeSpace;
import fr.emac.gind.gov.system_gov.GJaxbUpdateUser;
import fr.emac.gind.gov.system_gov.GJaxbUpdateUserResponse;
import fr.emac.gind.gov.system_gov.SystemGov;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.modeler.genericmodel.GJaxbEdge;
import fr.emac.gind.modeler.genericmodel.GJaxbGenericModel;
import fr.emac.gind.modeler.genericmodel.GJaxbNode;
import fr.emac.gind.modeler.genericmodel.GJaxbNode.BrokenEdges;
import fr.emac.gind.modeler.genericmodel.GJaxbProperty;
import fr.emac.gind.modeler.genericmodel.GJaxbRelationModeType;
import fr.emac.gind.modeler.metamodel.GJaxbEffectiveConceptType;
import fr.emac.gind.modeler.metamodel.GJaxbRelation;
import fr.emac.gind.models.generic.modeler.effective_meta_model.EffectiveMetaModelManager;
import fr.emac.gind.models.generic.modeler.generic_model.GenericModelHelper;
import fr.emac.gind.rio.PluginCollaborativeModel;
import io.dropwizard.auth.Auth;
import io.dropwizard.auth.AuthenticationException;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.StreamingOutput;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/generic-application/system")
//@Api("/users")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class SystemResource {

	private static Logger LOG = LoggerFactory.getLogger(SystemResource.class);

	private SystemGov systemClient = null;
	private ModelsGov modelsClient = null;
	private CoreGov coreClient = null;
	private AiChatbot aiChatBotClient = null;
	private Configuration conf = null;
	private ApplicationContext applicationContext = null;
	private GJaxbEffectiveConceptType collaborationMetaModel = null;
	private GJaxbEffectiveConceptType knowledgeSpaceMetaModel = null;
	private GJaxbRelation connectedToMetaModel = null;

	private EffectiveMetaModelManager manager = null;
	private static GJaxbNode DEFAULT_ADMIN = null;
	private static GJaxbNode CURRENT_USER = null;

	public SystemResource(ApplicationContext applicationContext, Configuration conf) throws Exception {
		this.conf = conf;
		this.applicationContext = applicationContext;
		this.systemClient = SystemGovClient
				.createClient(conf.getProperties().get("governance").replace("/gov", "/GovSystem"));
		this.modelsClient = ModelsGovClient
				.createClient(conf.getProperties().get("governance").replace("/gov", "/GovModels"));
		this.coreClient = CoreGovClient
				.createClient(conf.getProperties().get("governance").replace("/gov", "/GovCore"));
		this.aiChatBotClient = AIChatbotClient
				.createClient(conf.getProperties().get("governance").replace("/gov", "/GovAIChatbot"));

		manager = new EffectiveMetaModelManager(this.applicationContext.getSystem());
		collaborationMetaModel = manager.getConceptByType(new QName("http://fr.emac.gind/system", "Collaboration"));
		knowledgeSpaceMetaModel = manager.getConceptByType(new QName("http://fr.emac.gind/system", "Knowledge_Space"));
		connectedToMetaModel = manager.getRelationByType(new QName("http://fr.emac.gind/system", "Connected_To"));
		List<GJaxbNode> administrators = this.systemClient.getAdministrators(new GJaxbGetAdministrators()).getNode();
		if (administrators != null && administrators.size() > 0) {
			LOG.debug("Default admin already exist !!!");
			DEFAULT_ADMIN = administrators.get(0);
		}

		if (DEFAULT_ADMIN == null && Boolean.valueOf(conf.getProperties().get("defaultAdmin")) == true) {
			DEFAULT_ADMIN = createFirstAdmin(manager);
		}
		GJaxbFindUserByEmail emailReq = new GJaxbFindUserByEmail();
		emailReq.setEmail("${current_user_email}");
		GJaxbNode currentUser = this.systemClient.findUserByEmail(emailReq).getNode();
		CURRENT_USER = currentUser;

		if (CURRENT_USER == null) {
			CURRENT_USER = createCurrentUser(manager);
		}

	}

	public SystemGov getSystemClient() {
		return systemClient;
	}

	public GJaxbEffectiveConceptType getCollaborationMetaModel() {
		return collaborationMetaModel;
	}

	public static GJaxbNode getDEFAULT_ADMIN() {
		return DEFAULT_ADMIN;
	}

	public static void setDEFAULT_ADMIN(GJaxbNode dEFAULT_ADMIN) {
		DEFAULT_ADMIN = dEFAULT_ADMIN;
	}

	public static GJaxbNode getCURRENT_USER() {
		return CURRENT_USER;
	}

	public static void setCURRENT_USER(GJaxbNode cURRENT_USER) {
		CURRENT_USER = cURRENT_USER;
	}

	@POST
	@Path("/user/login")
	@Consumes(MediaType.TEXT_PLAIN)
	public GJaxbNode login(@Auth Optional<DWUser> userOpt, String token) throws Exception {
		LOG.debug("try to log with token: " + token);
		try {
			LOG.debug("token : " + token);

			final String decoded;
			try {
				decoded = new String(Base64.getDecoder().decode(token), StandardCharsets.UTF_8);
			} catch (IllegalArgumentException e) {
				LOG.warn("Error decoding credentials", e);
				return null;
			}

			// Decoded credentials is 'username:password'
			final int i = decoded.indexOf(':');
			if (i <= 0) {
				return null;
			}

			final String login = decoded.substring(0, i);
			final String password = decoded.substring(i + 1);

			GJaxbFindUserByLoginAndPassword req = new GJaxbFindUserByLoginAndPassword();
			req.setLogin(login);
			req.setPassword(password);
			GJaxbNode user = systemClient.findUserByLoginAndPassword(req).getNode();
			if (user == null) {
				throw new AuthenticationException("Login and/or password are invalid !!!");
			}

			/**
			 * GJaxbNode alreadyLoggedUser = DWUser.getAlreadyLoggedUser(user.getId()); if
			 * (alreadyLoggedUser != null) { user = alreadyLoggedUser; }
			 */

			DWUser dwUser = new DWUser(user, systemClient);
			JSONObject jsonMsg = new JSONObject("{ \"user\" : " + dwUser.getName() + "}");
			LOG.debug(jsonMsg.toString());

			if (ProjectResource.REALWORLD_UC_NODE != null && ProjectResource.WORLD_COLLABORATION_NODE != null) {
				GenericModelHelper.findProperty("Presence", dwUser.getUser().getProperty(), true).setValue("Available");

				if (dwUser.getCurrentKnowledgeSpaceName() == null) {
					dwUser.setCurrentCollaborationName(
							GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE));
					dwUser.setCurrentKnowledgeSpaceName(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE));
				}
				GJaxbUpdateUserResponse upResp = this.updateUser(null, dwUser.getUser());

				// create associate person/cell phone/watch in real world
				if (findAssociatedPerson(user.getId(),
						GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE),
						GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE)) == null) {
					publishPersonCellPhoneWatchToAssociateUserInRealWorld(dwUser.getUser(), this.coreClient,
							this.modelsClient, this.systemClient, this.aiChatBotClient);

					// set available presence on person in real world
					setStatusOnRealWorldPerson(dwUser.getUser(), "Available");
				}
			}

			if (dwUser.getCurrentKnowledgeSpaceName() == null || dwUser.getCurrentKnowledgeSpaceName().isBlank()) {
				if (!ProjectResource.USECASES_ALREADY_LOADED.isEmpty()) {
					LoadedUsecase uc = ProjectResource.USECASES_ALREADY_LOADED.values().iterator().next();
					GJaxbNode colladNode = uc.getCollaboration();
					GJaxbNode kownledgeNode = uc.getKnowledgeSpace();

					GenericModelHelper.findProperty("Presence", dwUser.getUser().getProperty(), true)
							.setValue("Available");

					dwUser.setCurrentCollaborationName(GenericModelHelper.getName(colladNode));
					dwUser.setCurrentKnowledgeSpaceName(GenericModelHelper.getName(kownledgeNode));

					GJaxbUpdateUserResponse upResp = this.updateUser(null, dwUser.getUser());

					GJaxbConnectUserToCollaboration reqConnColl = new GJaxbConnectUserToCollaboration();
					reqConnColl.setCollaborationId(colladNode.getId());
					reqConnColl.setUserId(dwUser.getUserId());
					this.connectUserToCollaboration(dwUser, reqConnColl);
				}
			}

			if (dwUser.getCurrentCollaborationName() == null) {

				setDefaultUserWorkSpace(user, this.systemClient, this.collaborationMetaModel,
						this.connectedToMetaModel);

			}
			return user;
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return null;
	}

	public GJaxbNode findAssociatedPerson(String userId, String collaboratinName, String knowledgeSpaceName)
			throws Exception {
		String cQuery = "match (p:`" + RegExpHelper.toRegexFriendlyName(collaboratinName) + "`:`"
				+ RegExpHelper.toRegexFriendlyName(knowledgeSpaceName) + "`:`"
				+ GenericModelHelper.collab("Person").toString() + "`)" + "-[:`"
				+ GenericModelHelper.collab("Assigned_To").toString() + "`]->" + "(u:`"
				+ GenericModelHelper.system("User").toString() + "` { modelNodeId : \"" + userId + "\"}) " + "return p";
		GJaxbQuery q = new GJaxbQuery();
		q.setQuery(cQuery);
		q.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
		q.getSelectedKnowledgeSpace().setCollaborationName(collaboratinName);
		q.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);
		GJaxbQueryResponse qResp = this.coreClient.query(q);
		if (qResp.getSingle() != null) {
			GJaxbGenericModel gmodel = qResp.getSingle().getGenericModel();
			if (gmodel != null && !gmodel.getNode().isEmpty()) {
				return gmodel.getNode().get(0);
			}
		}
		return null;
	}

	public static void publishPersonCellPhoneWatchToAssociateUserInRealWorld(GJaxbNode user, CoreGov coreClient,
			ModelsGov modelsClient, SystemGov systemClient, AiChatbot chatBot) throws Exception {

		// connect user to real world
		GJaxbConnectUserToCollaboration conReq = new GJaxbConnectUserToCollaboration();
		conReq.setUserId(user.getId());
		conReq.setCollaborationId(ProjectResource.WORLD_COLLABORATION_NODE.getId());
		systemClient.connectUserToCollaboration(conReq);

		GJaxbQuery query = new GJaxbQuery();
		query.setQuery("match (p:" + Neo4JReqConstant.collab("Person") + ")-[:" + Neo4JReqConstant.collab("Assigned_To")
				+ "]->(u:" + Neo4JReqConstant.system("User") + " { modelNodeId : \"" + user.getId() + "\"}) return p");
		query.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
		query.getSelectedKnowledgeSpace()
				.setCollaborationName(GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE));
		query.getSelectedKnowledgeSpace()
				.setKnowledgeName(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE));
		GJaxbQueryResponse queryResp = coreClient.query(query);
		if (queryResp.getSingle() == null || queryResp.getSingle().getGenericModel() == null
				|| queryResp.getSingle().getGenericModel().getNode().size() == 0) {
			GJaxbGenericModel worldPersonCellPhoneWatch = createPersonCellPhoneWatchFromUser(user, chatBot);
			GJaxbPublishSyncModel reqPub = new GJaxbPublishSyncModel();
			reqPub.setGenericModel(worldPersonCellPhoneWatch);
			reqPub.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
			reqPub.getSelectedKnowledgeSpace()
					.setCollaborationName(GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE));
			reqPub.getSelectedKnowledgeSpace()
					.setKnowledgeName(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE));
			modelsClient.publishSyncModel(reqPub);
		}
	}

	private static GJaxbGenericModel createPersonCellPhoneWatchFromUser(GJaxbNode user, AiChatbot chatBot)
			throws Exception {

		GJaxbInventPersons inventPersons = new GJaxbInventPersons();
		inventPersons.setContext(new GJaxbContext());
		inventPersons.getContext().setSelectedAiChatBot(GJaxbAiChatBotType.CHAT_GPT);
		inventPersons.getContext().setChatGPTContext(new ChatGPTContext());
		GJaxbPersonInput pi = new GJaxbPersonInput();
		pi.setGroupId("person_1");
		pi.setNumber(1);
		pi.setAddPhone(true);
		pi.setAddWatch(true);
		inventPersons.getPersonInput().add(pi);
		GJaxbInventPersonsResponse inventPersonsResp = chatBot.inventPersons(inventPersons);
		GJaxbGenericModel model = inventPersonsResp.getGenericModel();

		GJaxbNode person = GenericModelHelper.findNodesByType(GenericModelHelper.collab("Person"), model.getNode())
				.get(0);
		String lastName = GenericModelHelper.getName(person);
		String newName = GenericModelHelper.getName(user);
		for (GJaxbProperty prop : person.getProperty()) {
			GJaxbProperty userProp = GenericModelHelper.findProperty(prop.getName(), user.getProperty());
			if (userProp != null) {
				prop.setValue(userProp.getValue());
			}
		}
		person.setBrokenEdges(new BrokenEdges());

		for (GJaxbNode node : model.getNode()) {
			String nodeName = GenericModelHelper.getName(node);
			if (nodeName.contains(lastName)) {
				GenericModelHelper.findProperty("name", node.getProperty()).setValue(nodeName.replace(lastName, newName));
			}
		}

		GJaxbEdge person_user = GenericModelHelper.createEdgeBetweenNodes(person, user,
				new QName(PluginCollaborativeModel.COLLABORATIVE_NAMESPACE, "Assigned_To"),
				GJaxbRelationModeType.ASSOCIATION);

		model.getEdge().add(person_user);

		return model;
	}

	@GET
	@Path("/user/logout")
	@Consumes(MediaType.TEXT_PLAIN)
	public GJaxbNode logout(@Auth DWUser userOpt) throws Exception {
		LOG.debug("try to loggout ");
		JSONObject jsonMsg = new JSONObject("{ \"user\" : " + userOpt.getName() + "}");
		LOG.debug(jsonMsg.toString());
		try {

			GJaxbNode user = userOpt.getUser();
			this.updateUser(userOpt, user);
			if (user == null) {
				throw new AuthenticationException("Login and/or password are invalid !!!");
			}

			GenericModelHelper.findProperty("Presence", user.getProperty(), true).setValue("Offline");
			this.updateUser(userOpt, user);

			if (ProjectResource.REALWORLD_UC_NODE != null && ProjectResource.WORLD_COLLABORATION_NODE != null) {
				if (userOpt.getCurrentCollaborationName()
						.equals(GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE))
						&& userOpt.getCurrentKnowledgeSpaceName()
								.equals(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE))) {
					// set offline presence on person in real world
					setStatusOnRealWorldPerson(user, "Offline");
				}
			}

			return user;
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return null;
	}

	private void setStatusOnRealWorldPerson(GJaxbNode user, String status) throws FaultMessage {
		GJaxbQuery query = new GJaxbQuery();
		query.setQuery("match (p:`"
				+ RegExpHelper.toRegexFriendlyName(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE)) + "`:"
				+ Neo4JReqConstant.collab("Person") + ")-[:" + Neo4JReqConstant.collab("Assigned_To") + "]->(u:"
				+ Neo4JReqConstant.system("User") + " { modelNodeId : \"" + user.getId() + "\"}) return p");
		query.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
		query.getSelectedKnowledgeSpace()
				.setCollaborationName(GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE));
		query.getSelectedKnowledgeSpace()
				.setKnowledgeName(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE));
		GJaxbQueryResponse queryResp = coreClient.query(query);
		if (queryResp.getSingle() != null && queryResp.getSingle().getGenericModel() != null
				&& queryResp.getSingle().getGenericModel().getNode() != null
				&& !queryResp.getSingle().getGenericModel().getNode().isEmpty()) {
			GJaxbNode associatePerson = queryResp.getSingle().getGenericModel().getNode().get(0);
			GenericModelHelper.findProperty("Presence", associatePerson.getProperty(), true).setValue(status);
			GJaxbUpdateNode upNode = new GJaxbUpdateNode();
			upNode.setNode(associatePerson);
			upNode.setSelectedKnowledgeSpace(new fr.emac.gind.sharedOptions.GJaxbSelectedKnowledgeSpace());
			upNode.getSelectedKnowledgeSpace()
					.setCollaborationName(GenericModelHelper.getName(ProjectResource.WORLD_COLLABORATION_NODE));
			upNode.getSelectedKnowledgeSpace()
					.setKnowledgeName(GenericModelHelper.getName(ProjectResource.REALWORLD_UC_NODE));
			coreClient.updateNode(upNode);
		}
	}

	@POST
	@Path("/user/register")
	public synchronized GJaxbNode register(@Auth Optional<DWUser> userOpt, GJaxbNode user) throws Exception {
		GJaxbAddNodeResponse response = null;
		try {

			List<GJaxbNode> administrators = this.systemClient.getAdministrators(new GJaxbGetAdministrators())
					.getNode();
			JSONObject roles = new JSONObject(GenericModelHelper.findProperty("roles", user.getProperty()).getValue());
			if (this.getAdmins(administrators).size() > 0 && roles.has("admin")
					&& roles.getJSONObject("admin").getBoolean("checked") == true) {
				throw new Exception("Only the first admin can register himself with admin role");
			}

			String email = GenericModelHelper.findProperty("email", user.getProperty()).getValue();
			GJaxbFindUserByEmail emailReq = new GJaxbFindUserByEmail();
			emailReq.setEmail(email);
			GJaxbNode userWithSameEmail = systemClient.findUserByEmail(emailReq).getNode();

			if (userWithSameEmail != null) {
				throw new Exception("User already exist with this email '" + email
						+ "' . Please contact administrator if you have forgotten your password!!!");
			}

			GJaxbAddUser req = new GJaxbAddUser();
			req.setNode(user);
			this.systemClient.addUser(req);
			createUserWorkSpace(user, this.systemClient, this.collaborationMetaModel, this.connectedToMetaModel);

			if (ProjectResource.REALWORLD_UC_NODE != null && ProjectResource.WORLD_COLLABORATION_NODE != null) {
				GJaxbNode colladNode = ProjectResource.WORLD_COLLABORATION_NODE;
				GJaxbNode kownledgeNode = ProjectResource.REALWORLD_UC_NODE;

				DWUser dwUser = new DWUser(user, this.getSystemClient());
				GenericModelHelper.findProperty("Presence", user.getProperty(), true).setValue("Available");

				dwUser.setCurrentCollaborationName(GenericModelHelper.getName(colladNode));
				dwUser.setCurrentKnowledgeSpaceName(GenericModelHelper.getName(kownledgeNode));

				GJaxbUpdateUserResponse upResp = this.updateUser(null, dwUser.getUser());

				GJaxbConnectUserToCollaboration reqConnColl = new GJaxbConnectUserToCollaboration();
				reqConnColl.setCollaborationId(colladNode.getId());
				reqConnColl.setUserId(user.getId());

				this.connectUserToCollaboration(dwUser, reqConnColl);
			}

			if (userOpt != null && userOpt.isPresent()) {
				String currentCollaborationName = userOpt.get().getCurrentCollaborationName();
				String currentKnowledgeSpaceName = userOpt.get().getCurrentKnowledgeSpaceName();

				DWUser dwUser = new DWUser(user, this.getSystemClient());
				GenericModelHelper.findProperty("Presence", user.getProperty(), true).setValue("Available");

				dwUser.setCurrentCollaborationName(currentCollaborationName);
				dwUser.setCurrentKnowledgeSpaceName(currentKnowledgeSpaceName);

				GJaxbUpdateUserResponse upResp = this.updateUser(null, dwUser.getUser());

				GJaxbFindCollaborationsByName findReq = new GJaxbFindCollaborationsByName();
				findReq.setName(currentCollaborationName);
				GJaxbFindCollaborationsByNameResponse findResp = this.systemClient.findCollaborationsByName(findReq);
				GJaxbNode currentCollaboration = findResp.getNode().get(0);

				GJaxbConnectUserToCollaboration reqConnColl = new GJaxbConnectUserToCollaboration();
				reqConnColl.setCollaborationId(currentCollaboration.getId());
				reqConnColl.setUserId(user.getId());

				this.connectUserToCollaboration(dwUser, reqConnColl);
			}

			LOG.info("New Register Name: " + GenericModelHelper.findProperty("name", user.getProperty()).getValue()
					+ " Email: " + GenericModelHelper.findProperty("email", user.getProperty()).getValue());
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return user;
	}

	@GET
	@Path("/user/users")
	// @RolesAllowed({"admin"})
	public List<GJaxbNode> getAllUsers(@Auth DWUser user) throws Exception {
		List<GJaxbNode> users = null;
		try {
			users = this.systemClient.getAllUsers(new GJaxbGetAllUsers()).getNode();
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return users;
	}

	@GET
	@Path("/user/users/{id}")
	public GJaxbNode getUserById(@Auth DWUser user, @PathParam("id") String id, @QueryParam("userqId") String userqId)
			throws Exception {
		GJaxbNode userFound = null;

		try {
			LOG.debug("id = " + id);
			LOG.debug("qId = " + userqId);
			GJaxbGetUser request = new GJaxbGetUser();
			request.setId(userqId);
			GJaxbGetUserResponse response = this.systemClient.getUser(request);

			userFound = response.getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return userFound;
	}

	@GET
	@Path("/user/hasAdmins")
	public boolean hasAdministrators() throws Exception {
		boolean hasAdmin = false;
		try {
			List<GJaxbNode> admins = this.systemClient.getAdministrators(new GJaxbGetAdministrators()).getNode();
			if (admins != null && !admins.isEmpty()) {
				hasAdmin = true;
			}
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return hasAdmin;
	}

	@POST
	@Path("/user/users")
	public GJaxbUpdateUserResponse updateUser(@Auth DWUser user, GJaxbNode userNode) throws Exception {
		GJaxbUpdateUserResponse response = null;

		try {

			GJaxbUpdateUser updateReq = new GJaxbUpdateUser();
			updateReq.setNode(userNode);
			updateReq.setCreateIfNotExist(true);
			response = this.systemClient.updateUser(updateReq);

			if (user != null && user.getUser().getId().equals(userNode.getId())) {
				user.setUser(userNode);
			}

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	private List<GJaxbNode> getAdmins(List<GJaxbNode> users) {
		List<GJaxbNode> admins = new ArrayList<GJaxbNode>();
		for (GJaxbNode user : users) {
			JSONObject roles = new JSONObject(GenericModelHelper.findProperty("roles", user.getProperty()).getValue());
			if (roles.getJSONObject("admin").getBoolean("checked") == true) {
				admins.add(user);
			} else {
				LOG.debug("user " + GenericModelHelper.getName(user) + " is not admin");
			}
		}
		return admins;
	}

	private void setDefaultUserWorkSpace(GJaxbNode user, SystemGov systemClient2,
			GJaxbEffectiveConceptType collaborationMetaModel, GJaxbRelation connectedToMetaModel) throws Exception {

		String defaultWorkSpaceUser = GenericModelHelper.getName(user) + " WorkSpace";

		// Find collaboration
		GJaxbFindCollaborationsByName reqName = new GJaxbFindCollaborationsByName();
		reqName.setName(defaultWorkSpaceUser);
		GJaxbFindCollaborationsByNameResponse respName = systemClient.findCollaborationsByName(reqName);

		if (respName.getNode() != null && !respName.getNode().isEmpty()) {
			GJaxbNode userCollaboration = respName.getNode().get(0);

			// update user
			GenericModelHelper.findProperty("currentCollaborationName", user.getProperty(), true)
					.setValue(defaultWorkSpaceUser);
			GJaxbUpdateUser upUser = new GJaxbUpdateUser();
			upUser.setNode(user);
			systemClient.updateUser(upUser);

			// connect user to collaboration
			GJaxbConnectUserToCollaboration connect = new GJaxbConnectUserToCollaboration();
			connect.setCollaborationId(userCollaboration.getId());
			connect.setUserId(user.getId());
			systemClient.connectUserToCollaboration(connect);
		} else {
			LOG.warn("Default Workspace ('collaboration') not exist for user: " + GenericModelHelper.getName(user));
		}

	}

	public static void createUserWorkSpace(GJaxbNode user, SystemGov systemClient,
			GJaxbEffectiveConceptType collaborationMetaModel, GJaxbRelation connectedToMetaModel) throws Exception {
		GJaxbUpdateNodeResponse response;

		// create collaboration
		GJaxbNode userCollaboration = new GJaxbNode();
		if (userCollaboration.getId() == null || userCollaboration.getId().isBlank()) {
			userCollaboration.setId("collaboration_" + UUID.randomUUID().toString());
		}
		userCollaboration.setType(collaborationMetaModel.getType());

		GenericModelHelper.findProperty("name", userCollaboration.getProperty(), true)
				.setValue(GenericModelHelper.getName(user) + " WorkSpace");

		GJaxbAddCollaboration requestUserCollaboration = new GJaxbAddCollaboration();
		requestUserCollaboration.setNode(userCollaboration);
		systemClient.addCollaboration(requestUserCollaboration);

		// update user
		GenericModelHelper.findProperty("currentCollaborationName", user.getProperty(), true)
				.setValue(GenericModelHelper.getName(userCollaboration));
		GJaxbUpdateUser upUser = new GJaxbUpdateUser();
		upUser.setNode(user);
		systemClient.updateUser(upUser);

		// connect user to collaboration
		GJaxbConnectUserToCollaboration connect = new GJaxbConnectUserToCollaboration();
		connect.setCollaborationId(userCollaboration.getId());
		connect.setUserId(user.getId());
		systemClient.connectUserToCollaboration(connect);

	}

	// @RolesAllowed({"admin"})
	@DELETE
	@Path("/user/users/{id}")
	public GJaxbRemoveUserResponse deleteUser(@Auth DWUser user, @PathParam("id") String id,
			@QueryParam("qid") String qid) throws Exception {
		GJaxbRemoveUserResponse response = null;
		assert user != null;

		try {
			LOG.debug("id = " + id);
			LOG.debug("qId = " + qid);

			GJaxbRemoveUser removeReq = new GJaxbRemoveUser();
			removeReq.setId(qid);
			response = this.systemClient.removeUser(removeReq);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/user/export/{userId}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public Response exportUser(@Auth DWUser user, @PathParam("userId") String userId,
			@QueryParam("userqId") String userqId, @QueryParam("userName") String userName) throws Exception {
		Response response = null;
		try {

			GJaxbGetUser req = new GJaxbGetUser();
			req.setId(userqId);
			GJaxbNode userNode = this.systemClient.getUser(req).getNode();

			Document docUser = XMLJAXBContext.getInstance().marshallAnyElement(userNode);

			JSONArray array = new JSONArray();
			JSONObject fileProject = new JSONObject();
			fileProject.put("fileName", RegExpHelper.toRegexFriendlyName("user_" + userqId) + "/"
					+ RegExpHelper.toRegexFriendlyName("user_" + userqId) + ".xml");
			fileProject.put("fileContent", XMLPrettyPrinter.print(docUser).getBytes());
			array.put(fileProject);

			// find resources in resourceFolder
			File rootDirectory = new File(RIOConstant.RESOURCES_FOLDER);
			rootDirectory = new File(rootDirectory, "users");
			rootDirectory = new File(rootDirectory, RegExpHelper.toRegexFriendlyName("user_" + userqId));
			rootDirectory = new File(rootDirectory, "pictures");
			IOFileFilter filter = new IOFileFilter() {

				@Override
				public boolean accept(File dir, String name) {
					if (name.startsWith(".")) {
						return false;
					}
					return true;
				}

				@Override
				public boolean accept(File file) {
					if (file.getName().startsWith(".")) {
						return false;
					}
					return true;
				}
			};

			if (rootDirectory.exists() && rootDirectory.isDirectory()) {
				Collection<File> files = FileUtils.listFilesAndDirs(rootDirectory, filter, filter);
				if (files != null && files.size() > 0) {
					for (File file : files) {
						JSONObject fileResources = new JSONObject();
						if (file.isFile()) {
							String path = file.getPath().substring(
									file.getPath().indexOf(RegExpHelper.toRegexFriendlyName("user_" + userqId))
											+ RegExpHelper.toRegexFriendlyName("user_" + userqId).length(),
									file.getPath().indexOf(file.getName()));
							path = path.replace("\\", "/");
							fileResources.put("fileName",
									RegExpHelper.toRegexFriendlyName("user_" + userqId) + "/" + path + file.getName());
							fileResources.put("fileContent", FileUtils.readFileToByteArray(file));
							array.put(fileResources);
						}
					}
				}
			}

			StreamingOutput stream = new StreamingOutput() {
				@Override
				public void write(OutputStream output) throws IOException, WebApplicationException {
					ZipOutputStream zos = new ZipOutputStream(output);

					for (int i = 0; i < array.length(); i++) {
						ZipEntry ze = new ZipEntry(array.getJSONObject(i).getString("fileName"));
						zos.putNextEntry(ze);
						zos.write((byte[]) array.getJSONObject(i).get("fileContent"));
						zos.closeEntry();
					}

					zos.flush();
					zos.close();
				}
			};

			response = Response.ok(stream).header("Content-Disposition", "attachment; filename=" + userName + ".zip")
					.header("Content-Transfer-Encoding", "binary").build();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	private GJaxbNode createFirstAdmin(EffectiveMetaModelManager sysMan) throws Exception {
		GJaxbEffectiveConceptType userMetaModel = sysMan
				.getConceptByType(new QName("http://fr.emac.gind/system", "User"));

		fr.emac.gind.modeler.genericmodel.GJaxbNode firstAdmin = GenericModelHelper
				.createNodeFromMetaModel(userMetaModel, true);
		firstAdmin.setId("user_id_admin_1");
		GenericModelHelper.findProperty("firstName", firstAdmin.getProperty()).setValue("R-IO");
		GenericModelHelper.findProperty("lastName", firstAdmin.getProperty()).setValue("Suite");
		GenericModelHelper.findProperty("login", firstAdmin.getProperty()).setValue("gind");
		GenericModelHelper.findProperty("theme", firstAdmin.getProperty()).setValue("dark-theme");
		GenericModelHelper.findProperty("name", firstAdmin.getProperty(), true)
				.setValue(GenericModelHelper.findProperty("firstName", firstAdmin.getProperty()).getValue() + " "
						+ GenericModelHelper.findProperty("lastName", firstAdmin.getProperty()).getValue());
		GenericModelHelper.findProperty("email", firstAdmin.getProperty()).setValue("gind@mines-albi.fr");
		GenericModelHelper.findProperty("password", firstAdmin.getProperty()).setValue(GindCrypto.encrypt("admin"));
		GenericModelHelper.findProperty("re-password", firstAdmin.getProperty()).setValue(GindCrypto.encrypt("admin"));
		
		JSONObject roles = new JSONObject(
				GenericModelHelper.findProperty("roles", firstAdmin.getProperty()).getValue());
		roles.getJSONObject("admin").put("checked", true);
		roles.getJSONObject("user").put("checked", true);
		GenericModelHelper.findProperty("roles", firstAdmin.getProperty()).setValue(roles.toString());
		GenericModelHelper.findProperty("activate sounds", firstAdmin.getProperty()).setValue("no");

		this.register(null, firstAdmin);
		return firstAdmin;
	}

	private GJaxbNode createCurrentUser(EffectiveMetaModelManager sysMan) throws Exception {
		GJaxbEffectiveConceptType userMetaModel = sysMan
				.getConceptByType(new QName("http://fr.emac.gind/system", "User"));

		fr.emac.gind.modeler.genericmodel.GJaxbNode currentUser = GenericModelHelper
				.createNodeFromMetaModel(userMetaModel, true);

		currentUser.setId("user_id_current_1");
		GenericModelHelper.findProperty("firstName", currentUser.getProperty()).setValue("${current_user_firstname}");
		GenericModelHelper.findProperty("lastName", currentUser.getProperty()).setValue("${current_user_lastname}");
		GenericModelHelper.findProperty("login", currentUser.getProperty()).setValue("${login}");
		GenericModelHelper.findProperty("theme", currentUser.getProperty()).setValue("dark-theme");
		GenericModelHelper.findProperty("name", currentUser.getProperty(), true)
				.setValue(GenericModelHelper.findProperty("firstName", currentUser.getProperty()).getValue() + " "
						+ GenericModelHelper.findProperty("lastName", currentUser.getProperty()).getValue());
		GenericModelHelper.findProperty("email", currentUser.getProperty()).setValue("${current_user_email}");
		GenericModelHelper.findProperty("password", currentUser.getProperty()).setValue(GindCrypto.encrypt("user"));
		GenericModelHelper.findProperty("re-password", currentUser.getProperty()).setValue(GindCrypto.encrypt("user"));
		
		JSONObject roles = new JSONObject(
				GenericModelHelper.findProperty("roles", currentUser.getProperty()).getValue());
		roles.getJSONObject("admin").put("checked", false);
		roles.getJSONObject("user").put("checked", true);
		GenericModelHelper.findProperty("roles", currentUser.getProperty()).setValue(roles.toString());
		GenericModelHelper.findProperty("activate sounds", currentUser.getProperty()).setValue("no");

		this.register(null, currentUser);
		return currentUser;
	}

	@GET
	@Path("/collaboration/collaborations/{userId}")
	public List<GJaxbNode> findCollaborationsOfUser(@Auth DWUser user, @PathParam("userId") String id,
			@QueryParam("qUserId") String qid) throws Exception {
		List<GJaxbNode> collaborations = null;
		try {
			GJaxbFindCollaborationsOfUser req = new GJaxbFindCollaborationsOfUser();
			if (qid != null) {
				req.setUserId(qid);
			} else {
				req.setUserId(id);
			}
			collaborations = this.systemClient.findCollaborationsOfUser(req).getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return collaborations;
	}

	@GET
	@Path("/collaboration/collaborations")
	public List<GJaxbNode> getAllCollaborations(@Auth DWUser user) throws Exception {
		List<GJaxbNode> collaborations = null;
		try {

			collaborations = this.systemClient.getAllCollaborations(new GJaxbGetAllCollaborations()).getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return collaborations;
	}

	@POST
	@Path("/collaboration/findCollaborationsByName")
	public List<GJaxbNode> findCollaborationsByName(@Auth DWUser user, String collaborationName) throws Exception {
		List<GJaxbNode> collaborations = new ArrayList<GJaxbNode>();
		try {
			GJaxbFindCollaborationsByName req = new GJaxbFindCollaborationsByName();
			req.setName(collaborationName);
			GJaxbFindCollaborationsByNameResponse resp = this.systemClient.findCollaborationsByName(req);
			if (resp.getNode() != null && !resp.getNode().isEmpty()) {
				collaborations.addAll(resp.getNode());
			}
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return collaborations;
	}

	@POST
	@Path("/collaboration/collaborations")
	public GJaxbNode addCollaboration(@Auth DWUser user, GJaxbNode collaboration) throws Exception {

		GJaxbFindCollaborationsByName reqName = new GJaxbFindCollaborationsByName();
		reqName.setName(GenericModelHelper.getName(collaboration));
		GJaxbFindCollaborationsByNameResponse respName = this.systemClient.findCollaborationsByName(reqName);

		if (respName.getNode() == null || respName.getNode().isEmpty()) {
			GJaxbAddCollaboration req = new GJaxbAddCollaboration();
			req.setNode(collaboration);
			this.systemClient.addCollaboration(req);
		} else {
			collaboration = respName.getNode().get(0);
		}

		return collaboration;
	}

	@GET
	@Path("/collaboration/collaboration/{collaborationId}")
	public GJaxbNode getCollaboration(@Auth DWUser user, @PathParam("collaborationId") String id,
			@QueryParam("qCollaborationId") String qid) throws Exception {
		GJaxbNode collaboration = null;
		try {
			GJaxbGetCollaboration req = new GJaxbGetCollaboration();
			if (qid != null) {
				req.setId(qid);
			} else {
				req.setId(id);
			}
			collaboration = this.systemClient.getCollaboration(req).getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return collaboration;
	}

	@GET
	@Path("/collaboration/knowledgeSpace/{knowledgeSpaceId}")
	public GJaxbNode getKnowledgeSpace(@Auth DWUser user, @PathParam("knowledgeSpaceId") String id,
			@QueryParam("qknowledgeSpaceId") String qid) throws Exception {
		GJaxbNode knowledgeSpace = null;
		try {
			GJaxbGetKnowledgeSpace req = new GJaxbGetKnowledgeSpace();
			if (qid != null) {
				req.setId(qid);
			} else {
				req.setId(id);
			}
			req.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
			if(Neo4JId.getCollaborationInId(req.getId()) != null) {
				req.getSelectedKnowledgeSpace().setCollaborationName(Neo4JId.getCollaborationInId(req.getId()));
			}
			if(Neo4JId.getKnowledgeSpaceInId(req.getId()) != null) {
				req.getSelectedKnowledgeSpace().setKnowledgeName(Neo4JId.getKnowledgeSpaceInId(req.getId()));
			}
			knowledgeSpace = this.systemClient.getKnowledgeSpace(req).getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return knowledgeSpace;
	}

	@GET
	@Path("/collaboration/collaboration/select/{collaborationId}")
	public GJaxbNode selectCollaboration(@Auth DWUser user, @PathParam("collaborationId") String id,
			@QueryParam("qCollaborationId") String qid) throws Exception {
		GJaxbNode collaboration = null;
		try {
			collaboration = getCollaboration(user, id, qid);

			user.setCurrentCollaborationName(GenericModelHelper.getName(collaboration));

			if (user.getCurrentKnowledgeSpaceName() != null) {
				user.setCurrentKnowledgeSpaceName(null);
			}
			this.updateUser(user, user.getUser());

			GJaxbConnectUserToCollaboration req = new GJaxbConnectUserToCollaboration();
			req.setCollaborationId(collaboration.getId());
			req.setUserId(user.getUserId());
			this.systemClient.connectUserToCollaboration(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return collaboration;
	}

	@GET
	@Path("/collaboration/knowledge/select/{collaborationId}/{knowledgeSpaceId}")
	public GJaxbGenericModel selectCollaborationAndKnowledgeSpace(@Auth DWUser user,
			@PathParam("collaborationId") String cid, @QueryParam("qCollaborationId") String qcid,
			@PathParam("knowledgeSpaceId") String kid, @QueryParam("qKnowledgeSpaceId") String qkid) throws Exception {
		GJaxbGenericModel model = new GJaxbGenericModel();
		try {
			GJaxbNode collaboration = getCollaboration(user, cid, qcid);
			GJaxbNode knowledge = getKnowledgeSpace(user, kid, qkid);

			model.getNode().add(collaboration);
			model.getNode().add(knowledge);

			user.setCurrentCollaborationName(GenericModelHelper.getName(collaboration));
			user.setCurrentKnowledgeSpaceName(GenericModelHelper.getName(knowledge));

			this.updateUser(user, user.getUser());

			GJaxbConnectUserToCollaboration req = new GJaxbConnectUserToCollaboration();
			req.setCollaborationId(collaboration.getId());
			req.setUserId(user.getUserId());
			this.systemClient.connectUserToCollaboration(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return model;
	}

	@GET
	@Path("/collaboration/collaboration/export/{collaborationId}")
	public String exportCollaboration(@Auth DWUser user, @PathParam("collaborationId") String id,
			@QueryParam("qCollaborationId") String qid) throws Exception {
		String response = null;
		try {

			GJaxbExportCollaboration req = new GJaxbExportCollaboration();
			req.setCollaborationId(qid);
			GJaxbGenericModel resp = this.systemClient.exportCollaboration(req).getGenericModel();

			resp.getNode().stream().filter(n -> {
				return n.getType().equals(new QName("http://fr.emac.gind/system", "User"));
			}).forEach(u -> {
				u.getProperty().clear();
			});

			JSONObject json = new JSONObject();
			json.put("buffer", XMLPrettyPrinter.print(XMLJAXBContext.getInstance().marshallAnyElement(resp)));

			response = json.toString();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/collaboration/knowledges/{collaborationId}")
	public List<GJaxbNode> getKnowledgesInCollaboration(@Auth DWUser user, @PathParam("collaborationId") String id,
			@QueryParam("qCollaborationId") String qid) throws Exception {
		List<GJaxbNode> knowledges = null;
		try {
			GJaxbGetKnowledgeSpacesInCollaboration req = new GJaxbGetKnowledgeSpacesInCollaboration();
			if (qid != null) {
				req.setCollaborationId(qid);
			} else {
				req.setCollaborationId(id);
			}
			knowledges = this.systemClient.getKnowledgeSpacesInCollaboration(req).getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return knowledges;
	}

	@GET
	@Path("/collaboration/users/{collaborationId}")
	public List<GJaxbNode> getUsersInCollaboration(@Auth DWUser user, @PathParam("collaborationId") String id,
			@QueryParam("qCollaborationId") String qid) throws Exception {
		List<GJaxbNode> users = null;
		try {
			GJaxbGetUsersInCollaboration req = new GJaxbGetUsersInCollaboration();
			if (qid != null) {
				req.setCollaborationId(qid);
			} else {
				req.setCollaborationId(id);
			}
			users = this.systemClient.getUsersInCollaboration(req).getNode();

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return users;
	}

	@POST
	@Path("/collaboration/connectUserToCollaboration")
	public GJaxbConnectUserToCollaborationResponse connectUserToCollaboration(@Auth DWUser user,
			GJaxbConnectUserToCollaboration reqConnect) throws Exception {
		GJaxbConnectUserToCollaborationResponse response = null;
		try {
			response = this.systemClient.connectUserToCollaboration(reqConnect);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@DELETE
	@Path("/collaboration/collaborations/{id}")
	public GJaxbRemoveCollaborationResponse deleteCollaboration(@Auth DWUser user, @PathParam("id") String id,
			@QueryParam("qid") String qid) throws Exception {
		GJaxbRemoveCollaborationResponse response = null;
		assert user != null;

		try {
			LOG.debug("id = " + id);
			LOG.debug("qId = " + qid);

			GJaxbRemoveCollaboration req = new GJaxbRemoveCollaboration();
			req.setId(qid);
			response = this.systemClient.removeCollaboration(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/collaboration/attachKnowledgeSpaceToCollaboration")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public String attachKnowledgeSpaceToCollaboration(@Auth DWUser user,
			@QueryParam("knowledgeSpaceId") String knowledgeSpaceId,
			@QueryParam("knowledgeSpaceName") String knowledgeSpaceName,
			@QueryParam("collaborationName") String collaborationName) throws Exception {
		JSONObject res = new JSONObject();

		if (collaborationName == null) {
			collaborationName = user.getCurrentCollaborationName();
		}
		if (collaborationName == null) {
			throw new Exception("Collaboration name cannot be null !!!");
		}

		// create new association
		GJaxbFindCollaborationsByName reqCollab = new GJaxbFindCollaborationsByName();
		reqCollab.setName(collaborationName);
		GJaxbFindCollaborationsByNameResponse reqCollabResp = this.systemClient.findCollaborationsByName(reqCollab);
		if (reqCollabResp == null || reqCollabResp.getNode() == null || reqCollabResp.getNode().isEmpty()) {
			throw new Exception("Impossible to find collaboration : " + collaborationName);
		}
		GJaxbNode collaboration = reqCollabResp.getNode().get(0);

		GJaxbAttachKnowledgeSpaceToCollaboration req = new GJaxbAttachKnowledgeSpaceToCollaboration();
		if (collaboration != null) {
			req.setCollaborationId(collaboration.getId());
		}

		knowledgeSpaceId = Neo4JId.cleanId(knowledgeSpaceId);

		req.setKnowledgeSpaceId(knowledgeSpaceId);
		req.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
		req.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
		req.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);
		this.systemClient.attachKnowledgeSpaceToCollaboration(req);

		GJaxbGetKnowledgeSpace reqGetKn = new GJaxbGetKnowledgeSpace();
		reqGetKn.setId(knowledgeSpaceId);
		reqGetKn.setSelectedKnowledgeSpace(new GJaxbSelectedKnowledgeSpace());
		reqGetKn.getSelectedKnowledgeSpace().setCollaborationName(collaborationName);
		reqGetKn.getSelectedKnowledgeSpace().setKnowledgeName(knowledgeSpaceName);
		GJaxbGetKnowledgeSpaceResponse respGetKn = this.systemClient.getKnowledgeSpace(reqGetKn);

		if (respGetKn.getNode() == null) {
			throw new Exception("Impossible to find knowledgespace with id:" + knowledgeSpaceId);
		}

		if (!user.isFake()) {
			user.setCurrentKnowledgeSpaceName(knowledgeSpaceName);
			this.updateUser(user, user.getUser());
		}
		return res.toString();
	}

	@DELETE
	@Path("/collaboration/knowledgeSpace/{knowledgeSpaceId}")
	public void deleteKnowledgeSpace(@Auth DWUser user, @PathParam("knowledgeSpaceId") String id,
			@QueryParam("qknowledgeSpaceId") String qid) throws Exception {
		try {
			GJaxbRemoveKnowledgeSpace req = new GJaxbRemoveKnowledgeSpace();
			req.setId(qid);
			this.systemClient.removeKnowledgeSpace(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
	}

	@GET
	@Path("/ping")
	@Consumes(MediaType.APPLICATION_JSON)
	public String ping(@Auth Optional<DWUser> userOpt) throws Exception {
		LOG.debug("try to ping ");
		try {

			JSONObject json = new JSONObject();
			json.put("ping", true);
			return json.toString();
		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return null;
	}

	public static GJaxbNode createCollaboration(String collaborationName, SystemGov systemClient,
			GJaxbEffectiveConceptType collaborationMetaModel) throws Exception {
		// create collaboration
		GJaxbNode collaboration = new GJaxbNode();
		collaboration.setType(collaborationMetaModel.getType());

		GenericModelHelper.findProperty("name", collaboration.getProperty(), true).setValue(collaborationName);

		GJaxbAddCollaboration requestUserCollaboration = new GJaxbAddCollaboration();
		requestUserCollaboration.setNode(collaboration);
		GJaxbAddCollaborationResponse respCollab = systemClient.addCollaboration(requestUserCollaboration);
		collaboration.setId(respCollab.getId());
		return collaboration;

	}

}
