/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataParam;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import fr.emac.gind.commons.utils.xml.DOMUtil;
import fr.emac.gind.dataset.GJaxbDataset;
import fr.emac.gind.event.consumer.AbstractNotifierClient;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.RestWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.marshaller.XMLJAXBContext;
import fr.emac.gind.rio.dw.resources.gov.bo.SendEventToProviderTO;
import fr.emac.gind.sensors.controler.SensorControler;
import fr.emac.gind.sensors.controler.SensorControlerClient;
import fr.emac.gind.sensors.controler.data.GJaxbAddBreakpoints;
import fr.emac.gind.sensors.controler.data.GJaxbAddBreakpointsResponse;
import fr.emac.gind.sensors.controler.data.GJaxbAddDatasetAndRefreshModelOnSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbAddDatasetAndRefreshModelOnSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbCreateSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbCreateSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbFindSensorControlerByProtocol;
import fr.emac.gind.sensors.controler.data.GJaxbFindSensorControlerByProtocolResponse;
import fr.emac.gind.sensors.controler.data.GJaxbGetBreakpoints;
import fr.emac.gind.sensors.controler.data.GJaxbGetBreakpointsResponse;
import fr.emac.gind.sensors.controler.data.GJaxbGetSensorControlers;
import fr.emac.gind.sensors.controler.data.GJaxbGetSensorControlersResponse;
import fr.emac.gind.sensors.controler.data.GJaxbGetXMLDataOfDataset;
import fr.emac.gind.sensors.controler.data.GJaxbGetXMLDataOfDatasetResponse;
import fr.emac.gind.sensors.controler.data.GJaxbPauseDataSetOnTopicOfSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbPauseDataSetOnTopicOfSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbPlayDataSetOnTopicOfSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbPlayDataSetOnTopicOfSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbRemoveBreakpoints;
import fr.emac.gind.sensors.controler.data.GJaxbRemoveBreakpointsResponse;
import fr.emac.gind.sensors.controler.data.GJaxbRemoveDatasetOnSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbRemoveDatasetOnSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbSendEventToSensor;
import fr.emac.gind.sensors.controler.data.GJaxbSendEventToSensor.Payload;
import fr.emac.gind.sensors.controler.data.GJaxbSendEventToSensorResponse;
import fr.emac.gind.sensors.controler.data.GJaxbStopDataSetOnTopicOfSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbStopDataSetOnTopicOfSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbUpdateDatasetAndModelConfiguration;
import fr.emac.gind.sensors.controler.data.GJaxbUpdateDatasetAndModelConfigurationResponse;
import fr.emac.gind.sensors.controler.data.GJaxbUpdateDatasetAndModelOnSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbUpdateDatasetAndModelOnSensorControlerResponse;
import fr.emac.gind.sensors.controler.data.GJaxbUpdateSensorControler;
import fr.emac.gind.sensors.controler.data.GJaxbUpdateSensorControlerResponse;
import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/sensorControler")
//@Api("eventSensor")
@Produces(MediaType.APPLICATION_JSON)
public class SensorControlerResource {

	private static Logger LOG = LoggerFactory.getLogger(SensorControlerResource.class);

	private SensorControler sensorControlerClient = null;
	private FileResource fileResource = null;
	private Map<QName, String> subscriptionIds = Collections.synchronizedMap(new HashMap<QName, String>());
	private AbstractNotifierClient notifier = null;
	private Configuration conf = null;

	public SensorControlerResource(Configuration conf, FileResource fileResource, AbstractNotifierClient notifier)
			throws Exception {
		this.sensorControlerClient = SensorControlerClient
				.createClient(conf.getProperties().get("sensor-controler-server"));
		this.fileResource = fileResource;
		this.notifier = notifier;
		this.conf = conf;
	}

	@POST
	@Path("/getSensorControlers")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetSensorControlersResponse getSensorControlers(@Auth DWUser user, GJaxbGetSensorControlers request) throws Exception {

		GJaxbGetSensorControlersResponse response = null;
		try {
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			response = this.sensorControlerClient.getSensorControlers(request);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/createSensorControler")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbCreateSensorControlerResponse createSensorControler(@Auth DWUser user,
			GJaxbCreateSensorControler request) throws Exception {

		GJaxbCreateSensorControlerResponse response = null;
		try {
			response = this.sensorControlerClient.createSensorControler(request);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/updateSensorControler")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbUpdateSensorControlerResponse updateSensorControler(@Auth DWUser user,
			GJaxbUpdateSensorControler request) throws Exception {

		GJaxbUpdateSensorControlerResponse response = null;
		try {
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			response = this.sensorControlerClient.updateSensorControler(request);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	
	
	@POST
	@Path("/launchDataSetOnTopic")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public synchronized GJaxbPlayDataSetOnTopicOfSensorControlerResponse launchDataSetOnTopic(@Auth DWUser user,
			GJaxbPlayDataSetOnTopicOfSensorControler req) throws Exception {
		LOG.debug("play dataset on topic: " + req.getDatasetHandler().getName());
		GJaxbPlayDataSetOnTopicOfSensorControlerResponse response = null;
		try {

			// Subscribe for R-IOSEMIT
			if (subscriptionIds.get(req.getDatasetHandler().getTopicToSubscribe()) == null) {
				String subscriptionIdRIOSEMIT = this.notifier.subscribeOn(req.getInternalUrl(),
						req.getDatasetHandler().getTopicToSubscribe());
				subscriptionIds.put(req.getDatasetHandler().getTopicToSubscribe(), subscriptionIdRIOSEMIT);

			}

			req.setCollaborationName(user.getCurrentCollaborationName());
			req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			response = this.sensorControlerClient.playDataSetOnTopicOfSensorControler(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/pauseDataSetOnTopic")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbPauseDataSetOnTopicOfSensorControlerResponse pauseDataSetOnTopic(@Auth DWUser user,
			GJaxbPauseDataSetOnTopicOfSensorControler req) throws Exception {

		GJaxbPauseDataSetOnTopicOfSensorControlerResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = this.sensorControlerClient.pauseDataSetOnTopicOfSensorControler(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/stopDataSetOnTopic")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbStopDataSetOnTopicOfSensorControlerResponse stopDataSetOnTopic(@Auth DWUser user,
			GJaxbStopDataSetOnTopicOfSensorControler req) throws Exception {

		GJaxbStopDataSetOnTopicOfSensorControlerResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = this.sensorControlerClient.stopDataSetOnTopicOfSensorControler(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/addBreakpoints")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbAddBreakpointsResponse addBreakpoints(@Auth DWUser user, GJaxbAddBreakpoints req) throws Exception {

		GJaxbAddBreakpointsResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorControlerClient.addBreakpoints(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/removeBreakpoints")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbRemoveBreakpointsResponse removeBreakpoints(@Auth DWUser user, GJaxbRemoveBreakpoints req)
			throws Exception {

		GJaxbRemoveBreakpointsResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorControlerClient.removeBreakpoints(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/getBreakpoints")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetBreakpointsResponse getBreakpoints(@Auth DWUser user, GJaxbGetBreakpoints req) throws Exception {

		GJaxbGetBreakpointsResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorControlerClient.getBreakpoints(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@GET
	@Path("/findSensorControlerByProtocol/{type}/{nature}/{gameScenarioId}/{collaborationName}/{knowledgeSpaceName}")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbFindSensorControlerByProtocolResponse findSensorControlerByProtocol(@Auth DWUser user,
			@PathParam("type") String type, 
			@PathParam("nature") String nature, 
			@PathParam("gameScenarioId") String gameScenarioId,
			@PathParam("collaborationName") String collaborationName,
			@PathParam("knowledgeSpaceName") String knowledgeSpaceName) throws Exception {

		GJaxbFindSensorControlerByProtocolResponse response = null;
		try {
			GJaxbFindSensorControlerByProtocol request = new GJaxbFindSensorControlerByProtocol();
			request.setGameScenarioId(gameScenarioId);
			request.setCollaborationName(collaborationName);
			request.setKnowledgeSpaceName(knowledgeSpaceName);
			request.getType().add(type);
			request.getNature().add(nature);
			response = this.sensorControlerClient.findSensorControlerByProtocol(request);

		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/sendEventToSensor")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbSendEventToSensorResponse sendEventToSensor(@Auth DWUser user, SendEventToProviderTO parameters) {
		GJaxbSendEventToSensorResponse res = null;
		try {
			GJaxbSendEventToSensor request = new GJaxbSendEventToSensor();
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			request.setDataSourceId(parameters.getDataSourceId());
		    request.setGameScenarioId(parameters.getGameScenarioId());
			Document payload = DOMUtil.getInstance()
					.parse(new ByteArrayInputStream(parameters.getPayload().getBytes()));
			request.setPayload(new Payload());
			request.getPayload().setAny(payload.getDocumentElement());
			request.setStatus(parameters.getStatus());
			request.getLabel().addAll(parameters.getLabel());
			res = sensorControlerClient.sendEventToSensor(request);
		} catch (Exception e) {
			throw new RestWebApplicationException(e);
		}
		return res;
	}

	public static GJaxbDataset getDataset(FileResource fileResource, DWUser user, String datasetURL) throws Exception {
		GJaxbDataset dataset = null;
		if (datasetURL != null && !datasetURL.isBlank()) {
			String buffer = fileResource.download(user, datasetURL);
			if (buffer == null) {
				throw new Exception("EMPTY PATH to Dataset file");
			}
			Document doc = DOMUtil.getInstance().parse(new ByteArrayInputStream(buffer.getBytes()));
			dataset = XMLJAXBContext.getInstance().unmarshallDocument(doc, GJaxbDataset.class);
		}
		return dataset;
	}

	@POST
	@Path("/addDatasetAndRefreshModelOnSensorControlerFromFile")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	public GJaxbAddDatasetAndRefreshModelOnSensorControlerResponse addDatasetAndRefreshModelOnSensorControler(@Auth DWUser user,
			@FormDataParam("file") InputStream is, @FormDataParam("file") FormDataContentDisposition fileDetail,
			@QueryParam("data") String data) throws Exception {
		GJaxbAddDatasetAndRefreshModelOnSensorControlerResponse res = null;
		LOG.debug("\n\n*************************** injectDataset *******************\n\n");
		try {
			Document doc = DOMUtil.getInstance().parse(is);
			GJaxbDataset dataset = XMLJAXBContext.getInstance().unmarshallDocument(doc, GJaxbDataset.class);
			JSONObject params = new JSONObject(data);
			LOG.debug("");
			GJaxbAddDatasetAndRefreshModelOnSensorControler request = new GJaxbAddDatasetAndRefreshModelOnSensorControler();
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			
			request.setDataset(dataset);
			String sensorId = params.getString("sensorId");
			request.setDataSourceId(sensorId);
			String gameScenarioId = params.getString("gameScenarioId");
			request.setGameScenarioId(gameScenarioId);
			res = this.sensorControlerClient.addDatasetAndRefreshModelOnSensorControler(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
			;
		}

		return res;
	}

	@POST
	@Path("/updateDatasetOnSensorControlerFromFile")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	public GJaxbUpdateDatasetAndModelOnSensorControlerResponse updateDatasetOnSensorControler(@Auth DWUser user,
			@FormDataParam("file") InputStream is, @FormDataParam("file") FormDataContentDisposition fileDetail,
			@QueryParam("data") String data) throws Exception {
		GJaxbUpdateDatasetAndModelOnSensorControlerResponse res = null;
		LOG.debug("\n\n*************************** injectDataset *******************\n\n");
		try {
			Document doc = DOMUtil.getInstance().parse(is);
			GJaxbDataset dataset = XMLJAXBContext.getInstance().unmarshallDocument(doc, GJaxbDataset.class);
			JSONObject params = new JSONObject(data);
			LOG.debug("");
			GJaxbUpdateDatasetAndModelOnSensorControler request = new GJaxbUpdateDatasetAndModelOnSensorControler();
			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			request.setDataset(dataset);
			String sensorId = params.getString("sensorId");
			request.setDataSourceId(sensorId);
			String gameScenarioId = params.getString("gameScenarioId");
			request.setGameScenarioId(gameScenarioId);
			res = this.sensorControlerClient.updateDatasetAndModelOnSensorControler(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
			;
		}

		return res;
	}

	@POST
	@Path("/addDatasetAndRefreshModelOnSensorControlerFromJson")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbAddDatasetAndRefreshModelOnSensorControlerResponse addDatasetAndRefreshModelOnSensorControler(@Auth DWUser user, String jsonRequest)
			throws Exception {
		GJaxbAddDatasetAndRefreshModelOnSensorControlerResponse res = null;
		LOG.debug("\n\n*************************** injectDataset *******************\n\n");
		try {
			GJaxbAddDatasetAndRefreshModelOnSensorControler request = JSONJAXBContext.getInstance().unmarshall(
					"{ \"addDatasetOnSensorControler\":" + jsonRequest + "}", GJaxbAddDatasetAndRefreshModelOnSensorControler.class);

			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());

			res = this.sensorControlerClient.addDatasetAndRefreshModelOnSensorControler(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
			;
		}

		return res;
	}

	@POST
	@Path("/updateDatasetAndModelOnSensorControlerFromJson")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbUpdateDatasetAndModelOnSensorControlerResponse updateDatasetAndModelOnSensorControler(@Auth DWUser user,
			String jsonRequest) throws Exception {
		GJaxbUpdateDatasetAndModelOnSensorControlerResponse res = null;
		LOG.debug("\n\n*************************** injectDataset *******************\n\n");
		try {
			GJaxbUpdateDatasetAndModelOnSensorControler request = JSONJAXBContext.getInstance().unmarshall(
					"{ \"updateDatasetAndModelOnSensorControler\":" + jsonRequest + "}",
					GJaxbUpdateDatasetAndModelOnSensorControler.class);

			request.setCollaborationName(user.getCurrentCollaborationName());
			request.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());

			res = this.sensorControlerClient.updateDatasetAndModelOnSensorControler(request);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
			;
		}

		return res;
	}

	@POST
	@Path("/removeDatasetOnSensorControler")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbRemoveDatasetOnSensorControlerResponse removeDatasetOnSensorControler(@Auth DWUser user,
			GJaxbRemoveDatasetOnSensorControler req) throws Exception {
		GJaxbRemoveDatasetOnSensorControlerResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorControlerClient.removeDatasetOnSensorControler(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/getXMLDataOfDataset")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbGetXMLDataOfDatasetResponse getDataOfDataset(@Auth DWUser user, GJaxbGetXMLDataOfDataset req)
			throws Exception {
		GJaxbGetXMLDataOfDatasetResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorControlerClient.getXMLDataOfDataset(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}

	@POST
	@Path("/updateDatasetAndModelConfiguration")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public GJaxbUpdateDatasetAndModelConfigurationResponse updateDatasetAndModelConfiguration(@Auth DWUser user,
			GJaxbUpdateDatasetAndModelConfiguration req) throws Exception {
		GJaxbUpdateDatasetAndModelConfigurationResponse response = null;
		try {
			if (req.getCollaborationName() == null) {
				req.setCollaborationName(user.getCurrentCollaborationName());
			}
			if (req.getKnowledgeSpaceName() == null) {
				req.setKnowledgeSpaceName(user.getCurrentKnowledgeSpaceName());
			}

			response = sensorControlerClient.updateDatasetAndModelConfiguration(req);

		} catch (Exception e) {
			GindWebApplicationException.manageError(e, this);
		}
		return response;
	}


}