/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.r.iosepe;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.emac.gind.campaign.manager.client.CampaignManagerNotifierClient;
import fr.emac.gind.campaign.manager.server.CampaignManagerSubscriber;
import fr.emac.gind.campaign.manager.server.CampaignManagerWebService;
import fr.emac.gind.commons.utils.net.IPUtil.TRANSPORT_PROTOCOL;
import fr.emac.gind.commons.utils.regexp.RegExpHelper;
import fr.emac.gind.commons.utils.ws.LocalRegistry;
import fr.emac.gind.commons.utils.ws.SPIWebServicePrimitives;
import fr.emac.gind.eventtype.GJaxbAddCampaignSolutionEvent;
import fr.emac.gind.eventtype.GJaxbCreateCampaignEvent;
import fr.emac.gind.eventtype.GJaxbRemoveCampaignSolutionEvent;
import fr.emac.gind.eventtype.GJaxbUpdateCampaignEvent;
import fr.emac.gind.generic.application.utils.RestResourceManager;
import fr.emac.gind.marshaller.JSONJAXBContext;
import fr.emac.gind.r.ioda.RIODAService;
import fr.emac.gind.rio.dw.resources.ExcelResource;
import fr.emac.gind.rio.dw.resources.JsonResource;
import fr.emac.gind.websocket.command.data.GJaxbGetResult;
import io.dropwizard.core.Configuration;
import io.dropwizard.core.setup.Environment;

/**
 *
 *
 * @author Nicolas Salatge
 */
public class RIOSEPEService extends RIODAService {

	private CampaignManagerNotifierClient client = null;

	private static Logger LOG = LoggerFactory.getLogger(RIOSEPEService.class.getName());

	public RIOSEPEService() throws Exception {
		this(new HashMap<String, Object>());
	}

	public RIOSEPEService(Map<String, Object> context) throws Exception {
		super(context);
	}

	@Override
	public String getName() {
		return "r-iosepe";
	}

	public String getRedirection() {
		return "/riosepe";
	}

	@SuppressWarnings("unchecked")
	@Override
	public RIOSEPEApplicationContext createApplicationContext() throws Exception {
		return new RIOSEPEApplicationContext(this);
	}

	@Override
	public void doRun(Configuration configuration, Environment environment, RestResourceManager resourcesManager)
			throws Exception {
		super.doRun(configuration, environment, resourcesManager);
		resourcesManager.addResource(new ExcelResource(this.conf));
		resourcesManager.addResource(new JsonResource(this.conf));

		final String subscriberCampaignAddress = this.conf.getProperties().get("campaign-manager");

		final String notifierClientAddress = SPIWebServicePrimitives.createAddress(TRANSPORT_PROTOCOL.HTTP,
				conf.getHost(), Integer.parseInt(conf.getProperties().get("riosepe-notifier-port")),
				Integer.parseInt(conf.getProperties().get("proxy-port")),
				"/RIOSEPENotifier");

		this.client = new CampaignManagerNotifierClient(URI.create(subscriberCampaignAddress).toURL(),
				URI.create(notifierClientAddress).toURL()) {

			@Override
			public void onCreateCampaignEvent(GJaxbCreateCampaignEvent event) {
				try {
					GJaxbGetResult request = new GJaxbGetResult();
					request.setWebsocketId("/riosepe" + "/" + RegExpHelper.toRegexFriendlyName(event.getCollaborationName()).hashCode() + "/" + RegExpHelper.toRegexFriendlyName(event.getKnowledgeSpaceName()).hashCode());
					request.setJsonResult(JSONJAXBContext.getInstance().marshallAnyElement(event));

					WEB_SOCKET_COMMAND.getResult(request);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			
			@Override
			public void onUpdateCampaignEvent(GJaxbUpdateCampaignEvent event) {
				try {
					GJaxbGetResult request = new GJaxbGetResult();
					request.setJsonResult(JSONJAXBContext.getInstance().marshallAnyElement(event));

					request.setWebsocketId("/riosepe" + "/" + RegExpHelper.toRegexFriendlyName(event.getCollaborationName()).hashCode() + "/" + RegExpHelper.toRegexFriendlyName(event.getKnowledgeSpaceName()).hashCode());
					WEB_SOCKET_COMMAND.getResult(request);
					
					request.setWebsocketId("/riosepe" + "/" + RegExpHelper.toRegexFriendlyName(event.getCollaborationName()).hashCode() + "/" + RegExpHelper.toRegexFriendlyName(event.getKnowledgeSpaceName()).hashCode() + "/" + event.getCampaign().getCampaignId());
					WEB_SOCKET_COMMAND.getResult(request);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}

			@Override
			public void onAddSolutionEvent(GJaxbAddCampaignSolutionEvent event) {
				try {

					GJaxbGetResult request = new GJaxbGetResult();
					request.setWebsocketId("/riosepe" + "/" + RegExpHelper.toRegexFriendlyName(event.getCollaborationName()).hashCode() + "/" + RegExpHelper.toRegexFriendlyName(event.getKnowledgeSpaceName()).hashCode() + "/" + event.getCampaignSolution().getCampaignId());
					request.setJsonResult(JSONJAXBContext.getInstance().marshallAnyElement(event));

					WEB_SOCKET_COMMAND.getResult(request);
				} catch (Exception e) {
					e.printStackTrace();
				}

			}

			@Override
			public void onRemoveSolutionEvent(GJaxbRemoveCampaignSolutionEvent event) {
				try {
					GJaxbGetResult request = new GJaxbGetResult();
					request.setWebsocketId("/riosepe" + "/" + RegExpHelper.toRegexFriendlyName(event.getCollaborationName()).hashCode() + "/" + RegExpHelper.toRegexFriendlyName(event.getKnowledgeSpaceName()).hashCode() + "/" + event.getCampaignSolution().getCampaignId());
					request.setJsonResult(JSONJAXBContext.getInstance().marshallAnyElement(event));

					WEB_SOCKET_COMMAND.getResult(request);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}

		};
		this.client.start();

		CampaignManagerWebService campaignManagerWebService = LocalRegistry.getInstance()
				.findWS(CampaignManagerWebService.class);
		CampaignManagerSubscriber localTestCampaignSubscriber = ((SPIWebServicePrimitives) campaignManagerWebService)
				.getImplementation(CampaignManagerSubscriber.class);

		this.client.subscribeToCampaignManager(localTestCampaignSubscriber);
	}

	@Override
	public void stop() throws Exception {
		if (this.client != null) {
			this.client.stop();
		}
		super.stop();
	}

}
