/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.rio.dw.resources;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;

import org.apache.poi.openxml4j.util.ZipSecureFile;

import fr.emac.gind.campaign.manager.CampaignManagerItf;
import fr.emac.gind.campaign.manager.client.CampaignManagerClient;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetCampaignResponse;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetSolutionsFromCampaign;
import fr.emac.gind.campaign.manager.client.data.GJaxbGetSolutionsFromCampaignResponse;
import fr.emac.gind.campaign.manager.data.model.GJaxbCampaign;
import fr.emac.gind.campaign.manager.data.model.GJaxbCampaignSolution;
import fr.emac.gind.campaign.manager.data.model.GJaxbStatistic.StatisticMetric;
import fr.emac.gind.commons.utils.excel.ExcelProjectManager;
import fr.emac.gind.commons.utils.excel.ExcelSheet;
import fr.emac.gind.commons.utils.excel.ExcelTable;
import fr.emac.gind.commons.utils.excel.helpers.TableInfo;
import fr.emac.gind.commons.utils.maths.DoubleHelper;
import fr.emac.gind.generic.application.GindWebApplicationException;
import fr.emac.gind.generic.application.users.DWUser;
import fr.emac.gind.launcher.Configuration;
import fr.emac.gind.models.process.simulation.results.GJaxbIndicator;
import fr.emac.gind.workflow.AbstractProcessGenerator;

/*
 * #%L
 * dw-logistics-modeler
 * %%
 * Copyright (C) 2014 - 2015 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import io.dropwizard.auth.Auth;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 *
 *
 * @author Nicolas Salatge
 */
@Path("/{app}/r-iosepe/campaign/excel")
//@Api("excel")
@Consumes(MediaType.APPLICATION_JSON)
public class ExcelResource {

	private CampaignManagerItf campaignClient = null;
	private SimpleDateFormat formatter = new SimpleDateFormat("YYYY-MM-dd HH:mm");

	private SimpleDateFormat formatterLong2Time = new SimpleDateFormat("mm:ss");

	private List<AbstractProcessGenerator> processGenerators = new ArrayList<AbstractProcessGenerator>();

	public ExcelResource(Configuration conf) throws Exception {
		if (conf != null) {
			this.campaignClient = CampaignManagerClient
					.createClient((String) conf.getProperties().get("campaign-manager"));
		}
		this.initProcessGenerator();
	}

	public void initProcessGenerator() {
		ServiceLoader<AbstractProcessGenerator> processGeneratorLoader = ServiceLoader
				.load(AbstractProcessGenerator.class);
		// Discover and register the available commands
		processGenerators.clear();
		processGeneratorLoader.reload();
		Iterator<AbstractProcessGenerator> processGeneratorsIterator = processGeneratorLoader.iterator();
		while (processGeneratorsIterator.hasNext()) {
			AbstractProcessGenerator processGenerator = processGeneratorsIterator.next();

			this.processGenerators.add(processGenerator);
		}
	}

	@POST
	@Path("/exportCampaign")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public Response exportAsExcel(@Auth DWUser user, String campaignId) throws Exception {
		Response res = null;
		try {

			GJaxbGetCampaign request = new GJaxbGetCampaign();
			request.setCampaignId(campaignId);
			GJaxbGetCampaignResponse response = this.campaignClient.getCampaign(request);

			GJaxbCampaign campaign = response.getCampaign();

			GJaxbGetSolutionsFromCampaign getParam = new GJaxbGetSolutionsFromCampaign();
			getParam.setCampaignId(campaign.getCampaignId());
			GJaxbGetSolutionsFromCampaignResponse responseGet = this.campaignClient.getSolutionsFromCampaign(getParam);

			ExcelProjectManager excel = new ExcelProjectManager();
			createAllSheets(campaign, responseGet.getCampaignSolution(), excel);

			ZipSecureFile.setMinInflateRatio(-1.0d);
			res = Response.ok(excel.writeInByteArray().toByteArray(), MediaType.APPLICATION_OCTET_STREAM).build();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return res;
	}

	@POST
	@Path("/exportSolutions")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_OCTET_STREAM)
	public Response exportAsExcel(@Auth DWUser user, List<GJaxbCampaignSolution> scenarios) throws Exception {
		Response res = null;
		try {

			ExcelProjectManager excel = new ExcelProjectManager();
			createSheetSolutions(excel, scenarios);

			/**
			 * if(!scenarios.isEmpty()) { GJaxbCampaignSolution firstSolution = scenarios.get(0);
			 * 
			 * for(AbstractProcessGenerator gen: this.processGenerators) {
			 * if(gen.getCampaignAnalyzer() != null) {
			 * if(firstSolution.getSpecificAnalysis() != null &&
			 * gen.getDeductionStrategy().getClass().getSimpleName().contains(firstSolution.getSpecificAnalysis().getStrategyUsed()))
			 * { gen.getCampaignAnalyzer().createSpecificSheetSolutions(excel, campaign,
			 * scenarios); } } } }
			 */

			ZipSecureFile.setMinInflateRatio(-1.0d);
			res = Response.ok(excel.writeInByteArray().toByteArray(), MediaType.APPLICATION_OCTET_STREAM).build();
		} catch (Exception e) {
			e.printStackTrace();
			GindWebApplicationException.manageError(e, this);
		}
		return res;
	}

	public void createAllSheets(GJaxbCampaign campaign, List<GJaxbCampaignSolution> scenarios, ExcelProjectManager excel)
			throws Exception {
		createSheetCampaign(excel, campaign);
		createSheetSolutions(excel, scenarios);

		if (!scenarios.isEmpty()) {
			GJaxbCampaignSolution firstSolution = scenarios.get(0);

//			for (AbstractProcessGenerator gen : this.processGenerators) {
//				if (gen.getCampaignAnalyzer() != null) {
//					if (firstSolution.getSpecificAnalysis() != null && gen.getDeductionStrategy().getClass()
//							.getSimpleName().contains(firstSolution.getSpecificAnalysis().getStrategyUsed())) {
//						gen.getCampaignAnalyzer().createSpecificSheetSolutions(excel, campaign, scenarios);
//					}
//				}
//			}

		}
	}

	public void createSheetCampaign(ExcelProjectManager excel, GJaxbCampaign campaign) throws Exception {
		ExcelSheet sheet = excel.addSheet("Campaign Information");
		int currentRow = 2;
		int currentCol = 2;
		int maxWidth = 0;

		// 1st table
		TableInfo tableInfo = createBasicInformationCampaignTable(sheet, currentRow, currentCol, campaign);

		tableInfo = createNumericalMetricCampaignTable(sheet, currentRow, tableInfo.getTotalWidth() + 4, campaign);
	}

	public void createSheetSolutions(ExcelProjectManager excel, List<GJaxbCampaignSolution> scenario) throws Exception {
		ExcelSheet sheet = excel.addSheet("Solutions");
		int currentRow = 2;
		int currentCol = 2;
		int maxWidth = 0;

		TableInfo tableInfo = createSolutionTable(sheet, currentRow, currentCol, scenario);

	}

	private TableInfo createSolutionTable(ExcelSheet sheet, int currentRow, int currentCol,
			List<GJaxbCampaignSolution> scenarios) throws Exception {
		Object[][] values = new Object[0][];
		if (!scenarios.isEmpty()) {
			values = new Object[scenarios.size()][scenarios.get(0).getIndicator().size() + 1 + 3];
		}
		List<String> headerCols = new ArrayList<String>();
		int row = 0;
		for (GJaxbCampaignSolution scenario : scenarios) {
			values[row][0] = scenario.getName();
			if (row == 0) {
				headerCols.add("Name");
			}
			int col = 1;
			for (GJaxbIndicator impedance : scenario.getIndicator()) {
				values[row][col] = getImpedanceProcessValue(impedance);
				col = col + 1;
				if (row == 0) {
					headerCols.add(impedance.getName());
				}
			}
			values[row][col] = scenario.getRisksActivated();
			col = col + 1;
			values[row][col] = scenario.getPreventivesStrategiesActivated();
			col = col + 1;
			values[row][col] = scenario.getCorrectivesStrategiesActivated();
			col = col + 1;

			row = row + 1;
		}

		headerCols.add("Risks");
		headerCols.add("Preventives Strategies");
		headerCols.add("Correctives Strategies");

		ExcelTable table = sheet.addTable("Solutions:");
		return table.setValue(currentRow, currentCol, headerCols, Arrays.asList(), values, null);
	}

	private Object getImpedanceProcessValue(GJaxbIndicator val) {
		if (val.getIndicatorValue().getSimulatedValue() != null) {
			return val.getIndicatorValue().getSimulatedValue();
		}
		return null;
	}

	private TableInfo createBasicInformationCampaignTable(ExcelSheet sheet, int currentRow, int currentCol,
			GJaxbCampaign campaign) throws Exception {
		Object[][] values = new Object[10][1];
		values[0][0] = campaign.getCampaignName();
		values[1][0] = formatter.format(campaign.getStartedAt().toGregorianCalendar().getTime());
		if (campaign.getEndedAt() != null) {
			values[3][0] = formatter.format(campaign.getEndedAt().toGregorianCalendar().getTime());
		} else {
			values[3][0] = "";
		}
		values[4][0] = campaign.getStatus().toString();
		if (campaign.getTime() != null) {
			values[5][0] = formatterLong2Time.format(new Date(campaign.getTime()));
		} else {
			values[5][0] = "";
		}
		values[6][0] = campaign.getNumberOfSolution();
		values[7][0] = campaign.getNumberOfSolutionAchieved();
		values[8][0] = campaign.getNumberOfSuccessfulSolution();
		values[9][0] = campaign.getNumberOfFailureSolution();

		ExcelTable table = sheet.addTable("Basic Information:");
		return table.setValue(currentRow, currentCol, Arrays.asList("Value"),
				Arrays.asList("Name", "Created At", "Started At", "Ended At", "Status", "Duration",
						"Number of Solutions", "Number of Solutions Achieved", "Number of Successful Solutions",
						"Number of Failure Solutions"),
				values, null);
	}

	private TableInfo createNumericalMetricCampaignTable(ExcelSheet sheet, int currentRow, int currentCol,
			GJaxbCampaign campaign) throws Exception {
		Object[][] values = new Object[campaign.getStatistic().getStatisticMetric().size()][3];
		List<String> headerRows = new ArrayList<String>();
		int row = 0;
		for (StatisticMetric numVal : campaign.getStatistic().getStatisticMetric()) {
			headerRows.add(numVal.getName());
			values[row][0] = numVal.getMin();
			values[row][1] = numVal.getMax();
			values[row][2] = DoubleHelper.formatWith3Decimal(
					 (double) (numVal.getAverage() / campaign.getNumberOfSolutionAchieved()));
			row = row + 1;
		}

		ExcelTable table = sheet.addTable("Statistics:");
		return table.setValue(currentRow, currentCol, Arrays.asList("Min", "Max", "Average"), headerRows, values, null);
	}


}
