/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * first author: Nicolas SALATGE
 */
package fr.emac.gind.generic.application.websocket.pubsub;

/*
 * #%L
 * dw-generic-application
 * %%
 * Copyright (C) 2014 - 2017 EMAC - Gind
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import fr.emac.gind.commons.utils.map.MapHelper;
import fr.emac.gind.logger.GindLogger;

/**
 *
 *
 * @author Nicolas Salatge
 */
public class TopicManager {

	private static Logger LOG = LoggerFactory.getLogger(TopicManager.class.getName());

	private static TopicManager INSTANCE;
	private static Map<String, List<PubSubEndpoint>> topicsMap = Collections
			.synchronizedMap(new HashMap<String, List<PubSubEndpoint>>());
	private Map<String, MessageLocker> messageLockersMap = Collections
			.synchronizedMap(new HashMap<String, MessageLocker>());

	private TopicManager() {
	}

	public static TopicManager getInstance() {
		if (INSTANCE == null) {
			INSTANCE = new TopicManager();
		}
		return INSTANCE;
	}

	public void addTopic(PubSubEndpoint endpoint) {
		String topic = endpoint.getSession().getUpgradeRequest().getRequestURI().getPath().replace("/ws/pubsub/", "");
		if (topicsMap.get(topic) == null) {
			topicsMap.put(topic, Collections.synchronizedList(new ArrayList<PubSubEndpoint>()));
		}
		topicsMap.get(topic).add(endpoint);
		GindLogger.log(LOG, Level.DEBUG, "wssocket: subscribe new endpoint to topic",
				MapHelper.mapOf("topic", topic, "consumer", endpoint.getSession().getRemoteAddress().toString()));
	}

	public List<PubSubEndpoint> findTopic(String topicName) {
		if (topicsMap.get(topicName) != null) {
			return Collections.unmodifiableList(Collections.synchronizedList(topicsMap.get(topicName)));
		}

		if (topicName != null && topicName.startsWith("/")) {
			topicName = topicName.replaceFirst("/", "");
			if (topicsMap.get(topicName) != null) {
				return Collections.unmodifiableList(Collections.synchronizedList(topicsMap.get(topicName)));
			}
		}

		return null;
	}

	public Map<String, List<PubSubEndpoint>> getTopicsMap() {
		return topicsMap;
	}

	public Map<String, MessageLocker> getMessageLockersMap() {
		return messageLockersMap;
	}

	public void broadcastMessage(String topic, String message) {
		List<PubSubEndpoint> endpoints = new ArrayList<PubSubEndpoint>(this.findTopic(topic));
		List<PubSubEndpoint> endpointsToRemote = new ArrayList<PubSubEndpoint>();
		if (endpoints != null && !endpoints.isEmpty()) {
			Iterator<PubSubEndpoint> it = endpoints.iterator();
			synchronized (it) {
				while (it.hasNext()) {
					PubSubEndpoint p = it.next();
					if (p != null) {
						try {
							p.getSession().getRemote().sendString(message);
							GindLogger.log(LOG, Level.DEBUG, "wssocket: send message", MapHelper.mapOf("topic", topic,
									"consumer", p.getSession().getRemoteAddress().toString(), "payload", message));
						} catch (Throwable e) {
							GindLogger.log(LOG, Level.WARN, "wssocket: error to send message",
									MapHelper.mapOf("topic", topic, "consumer",
											p.getSession().getRemoteAddress().toString(), "payload", message,
											"errorMsg", e.getMessage()));
							LOG.warn(e.getMessage(), e);
							endpointsToRemote.add(p);
						}
					}
				}
			}
		}
		/**
		 * if(!endpointsToRemote.isEmpty()) { endpointsToRemote.forEach(ep -> {
		 * endpoints.remove(ep); EndpointHandler.getInstance().removeByInstance(ep); });
		 * }
		 */
	}

	public void removeEndpoint(PubSubEndpoint endpoint) {
		LOG.debug("Remove endpoint " + endpoint);
		for (Entry<String, List<PubSubEndpoint>> entry : topicsMap.entrySet()) {
			if (entry.getValue().contains(endpoint)) {
				LOG.debug("... removing endpoint in topic " + entry.getKey());
				entry.getValue().remove(endpoint);
				break;
			}
		}
	}

}